/**
	\file
	\brief		Header file for plugin user interface hierarchical list widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_HIERARCHICAL_VIEW__
#define __HDI_CORE_HIERARCHICAL_VIEW__

#include <vector>

#include "hdicoreMacros.h"

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class TreeEntry;

		/**
			\brief	Hierarchical view widget, much like a list view but allowing for a tree-like structure of entries
		*/
		class HierarchicalView : public Widget
		{
			public:
				typedef std::vector<TreeEntry> EntryVector;
				typedef std::vector<int32_t> IndexVector;
				typedef std::vector<std::string> IDVector;
				typedef std::vector<std::string> ValueVector;
			
				/**
					\brief	Constructs an empty HierarchicalView object
					\author	GW
					\date	09/2014
					
					\note	To test if a HierarchicalView object is empty, call isEmpty() on it
					\note	Empty HierarchicalView objects do not relate to any actual UI widget; they are designed to
							be "receivers" of some other HierarchicalView object via the overloaded assignment operator.
							Empty HierarchicalView objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				HierarchicalView();

				/**
					\brief	Constructs a new HierarchicalView object from an existing HierarchicalView object (copy
							constructor)
					\author	GW
					\date	09/2014

					\param	hierView_	Existing HierarchicalView object
				*/
				HierarchicalView(const HierarchicalView& hierView_);

				/**
					\brief	Constructs a hierarchical view with a frame only, no contents
					\author	GW
					\date	09/2014

					\param	frame_	Position and size of the list, in 1x resolution coordinates
				*/
				HierarchicalView(const Rect& frame_);

				/**
					\brief	Constructs a hierarchical view
					\author	GW
					\date	09/2014

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	entries_	Vector of entries
					\param	indexPath_	Initial index path to select (must exist in entries_, or be an empty vector)
				*/
				HierarchicalView(
					const Rect& frame_,
					const EntryVector& entries_,
					const IndexVector& indexPath_
				);

				/**
					\brief	Constructor for a hierarchical view with an initial ID selected
					\author	GW
					\date	09/2014

					\param	frame_		Position and size of the list, in 1x resolution coordinates
					\param	entries_	Vector of entries
					\param	idPath_		Initial ID path to select, as UTF-8 (must exist in entries_, or be an empty
										vector)
				*/
				HierarchicalView(
					const Rect& frame_,
					const EntryVector& entries_,
					const IDVector& idPath_
				);

				/**
					\brief		Allows one HierarchicalView object to be assigned from another
					\author		GW
					\date		09/2014

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target HierarchicalView object, but with its value updated to match that of rhs_
				*/
				virtual HierarchicalView& operator=(const HierarchicalView& rhs_);
				
				/**
					\brief	HierarchicalView destructor
					\author	GW
					\date	09/2014
				*/
				virtual ~HierarchicalView();
				
				/**
					\brief		Convenience method to clone a HierarchicalView object on the heap
					\author		GW
					\date		09/2014

					\returns	A pointer to the new HierarchicalView object
					
					\note		If you subclass HierarchicalView, you MUST overload this method yourself! If you don't
								and/or your clone() method does not return an instance of your HierarchicalView subclass,
								you will experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned HierarchicalView object.
				*/
				virtual HierarchicalView* clone() const;
				
				/**
					\brief		Convenience method to duplicate a HierarchicalView object, creating a new and identical
								UI element to the target (but not belonging to the same containing window)
					\author		GW
					\date		09/2014
					
					\returns	A pointer to the new HierarchicalView object (and new UI element)
					
					\note		The new hierarchical view will have copies of all entries present in the target, but be
								aware that copies of the user data in each entry cannot be made (because they are simply
								void* variables). As such, the user data in the copies will simply be set to NULL by
								this method.
								
					\warning	The caller must manage the memory for the returned HierarchicalView object.
				*/
				virtual HierarchicalView* duplicate() const;

				/**
					\brief	Deselects all currently selected entries in the list
					\author	GW
					\date	09/2014
				*/
				virtual void deselectAll();

				/**
					\brief		Gets the current text value of the list
					\author		GW
					\date		09/2014
					
					\returns	The UTF-8 text value of the currently selected entry, or "" for no selection
				*/
				virtual std::string text() const;

				/**
					\brief	Does nothing
					\author	GW
					\date	09/2014

					\note	Since a HierarchicalView inherently stores tree-like data, setting the text value has 
							potentially no meaning or an ambiguous meaning. Therefore, this method performs no action.
				*/
				virtual void setText(const std::string&);

				/**
					\brief		Gets the index path of the currently selected entry
					\author		GW
					\date		09/2014

					\returns	The index path of the currently selected entry, or empty if no entry is selected
				*/
				virtual IndexVector currentIndexPath() const;
				
				/**
					\brief		Sets the currently selected entry by index path
					\author		GW
					\date		09/2014

					\param		indexPath_	New index path to select
					\returns	true if the selection occurred successfully, or false if e.g. out of range
				*/
				virtual bool setCurrentIndexPath(const IndexVector& indexPath_);
				
				/**
					\brief		Gets the ID path of the currently selected entry
					\author		GW
					\date		09/2014

					\returns	UTF-8 ID path of the currently selected entry, or empty if no entry is currently selected
				*/
				virtual IDVector currentIDPath() const;
				
				/**
					\brief		Sets the currently selected entry by ID path
					\author		GW
					\date		09/2014

					\param		idPath_		ID path of new value to select in the hierarchical view, as UTF-8
					\returns	true if the entry with the given ID path was selected, or false if invalid
				*/
				virtual bool setCurrentIDPath(const IDVector& idPath_);
				
				/**
					\brief		Gets the string value path of the currently selected entry
					\author		GW
					\date		09/2014

					\returns	Value path of the hierarchical view, as UTF-8
				*/
				virtual ValueVector currentValuePath() const;
				
				/**
					\brief		Sets the currently selected entry by value path
					\author		GW
					\date		09/2014

					\param		valuePath_	New value to set for the hierarchical view (if it exists as an entry), as
											UTF-8
					\returns	true if the value path was set, false otherwise
				*/
				virtual bool setCurrentValuePath(const ValueVector& valuePath_);

				/**
					\brief		Gets the current entry object
					\author		GW
					\date		09/2014

					\returns	The TreeEntry object for the currently selected item
				*/
				virtual std::unique_ptr<TreeEntry> currentEntry() const;
				
				/**
					\brief		Gets the user data for the current entry
					\author		GW
					\date		09/2014

					\param		data__	Return-by-reference for the current user data; ignore if false is returned
					\returns	true if the method succeeds in acquiring the contained entry data, false otherwise
				*/
				virtual bool currentUserData(void*& data__) const;

				/**
					\brief		Gets the entries vector
					\author		GW
					\date		09/2014

					\returns	The entries vector for the target widget, containing all currently known entries
					
					\note		Modifying any value in an entry will not cause the widget to update automatically; call
								update() to update the UI for changes made to entry data.
				*/
				virtual const std::vector< std::shared_ptr<TreeEntry> >* const entries() const;
				
				/**
					\brief		Gets whether the entries vector already contains an entry, based on its ID path
					\author		GW
					\date		09/2014

					\param		idPath_		ID path of the entry to search for, as UTF-8
					\returns	true if the widget already contains an entry with the given ID path, false otherwise
				*/
				virtual bool hasEntry(const IDVector& idPath_) const;
				
				/**
					\brief		Gets the entry with the given ID path, provided it exists within the widget
					\author		GW
					\date		12/2022

					\param		idPath_		ID path of the entry to search for
					\returns	A pointer to the widget with the given ID path, or NULL if not found
				*/
				virtual std::shared_ptr<TreeEntry> entryWithIDPath(const IDVector& idPath_) const;
				
				/**
					\brief	Allows the entries in a hierarchical view to be changed out entirely
					\author	GW
					\date	09/2014

					\param	entries_	New set of entries that the widget will contain
				*/
				virtual void setEntries(const EntryVector& entries_);
				
				/**
					\brief		Inserts an entry into the list of entries at a given position
					\author		GW
					\date		12/2022

					\param		entry_	New entry object to insert into the list of entries
					\param		idPath_	Existing entry immediately before the position at which the new entry will be
										inserted; if the passed value is an empty string, or the leaf is an empty string,
										the new entry object will be inserted first at the root level or first within the
										penultimate container ID, respectively
					\returns	true if the entry is inserted successfully, false otherwise

					\note		If the entry for idPath_ doesn't exist in the list, the new entry will not be inserted.
					\note		If the passed entry is already present in the list (based on its ID path), it will not be
								inserted.
				*/
				virtual bool insertEntry(const TreeEntry& entry_, const IDVector& idPath_);
				
				/**
					\brief		Removes a specific entry from the list of entries
					\author		GW
					\date		12/2022

					\param		idPath_		ID path of the entry to be removed
					\returns	The removed entry, so the caller can inspect it (if no entries are left, NULL will be
								returned)

					\note		If the entry for idPath_ doesn't exist in the list, nothing will happen and NULL will be
								returned.
				*/
				virtual std::shared_ptr<TreeEntry> removeEntry(const IDVector& idPath_);

				/**
					\brief		Reorders a target entry (based on ID path) to be newly located just after another existing
								entry in the list
					\author		GW
					\date		12/2022

					\param		targetID_	ID path of an existing entry in the list to be relocated
					\param		positionID_	ID path of an existing entry immediately before the position at which
											targetID_ should be moved; if the passed value is an empty string, or the
											leaf is an empty string, the target entry will be placed first at the root
											level or first within the penultimate container ID, respectively
					\returns	true if the entry is repositioned successfully, false otherwise

					\note		If the entry for targetID_ or for positionID_ doesn't exist in the list, this method
								will bail.
				*/
				virtual bool reorderEntry(const IDVector& targetID_, const IDVector& positionID_);

				/**
					\brief	Clears all entries contained by the widget
					\author	GW
					\date	09/2014
				*/
				virtual void clearEntries();

				/**
					\brief	Simulates the hierarchical view having its value changed
					\author	GW
					\date	09/2014
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the hierarchical view
					\author		GW
					\date		09/2014

					\returns	A pointer to the currently registered value changed callback

					\note		In this context, "value" refers to the selected index/ID/value path changing.
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	09/2014

					\param	callback_	New callback for when the hierarchical view's value has changed

					\note	In this context, "value" refers to the selected index/ID/value path changing.
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
				
				/**
					\brief		Gets the double-click callback for the hierarchical view
					\author		GW
					\date		11/2024

					\returns	A pointer to the currently registered double-click callback
				*/
				virtual Callback* const doubleClickCallback() const;
				
				/**
					\brief	Sets the double-click callback
					\author	GW
					\date	11/2024

					\param	callback_	New callback for when the hierarchical view's current entry is double clicked

					\note	This callback will not be fired when editable entries are double clicked.
				*/
				virtual void setDoubleClickCallback(const Callback& callback_);
				
				/**
					\brief	Force the hierarchical view to update its entries
					\author	GW
					\date	09/2014
				*/
				virtual void update();
				
				/**
					\brief		Gets the callback set for when an entry is expanded/collapsed
					\author		GW
					\date		11/2015

					\returns	A pointer to the currently registered expansion callback
				*/
				virtual Callback* const expansionCallback() const;
				
				/**
					\brief	Sets/updates the callback for when an entry is expanded/collapsed
					\author	GW
					\date	11/2015

					\param	callback_	New callback for a user expanding/collapsing an entry

					\note	When your callback is executed, you can determine which entry was expanded/collapsed by
							calling the expansionEntry() method.
				*/
				virtual void setExpansionCallback(const Callback& callback_);
				
				/**
					\brief		Gets the entry that was last expanded/collapsed
					\author		GW
					\date		11/2015
					
					\returns	The expanded entry object
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the expansion callback.
				*/
				virtual std::unique_ptr<TreeEntry> expansionEntry() const;
				
				/**
					\brief		Gets the callback for when a cell is modified via double-click inline editing
					\author		GW
					\date		07/2022

					\returns	A pointer to the currently registered cell modification callback
				*/
				virtual Callback* const cellChangedCallback() const;
				
				/**
					\brief	Sets the callback for when a cell is modified via double-click inline editing
					\author	GW
					\date	07/2022

					\param	callback_	New callback for a user modifying a cell via inline editing

					\note	When your callback is executed, you can determine which entry was clicked by calling the
							changedEntry() method.
				*/
				virtual void setCellChangedCallback(const Callback& callback_);
				
				/**
					\brief		Gets the entry that was last modified via double-click inline editing
					\author		GW
					\date		07/2022
					
					\returns	The changed entry object
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the cell changed callback.
				*/
				virtual std::unique_ptr<TreeEntry> changedEntry() const;
				
				/**
					\brief		Gets whether reordering support for the list is enabled
					\author		GW
					\date		12/2022
				 
					\returns	true if drag-and-drop reordering support is enabled, false otherwise
				*/
				virtual bool reorderable() const;
			
				/**
					\brief	Sets whether reordering support for the list is enabled
					\author	GW
					\date	12/2022
				 
					\param	enable_		true to enable drag-and-drop reordering support, false otherwise
				*/
				virtual void setReorderable(const bool enable_);

				/**
					\brief		Gets the current potential-reorder callback for the list
					\author		GW
					\date		12/2022

					\returns	A pointer to the currently registered potential-reorder callback
				*/
				virtual Callback* const potentialReorderCallback() const;
			
				/**
					\brief	Sets the potential-reorder callback for the widget
					\author	GW
					\date	12/2022

					\param	callback_	New callback (target and action) for each time the potential reorder target changes
					
					\note	When your callback is executed, you can determine which entry is being proposed for reorder
							and its proposed new position by calling the potentialReorderSource() and
							potentialReorderPosition() methods, respectively.
				*/
				virtual void setPotentialReorderCallback(const Callback& callback_);
			
				/**
					\brief		Gets the source item involved in the potential reorder, i.e. the entry being dragged
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry that might be reordered
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual std::vector<std::string> potentialReorderSource() const;

				/**
					\brief		Gets the position for the reorder, i.e. the entry just before the proposed reposition
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry just before the potential drop point for a reposition; if the
								returned vector contains only an empty string, or the leaf is an empty string, this refers
								to the proposed drop position being the first at the root level or the first within the
								penultimate container ID, respectively.
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual std::vector<std::string> potentialReorderPosition() const;

				/**
					\brief		Gets whether the potential reorder is acceptable; if not, the drag cursor will indicate
								the proposed drop position is not allowed
					\author		GW
					\date		12/2022
				
					\returns	true if the specific reorder drop position is allowed, false otherwise
					
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the potential-reorder callback.
				*/
				virtual bool potentialReorderGood() const;

				/**
					\brief	Sets whether the potential reorder is acceptable; if not, the drag cursor will indicate the
							proposed drop position is not allowed
					\author	GW
					\date	12/2022
				
					\param	good_	Whether the potential reorder position is acceptable
					
					\note	This method will have no effect if it is called any other time than from within a potential-
							reorder callback.
					\note	A potential reorder is always assumed to be good, so the true/false nature of a potential
							reorder will always be true unless you have called this method within your callback and
							passed false.
					\note	Calling this method and passing false will not stop a potential reorder from occurring
							altogether; rather, it merely indicates whether a specific reorder proposed drop position is
							acceptable. It is still up to you whether the entries, and any underlying associated data,
							is actually reordered once your reorder callback is executed.
				*/
				virtual void setPotentialReorderGood(const bool good_);

				/**
					\brief		Gets the current reorder callback for the widget
					\author		GW
					\date		12/2022

					\returns	A pointer to the currently registered reorder callback
				*/
				virtual Callback* const reorderCallback() const;
			
				/**
					\brief	Sets the reorder callback for the widget
					\author	GW
					\date	12/2022

					\param	callback_	New callback (target and action) for when a reorder is finalized (via dropping)
					
					\note	When your callback is executed, you can determine which entry is being reordered and its new
							position by calling the reorderSource() and reorderPosition() methods, respectively.
				*/
				virtual void setReorderCallback(const Callback& callback_);
			
				/**
					\brief		Gets the source item involved in the reorder, i.e. the entry being dropped
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry that was reordered
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the reorder callback.
				*/
				virtual std::vector<std::string> reorderSource() const;

				/**
					\brief		Gets the position for the reorder, i.e. the entry just before the reposition
					\author		GW
					\date		12/2022
				 
					\returns	The ID path of the entry just before the drop point for a reposition; if the returned
								vector contains only an empty string, or the leaf is an empty string, this refers to
								the drop position being the first at the root level or the first within the penultimate
								container ID, respectively.
				 
					\note		The return value of this method is only guaranteed to be valid during the execution of
								the reorder callback.
				*/
				virtual std::vector<std::string> reorderPosition() const;
		};
		
		typedef std::unique_ptr<HierarchicalView> HierarchicalViewUP;
		typedef std::shared_ptr<HierarchicalView> HierarchicalViewSP;
		typedef std::weak_ptr<HierarchicalView> HierarchicalViewWP;
	}
}

#endif
// __HDI_CORE_HIERARCHICAL_VIEW__
