/**
	\file
	\brief		Header file for handling image IDs and their higher resolution variants
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_IMAGE_IDS__
#define __HDI_CORE_IMAGE_IDS__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Stores both light and dark image PNGI resource IDs, such that app UI brightness changes can be
					properly handled
		*/
		class ImageIDs
		{
			public:
				/**
					\brief		Default image ID to indicate that no image has been set

					\warning	Do not use 0 as your own PNGI image ID!
				*/
				static const int16_t noneID = 0;
			
				/**
					\brief	ID of the PNGI resource that should be used when the UI is light in color
				*/
				int16_t lightID;
				
				/**
					\brief	ID of the PNGI resource that should be used when the UI is light in color at 1.5x resolution
					
					\note	Ignored by all Mac versions
				*/
				int16_t lightID_3to2x;
				
				/**
					\brief	ID of the PNGI resource that should be used when the UI is light in color at 2x resolution
				*/
				int16_t lightID_2x;
 				
 				/**
					\brief	ID of the PNGI resource that should be used when the UI is light in color at 3x resolution
				*/
				int16_t lightID_3x;
				
				/**
					\brief	ID of the PNGI resource that should be used when the UI is light in color at 4x resolution
				*/
				int16_t lightID_4x;
				
				
				/**
					\brief	ID of the PNGI resource that should be used when the UI is dark in color
				*/
				int16_t darkID;
				
				/**
					\brief	ID of the PNGI resource that should be used when the UI is dark in color at 1.5x resolution
					
					\note	Ignored by all Mac versions
				*/
				int16_t darkID_3to2x;
				
				/**
					\brief	ID of the PNGI resource that should be used when the UI is dark in color at 2x resolution
				*/
				int16_t darkID_2x;
 				
 				/**
					\brief	ID of the PNGI resource that should be used when the UI is dark in color at 3x resolution
				*/
				int16_t darkID_3x;
				
				/**
					\brief	ID of the PNGI resource that should be used when the UI is dark in color at 4x resolution
				*/
				int16_t darkID_4x;
				
				/**
					\brief	Constructs a new ImageIDs object with default values (i.e. noneID)
					\author	GW
					\date	02/2015
				*/
				ImageIDs();
				
				/**
					\brief	Constructs a new ImageIDs object with the same values as an existing object
					\author	GW
					\date	02/2015
					
					\param	img_	ImageIDs object to copy values from
				*/
				ImageIDs(const ImageIDs& img_);
				
				/**
					\brief	Constructs a new ImageIDs object from light and dark image IDs, setting all higher resolutions
							to noneID
					\author	GW
					\date	02/2015
					
					\param	light_	Light UI resource ID
					\param	dark_	Dark UI resource ID
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageIDs(const int16_t light_, const int16_t dark_);
				
				/**
					\brief	Constructs a new ImageIDs object from light and dark image IDs individually for each
							resolution at 2x and below
					\author	GW
					\date	02/2015
					
					\param	light_		Light UI resource ID
					\param	light3to2x_	Light UI resource ID at 1.5x
					\param	light2x_	Light UI resource ID at 2x
					\param	dark_		Dark UI resource ID
					\param	dark3to2x_	Dark UI resource ID at 1.5x
					\param	dark2x_		Dark UI resource ID at 2x
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageIDs(
					const int16_t light_,
					const int16_t light3to2x_,
					const int16_t light2x_,
					const int16_t dark_,
					const int16_t dark3to2x_,
					const int16_t dark2x_
				);
 				
 				/**
					\brief	Constructs a new ImageIDs object from light and dark image IDs individually for each
							resolution
					\author	GW
					\date	08/2018
					
					\param	light_		Light UI resource ID
					\param	light3to2x_	Light UI resource ID at 1.5x
					\param	light2x_	Light UI resource ID at 2x
					\param	light3x_	Light UI resource ID at 3x
					\param	light4x_	Light UI resource ID at 4x
					\param	dark_		Dark UI resource ID
					\param	dark3to2x_	Dark UI resource ID at 1.5x
					\param	dark2x_		Dark UI resource ID at 2x
					\param	dark3x_		Dark UI resource ID at 3x
					\param	dark4x_		Dark UI resource ID at 4x
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageIDs(
					const int16_t light_,
					const int16_t light3to2x_,
					const int16_t light2x_,
					const int16_t light3x_,
					const int16_t light4x_,
					const int16_t dark_,
					const int16_t dark3to2x_,
					const int16_t dark2x_,
					const int16_t dark3x_,
					const int16_t dark4x_
				);
				
				/**
					\brief	Destructs an ImageIDs object
					\author	GW
					\date	02/2015
				*/
				virtual ~ImageIDs();
				
				/**
					\brief		Overloaded assignment operator to copy values from one ImageIDs object to another
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the assignment
					\returns	The lefthand side of the assignment, but with its values updated to match rhs_
				*/
				ImageIDs& operator=(const ImageIDs& rhs_);
				
				/**
					\brief		Overloaded equality operator to compare two ImageIDs objects with one another
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the same image
								IDs
				*/
				bool operator==(const ImageIDs& rhs_) const;
				
				/**
					\brief		Overloaded inequality operator to compare two ImageIDs objects with one another
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the differing
								image IDs
				*/
				bool operator!=(const ImageIDs& rhs_) const;
		};
		
		typedef std::unique_ptr<ImageIDs> ImageIDsUP;
		typedef std::shared_ptr<ImageIDs> ImageIDsSP;
		typedef std::weak_ptr<ImageIDs> ImageIDsWP;
		
		

		/**
			\brief	Allows for image rollover support
		*/
		class ImageWithRolloverIDs : public ImageIDs
		{
			public:
				/**
					\brief	PNGI resource ID for the 1x rollover image variant for the app's light-themed UI
				*/
				int16_t lightOverID;
				
				/**
					\brief	PNGI resource ID for the 1.5x rollover image variant for the app's light-themed UI
					
					\note	Ignored by all Mac versions
				*/
				int16_t lightOverID_3to2x;
				
				/**
					\brief	PNGI resource ID for the 2x rollover image variant for the app's light-themed UI
				*/
				int16_t lightOverID_2x;
 				
 				/**
					\brief	PNGI resource ID for the 3x rollover image variant for Illustrator's light-themed UI
				*/
				int16_t lightOverID_3x;
				
				/**
					\brief	PNGI resource ID for the 4x rollover image variant for Illustrator's light-themed UI
				*/
				int16_t lightOverID_4x;
				
				/**
					\brief	PNGI resource ID for the 1x rollover image variant for the app's dark-themed UI
				*/
				int16_t darkOverID;

				/**
					\brief	PNGI resource ID for the 1.5x rollover image variant for the app's dark-themed UI
					
					\note	Ignored by all Mac versions
				*/
				int16_t darkOverID_3to2x;

				/**
					\brief	PNGI resource ID for the 2x rollover image variant for the app's dark-themed UI
				*/
				int16_t darkOverID_2x;

				/**
					\brief	PNGI resource ID for the 3x rollover image variant for Illustrator's dark-themed UI
				*/
				int16_t darkOverID_3x;

				/**
					\brief	PNGI resource ID for the 4x rollover image variant for Illustrator's dark-themed UI
				*/
				int16_t darkOverID_4x;
			
				/**
					\brief	Default ImageWithRolloverIDs constructor, setting all ID values to noneID
					\author	GW
					\date	03/2015
				*/
				ImageWithRolloverIDs();
				
				/**
					\brief	ImageWithRolloverIDs copy constructor, using values from an existing object
					\author	GW
					\date	03/2015
					
					\param	ids_	Existing ImageWithRolloverIDs object to copy values from
				*/
				ImageWithRolloverIDs(const ImageWithRolloverIDs& ids_);
				
				/**
					\brief	Constructs an ImageWithRolloverIDs object from light-UI, light-UI rollover, dark-UI, and
							dark-UI resource IDs; sets all higher resolutions to noneID
					\author	GW
					\date	03/2015
					
					\param	light_		Light UI resource ID
					\param	lightOver_	Light UI rollover resource ID
					\param	dark_		Dark UI resource ID
					\param	darkOver_	Dark UI rollover resource ID
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageWithRolloverIDs(
					const int16_t light_,
					const int16_t lightOver_,
					const int16_t dark_,
					const int16_t darkOver_
				);
				
				/**
					\brief	Creates an ImageWithRolloverIDs object from light-UI and dark-UI resource IDs for all supported
							resolutions at 2x and below (without any rollovers)
					\author	GW
					\date	03/2015
					
					\param	light_		Light UI resource ID
					\param	light3to2x_	Light UI resource ID at 1.5x
					\param	light2x_	Light UI resource ID at 2x
					\param	dark_		Dark UI resource ID
					\param	dark3to2x_	Dark UI resource ID at 1.5x
					\param	dark2x_		Dark UI resource ID at 2x
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageWithRolloverIDs(
					const int16_t light_,
					const int16_t light3to2x_,
					const int16_t light2x_,
					const int16_t dark_,
					const int16_t dark3to2x_,
					const int16_t dark2x_
				);
 				
 				/**
 					\brief	Creates an ImageWithRolloverIDs object from light-UI and dark-UI resource IDs for all supported
							resolutions (without any rollovers)
					\author	GW
					\date	08/2018
					
					\param	light_		Light UI resource ID
					\param	light3to2x_	Light UI resource ID at 1.5x
					\param	light2x_	Light UI resource ID at 2x
					\param	light3x_	Light UI resource ID at 3x
					\param	light4x_	Light UI resource ID at 4x
					\param	dark_		Dark UI resource ID
					\param	dark3to2x_	Dark UI resource ID at 1.5x
					\param	dark2x_		Dark UI resource ID at 2x
					\param	dark3x_		Dark UI resource ID at 3x
					\param	dark4x_		Dark UI resource ID at 4x
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageWithRolloverIDs(
					const int16_t light_,
					const int16_t light3to2x_,
					const int16_t light2x_,
					const int16_t light3x_,
					const int16_t light4x_,
					const int16_t dark_,
					const int16_t dark3to2x_,
					const int16_t dark2x_,
					const int16_t dark3x_,
					const int16_t dark4x_
				);
				
				/**
					\brief	Creates an ImageWithRolloverIDs object from light-UI and dark-UI resource IDs for all supported
							resolutions at 2x and below
					\author	GW
					\date	03/2015
					
					\param	light_			Light UI resource ID
					\param	light3to2x_		Light UI resource ID at 1.5x
					\param	light2x_		Light UI resource ID at 2x
					\param	dark_			Dark UI resource ID
					\param	dark3to2x_		Dark UI resource ID at 1.5x
					\param	dark2x_			Dark UI resource ID at 2x
					\param	lightOver_		Light rollover UI resource ID
					\param	lightOver3to2x_	Light rollover UI resource ID at 1.5x
					\param	lightOver2x_	Light rollover UI resource ID at 2x
					\param	darkOver_		Dark rollover UI resource ID
					\param	darkOver3to2x_	Dark rollover UI resource ID at 1.5x
					\param	dark2Overx_		Dark rollover UI resource ID at 2x
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageWithRolloverIDs(
					const int16_t light_,
					const int16_t light3to2x_,
					const int16_t light2x_,
					const int16_t dark_,
					const int16_t dark3to2x_,
					const int16_t dark2x_,
					const int16_t lightOver_,
					const int16_t lightOver3to2x_,
					const int16_t lightOver2x_,
					const int16_t darkOver_,
					const int16_t darkOver3to2x_,
					const int16_t darkOver2x_
				);
 				
 				/**
					\brief	Creates an ImageWithRolloverIDs object from light-UI and dark-UI resource IDs for all supported
							resolutions
					\author	GW
					\date	08/2018
					
					\param	light_			Light UI resource ID
					\param	light3to2x_		Light UI resource ID at 1.5x
					\param	light2x_		Light UI resource ID at 2x
					\param	light3x_		Light UI resource ID at 3x
					\param	light4x_		Light UI resource ID at 4x
					\param	dark_			Dark UI resource ID
					\param	dark3to2x_		Dark UI resource ID at 1.5x
					\param	dark2x_			Dark UI resource ID at 2x
					\param	dark3x_			Dark UI resource ID at 3x
					\param	dark4x_			Dark UI resource ID at 4x
					\param	lightOver_		Light rollover UI resource ID
					\param	lightOver3to2x_	Light rollover UI resource ID at 1.5x
					\param	lightOver2x_	Light rollover UI resource ID at 2x
					\param	lightOver3x_	Light rollover UI resource ID at 3x
					\param	lightOver4x_	Light rollover UI resource ID at 4x
					\param	darkOver_		Dark rollover UI resource ID
					\param	darkOver3to2x_	Dark rollover UI resource ID at 1.5x
					\param	darkOver2x_		Dark rollover UI resource ID at 2x
					\param	darkOver3x_		Dark rollover UI resource ID at 3x
					\param	darkOver4x_		Dark rollover UI resource ID at 4x
					
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				ImageWithRolloverIDs(
					const int16_t light_,
					const int16_t light3to2x_,
					const int16_t light2x_,
					const int16_t light3x_,
					const int16_t light4x_,
					const int16_t dark_,
					const int16_t dark3to2x_,
					const int16_t dark2x_,
					const int16_t dark3x_,
					const int16_t dark4x_,
					const int16_t lightOver_,
					const int16_t lightOver3to2x_,
					const int16_t lightOver2x_,
					const int16_t lightOver3x_,
					const int16_t lightOver4x_,
					const int16_t darkOver_,
					const int16_t darkOver3to2x_,
					const int16_t darkOver2x_,
					const int16_t darkOver3x_,
					const int16_t darkOver4x_
				);
				
				/**
					\brief	ImageWithRolloverIDs destructor
					\author	GW
					\date	03/2015
				*/
				virtual ~ImageWithRolloverIDs();
				
				/**
					\brief		Overloaded assignment operator to copy values from one ImageWithRolloverIDs object to
								another
					\author		GW
					\date		03/2015
					
					\param		rhs_	Righthand side of the assignment
					\returns	The lefthand side of the assignment, but with its values updated to match rhs_
				*/
				ImageWithRolloverIDs& operator=(const ImageWithRolloverIDs& rhs_);
				
				/**
					\brief		Overloaded equality operator to compare two ImageWithRolloverIDs objects with one another
					\author		GW
					\date		03/2015
					
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the same image
								IDs
				*/
				bool operator==(const ImageWithRolloverIDs& rhs_) const;
				
				/**
					\brief		Overloaded inequality operator to compare two ImageWithRolloverIDs objects with one another
					\author		GW
					\date		03/2015
					
					\param		rhs_	Righthand side of the comparison
					\returns	true if the lefthand side and righthand side of the comparison have the differing
								image IDs
				*/
				bool operator!=(const ImageWithRolloverIDs& rhs_) const;
		};
		
		typedef std::unique_ptr<ImageWithRolloverIDs> ImageWithRolloverIDsUP;
		typedef std::shared_ptr<ImageWithRolloverIDs> ImageWithRolloverIDsSP;
		typedef std::weak_ptr<ImageWithRolloverIDs> ImageWithRolloverIDsWP;
	}
}

#endif
// __HDI_CORE_IMAGE_IDS__
