/**
	\file
	\brief		Header file for basic internet ICMP ping requests
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_INTERNET_PING__
#define __HDI_CORE_INTERNET_PING__

#include <string>

namespace hdi
{
	namespace core
	{
		class Callback;

		/**
			\brief	Allows for the creation of simple ICMP ping clients to check the availability of a remote host
			
			\note	All pings have a time-to-live of 60 seconds.
		*/
		class InternetPing
		{
			public:
				/**
					\brief	Creates a synchronous (blocking) InternetPing object to perform a ping request
					\author	GW
					\date	03/2017
					
					\param	domain_		Domain name to ping, as UTF-8
					\param	success_	Callback to perform when the request completes successfully
					\param	failure_	Callback to perform when the request fails
					
					\note	The callbacks will not be called if their scheduling comes during or after plugin
							pre-shutdown (this would lead to a crash on quit).
				*/
				InternetPing(
					const std::string& domain_,
					const Callback& success_,
					const Callback& failure_
				);
				
				/**
					\brief		Creates an asynchronous (non-blocking) InternetPing object to perform a ICMP ping request
					\author		GW
					\date		03/2017
					
					\param		domain_		Domain name to ping, as UTF-8
					\param		success_	Callback to perform when the request completes successfully
					\param		failure_	Callback to perform when the request fails
					
					\warning	The responsibility for the memory of the returned InternetPing pointer lies with the
								caller, and it must not be deleted until after either the success or failure callbacks
								are called.

					\note		Asynchronous InternetPing objects must be allocated using operator new and have several
								internal flags set (only synchronous InternetPing objects can be allocated on the stack),
								which this function does automatically.
					\note		The callbacks will not be called if their scheduling comes during or after plugin
								pre-shutdown (this would lead to a crash on quit).
				*/
				static InternetPing* Async(
					const std::string& domain_,
					const Callback& success_,
					const Callback& failure_
				);

				/**
					\brief	Destructs an InternetPing object
					\author	GW
					\date	03/2017
				*/
				virtual ~InternetPing();
				
				/**
					\brief		Gets whether the target object is asynchronous (i.e. executes in a background thread)
					\author		GW
					\date		03/2017
					
					\returns	true for asynchronous, false for synchronous
				*/
				virtual bool asynchronous() const;
				
				/**
					\brief	Starts the ping request
					\author	GW
					\date	03/2017
				*/
				virtual void run();
				
				/**
					\brief		Gets whether the ping request is currently running
					\author		GW
					\date		03/2017
					
					\returns	true if the request is running, false otherwise
				*/
				virtual bool running() const;
				
				/**
					\brief		Gets whether the remote host was reached when pinged
					\author		GW
					\date		03/2017
					
					\returns	true if the remote host was reached, false otherwise
					
					\note		The results of this method are only valid when it is called from the "success" callback
								(see the InternetPing constructor).
				*/
				virtual bool reached() const;
			
				/**
					\brief		Gets how long it took to reach the remote host
					\author		GW
					\date		03/2017
					
					\returns	Total round-trip time, in seconds, to reach the host, or -1.0 if the host was not reached
								at all
					
					\note		The results of this method are only valid when it is called from the "success" callback
								(see the InternetPing constructor).
				*/
				virtual double timing() const;
			
				/**
					\brief		Gets the IP address of the remote host for the domain given when the InternetPing object
								was constructed
					\author		GW
					\date		03/2017
					
					\returns	The IPv4 or IPv6 address of the host (whichever was actually used for the ping), or "" if
								the host was not reached at all
					
					\note		The results of this method are only valid when it is called from the "success" callback
								(see the InternetPing constructor).
				*/
				virtual std::string address() const;
				
				/**
					\brief		Gets an error description if the ping request failed
					\author		GW
					\date		03/2017
					
					\returns	The resultant error of the request, as UTF-8

					\note		The results of this method are only valid when it is called from the "failure" callback
								(see the InternetPing constructor).
				*/
				virtual std::string error() const;
				

			private:
				/**
					\brief	Private implementation object
				*/
				void* __impl;
				
				/**
					\brief	Empty InternetPing objects cannot be constructed
					\author	GW
					\date	03/2017
				*/
				InternetPing();
			
				/**
					\brief	InternetPing objects cannot be copied
					\author	GW
					\date	03/2017
				*/
				InternetPing(const InternetPing&);
			
				/**
					\brief	InternetPing objects cannot be copied
					\author	GW
					\date	03/2017
				*/
				InternetPing& operator=(const InternetPing&);
		};
	}
}

#endif
// __HDI_CORE_INTERNET_PING__
