/**
	\file
	\brief		Header file for creating entry objects to be added to Widget subclasses (that support them)
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_LIST_ENTRY__
#define __HDI_CORE_LIST_ENTRY__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace pui
	{
		class BaseEntryData;
	}
	
	namespace core
	{
		/**
			\brief	Allows combo boxes, popups, and lists to store more data than simply strings and indices
		*/
		class ListEntry
		{
			public:
				/**
					\brief	When you want a separator in a container widget, create a ListEntry object with this as its
							value
				*/
				static const char* separatorValue;
			
				/**
					\brief	Constructs an empty ListEntry object
					\author	GW
					\date	09/2013
					
					\note	To test if an ListEntry object is empty, call isEmpty() on it
					\note	Empty ListEntry objects do not relate to any entry in any widget; they are designed to
							be "receivers" of some other ListEntry object via the overloaded assignment operator. Empty
							ListEntry objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				ListEntry();
			
				/**
					\brief	Constructs a new ListEntry object from an existing ListEntry object (copy constructor)
					\author	GW
					\date	09/2013

					\param	e_	Existing ListEntry object
				*/
				ListEntry(const ListEntry& e_);

				/**
					\brief	Constructs a ListEntry object with some custom data
					\author	GW
					\date	09/2013

					\param	id_			Entry ID string (must be unique across all entries in the target container), as
										UTF-8
					\param	value_		Entry string value, as UTF-8
					\param	userData_	Data to store in the entry
					\param	enabled_	Whether the entry should be enabled
					\param	editable_	Whether the entry should be inline editable
				*/
				ListEntry(
					const std::string& id_,
					const std::string& value_,
					void* userData_,
					const bool enabled_,
					const bool editable_
				);

				/**
					\brief	Destructs a ListEntry object
					\author	GW
					\date	09/2013
				*/
				virtual ~ListEntry();
				
				/**
					\brief		Assigns one ListEntry object to another
					\author		GW
					\date		09/2013

					\param		rhs_	Existing ListEntry object to copy values from
					\returns	The target ListEntry object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual ListEntry& operator=(const ListEntry& rhs_);

				/**
					\brief		Gets whether the target ListEntry object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2013
					
					\returns	true if the target ListEntry object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the ID string of an entry
					\author		GW
					\date		09/2013

					\returns	The unique ID of the entry, as UTF-8
				*/
				virtual std::string entryID() const;
				
				/**
					\brief	Sets the ID string of an entry
					\author	GW
					\date	09/2013

					\param	id_		New entry ID string, as UTF-8

					\note	It is up to the caller to ensure that the ID is not in use with other entries within the
							same container widget.
				*/
				virtual void setEntryID(const std::string& id_);
				
				/**
					\brief		Gets the string value of an entry (what the user sees in the popup/list/etc.)
					\author		GW
					\date		09/2013

					\returns	The entry's string value, as UTF-8
				*/
				virtual std::string value() const;
				
				/**
					\brief	Sets the string value of an entry (what the user sees in the popup/list/etc.)
					\author	GW
					\date	09/2013

					\param	value_	New entry string value, as UTF-8
				*/
				virtual void setValue(const std::string& value_);

				/**
					\brief		Gets whether the entry should be enabled (selectable by the user in the popup/list/etc.)
					\author		GW
					\date		09/2013
					
					\returns	true for enabled, false otherwise
				*/
				virtual bool enabled() const;

				/**
					\brief	Sets whether the entry should be enabled (selectable by the user in the popup/list/etc.)
					\author	GW
					\date	09/2013
					
					\param	enabled_	true for enabled, false otherwise
				*/
				virtual void setEnabled(const bool enabled_);
				
				/**
					\brief		Gets the editability of the entry (whether the user can edit it inline)
					\author		GW
					\date		07/2022
					
					\returns	true for enabled, false otherwise
				*/
				virtual bool editable() const;

				/**
					\brief	Sets the editability of the entry (whether the user can edit it inline)
					\author	GW
					\date	07/2022
					
					\param	enabled_	true for enabled, false otherwise
				*/
				virtual void setEditable(const bool editable_);

				/**
					\brief		Gets the custom data of an entry
					\author		GW
					\date		09/2013

					\returns	The data stored inside the target entry object
				*/
				virtual void* const userData() const;
				
				/**
					\brief	Sets the custom data of an entry
					\author	GW
					\date	09/2013

					\param	data_	New custom data to store
				*/
				virtual void setUserData(void* data_);
				
				/**
					\brief		Tests whether a given ListEntry object is the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	ListEntry to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same entry, false otherwise
				*/
				virtual bool operator==(const ListEntry& rhs_) const;
				
				/**
					\brief		Tests whether a given ListEntry object is not the same as another
					\author		GW
					\date		09/2013

					\param		rhs_	ListEntry to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different entries, false otherwise
				*/
				virtual bool operator!=(const ListEntry& rhs_) const;


			private:
				friend pui::BaseEntryData* __accessImpl(const ListEntry&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<ListEntry> ListEntryUP;
		typedef std::shared_ptr<ListEntry> ListEntrySP;
		typedef std::weak_ptr<ListEntry> ListEntryWP;
		
		extern pui::BaseEntryData* __accessImpl(const ListEntry&);
	}
}

#endif
// __HDI_CORE_LIST_ENTRY__
