/**
	\file
	\brief		Header file for plugin string localization
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_LOCALIZER__
#define __HDI_CORE_LOCALIZER__

#include <map>
	
#include "hdicoreTypes.h"

/**
	\brief	Slightly shorter way to access the Localizer instance, but still utilizing a non-colliding name scheme
*/
#define HDI_CORE_LOCALIZER	hdi::core::Localizer::instance()

namespace hdi
{
	namespace core
	{
		/**
			\brief		Singleton class to aid in runtime localization
			\details	Strings are added to this container with a key, locale for the value, and the localized value
						itself. Localized strings can then be retrieved from the container at runtime using a variety
						of methods.

			\code
				// It's best to picture the localized strings as being stored in a map keyed on the locale, with this
				// map then being stored in another map keyed on a unique string.
				// To add strings:
				HDI_CORE_LOCALIZER->addString("HelloKey", Locale_ENG_US, "Hello");
				HDI_CORE_LOCALIZER->addString("HelloKey", Locale_FRA_FR, "Bonjour");
				HDI_CORE_LOCALIZER->addString("HelloKey", Locale_DEU_DE, "Hallo");
				HDI_CORE_LOCALIZER->addString("HelloKey", Locale_SPA_MX, "Hola");
				// To get strings:
				// (This would return a map keyed on locale, with the values as in the example above)
				HDI_CORE_LOCALIZER->getAllStrings("HelloKey");
				// (This would return the string "Bonjour")
				HDI_CORE_LOCALIZER->getString("HelloKey", Locale_FRA_FR);
				// (This would use whatever the app's current locale is to acquire the string, defaulting to
				// defaultLocale() if the current locale has not been added for the key "HelloKey")
				HDI_CORE_LOCALIZER->getLocalString("HelloKey");
			\endcode
		*/
		class Localizer
		{
			public:
				typedef std::map<Locale, std::string> StringMap;
			
				/**
					\brief		Allows access to the singleton localizer instance object
					\author		GW
					\date		08/2013

					\returns	The localizer instance object
				*/
				static Localizer* instance();

				/**
					\brief		Gets the default locale of the Localizer instance
					\author		GW
					\date		08/2013

					\returns	Default locale; always set to Locale_ENG_US initially
				*/
				Locale defaultLocale() const;

				/**
					\brief	Sets the default locale of the Localizer instance
					\author	GW
					\date	08/2013

					\param	defaultLocale_	New default locale
				*/
				void setDefaultLocale(const Locale defaultLocale_);

				/**
					\brief	Adds a new localized string value, keyed by a unique string and the value's locale
					\author	GW
					\date	08/2013

					\param	key_		Unique string key for the value, as UTF-8
					\param	locale_		Locale of the value
					\param	str_		Localized string value, as UTF-8
				*/
				void addString(const std::string& key_, const Locale locale_, const std::string& str_);

				/**
					\brief	Removes an existing localized string value by its key and locale
					\author	GW
					\date	08/2013

					\param	key_		Unique string key for the value, as UTF-8
					\param	locale_		Locale of the value
				*/
				void removeString(const std::string& key_, const Locale locale_);

				/**
					\brief		Gets a map of locales and their corresponding localized strings for a given key
					\author		GW
					\date		08/2013

					\param		key_	Unique string key for the value
					\returns	A map of all locales and their values, as UTF-8, for a given key, or an empty map for an
								unknown key
				*/
				StringMap getAllStrings(const std::string& key_) const;

				/**
					\brief		Gets a localized string for a given key and locale
					\author		GW
					\date		08/2013

					\param		key_		Unique string key for the value, as UTF-8
					\param		locale_		Locale of the value
					\returns	The localized string for the given key/locale combination, as UTF-8

					\note		Will return "" in the case of key_, locale_, and defaultLocale() all being invalid
				*/
				std::string getString(const std::string& key_, const Locale locale_) const;

				/**
					\brief		Gets a localized string for a given key and whatever locale the app currently has
					\author		GW
					\date		08/2013

					\param		key_	Unique string key for the value, as UTF-8
					\returns	The localized string for the given key and current locale combination, as UTF-8

					\note		If the given key does not have a value for the current locale, this method will default
								to the value for defaultLocale() and the given key.
					\note		Returns "" in the case of key_, the current locale, and defaultLocale() all being invalid
				*/
				std::string getLocalString(const std::string& key_) const;


			private:
				friend class Plugin;

				/**
					\brief	Singleton instance for this class
				*/
				static Localizer* __instance;
			
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Localizer();

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				~Localizer();
		};
	}
}

#endif
// __HDI_CORE_LOCALIZER__
