/**
	\file
	\brief		Header file for a class to create and manipulate plugin modal dialogs
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_MODAL_DIALOG__
#define __HDI_CORE_MODAL_DIALOG__

#include "hdicoreSize.h"
#include "hdicoreWindow.h"

namespace hdi
{
	namespace core
	{
		class Button;
		class Callback;
		class Widget;

		/**
			\brief	Used to create and display a modal dialog to the user, blocking input to other UI elements
		*/
		class ModalDialog : public Window
		{
			public:
				/**
					\brief	Identifies which button was clicked
				*/
				enum ChosenButton
				{
					UnknownButtonChosen	= 0,
					OKButtonChosen		= 1,
					CancelButtonChosen	= 2
				};

				/**
					\brief	Controls button options for the ModalDialog constructor
				*/
				class ButtonOptions
				{
					public:
						/**
							\brief	Whether the button should be shown
						*/
						bool enable;
						
						/**
							\brief	Whether the modal should be automatically dismissed when the button is chosen
						*/
						bool dismiss;
						
						/**
							\brief	Button text value, as UTF-8

							\note	If enable is true and title is empty (""), a localized default will be used (e.g.
									"OK", "Cancel", etc.)
						*/
						std::string title;
						
						/**
							\brief	Constructs a ButtonOptions object with default settings
							\author	GW
							\date	09/2013
						*/
						ButtonOptions();
						
						/**
							\brief	Constructs a ButtonOptions object with the same settings as another object
							\author	GW
							\date	02/2015
						*/
						ButtonOptions(const ButtonOptions& bo_);
						
						/**
							\brief	Constructs a ButtonOptions object with the given settings
							\author	GW
							\date	09/2013

							\param	enable_		Whether the button is enabled
							\param	callback_	Button callback
							\param	dismiss_	Whether the automatically dismiss the dialog when the button is chosen
							\param	title_		Title for the button (pass "" to use the default localized title), as
												UTF-8
						*/
						ButtonOptions(
							const bool enable_,
							const Callback& callback_,
							const bool dismiss_ = true,
							const std::string& title_ = ""
						);

						/**
							\brief	Destructs a ButtonOptions object
							\author	GW
							\date	09/2013
						*/
						virtual ~ButtonOptions();
						
						/**
							\brief		Overloaded assignment operator to copy values from one ButtonOptions object to
										another
							\author		GW
							\date		02/2015
							
							\param		rhs_	Righthand side of the assignment
							\returns	The lefthand side of the assignment, but with its values updated
						*/
						ButtonOptions& operator=(const ButtonOptions& rhs_);

						/**
							\brief		Gets the callback for the modal dialog button
							\author		GW
							\date		09/2013

							\returns	A pointer to the registered callback
						*/
						Callback* const callback() const;

						/**
							\brief	Sets the callback for the modal dialog button
							\author	GW
							\date	09/2013

							\param	cb_		New button callback
						*/
						void setCallback(const Callback& cb_);


					private:				
						/**
							\brief	Button callback
						*/
						Callback* __cb;
				};

				/**
					\brief	Constructs an empty ModalDialog object
					\author	GW
					\date	10/2013
					
					\note	To test if a ModalDialog object is empty, call isEmpty() on it
					\note	Empty ModalDialog objects do not relate to any actual UI modal dialog; they are designed to
							be "receivers" of some other ModalDialog object via the overloaded assignment operator.
							Empty ModalDialog objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				ModalDialog();

				/**
					\brief	Constructs a new ModalDialog object from an existing ModalDialog object (copy constructor)
					\author	GW
					\date	10/2013

					\param	md_		Existing ModalDialog object
				*/
				ModalDialog(const ModalDialog& md_);
			
				/**
					\brief	Constructs a ModalDialog object with a given name and size
					\author	GW
					\date	09/2013

					\param	name_	Name (title) of the dialog, as UTF-8
					\param	size_	Width and height, in 1x resolution coordinates

					\note	All modal dialogs are presented in the center of the screen.
					\note	This constructor creates a modal dialog with no buttons; it's up to you to add buttons that
							dismiss the dialog.
				*/
				ModalDialog(const std::string& name_, const Size& size_);
			
				/**
					\brief	Constructs a ModalDialog object with a given name, size, and buttons
					\author	GW
					\date	09/2013

					\param	name_			Name (title) of the dialog, as UTF-8
					\param	size_			Width and height, in 1x resolution coordinates
					\param	okBtnOpts_		Whether to show the OK button, what its title is, and what its callback
											should be
					\param	cancelBtnOpts_	Whether to show the cancel button, what its title is, and what its callback
											should be

					\note	All modal dialogs are presented in the center of the screen.
				*/
				ModalDialog(
					const std::string& name_,
					const Size& size_,
					const ButtonOptions& okBtnOpts_,
					const ButtonOptions& cancelBtnOpts_
				);

				/**
					\brief	Destructs a ModalDialog object
					\author	GW
					\date	09/2013
				*/
				virtual ~ModalDialog();

				/**
					\brief		Allows one ModalDialog object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ModalDialog object, but with its value updated to match that of rhs_
				*/
				virtual ModalDialog& operator=(const ModalDialog& rhs_);
				
				/**
					\brief		Convenience method to clone a ModalDialog object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ModalDialog object
					
					\note		If you subclass ModalDialog, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ModalDialog subclass, you will
								experience "object slicing" when e.g. adding the modal dialog to a window collection.

					\warning	The caller must manage the memory for the returned ModalDialog object.
				*/
				virtual ModalDialog* clone() const;

				/**
					\brief	Destroys the UI modal dialog, converting the target object to an empty ModalDialog object
							(see the default constructor for more info on empty ModalDialog objects)
					\author	GW
					\date	10/2013

					\note	This method must be called for any existing modal dialogs by the time the plugin is shutting
							down. If you no longer retain any of your own references to a modal dialog then the lib will
							call this method for you as it is shutting down.
				*/
				virtual void destroy();

				/**
					\brief		Gets the platform-specific modal dialog, around which the target object is wrapped
					\author		GW
					\date		11/2013

					\returns	Pointer to platform-specific modal dialog. See PlatformModalDialogPtr typedef.

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of UI functionality is not handled by this class (or related
								classes), then it should probably be added to the hdi_core library.
				*/
				virtual PlatformModalDialogPtr platformModalDialog() const;

				/**
					\brief		Gets whether the dialog is visible
					\author		GW
					\date		09/2013

					\returns	true if the dialog is visible, false otherwise
				*/
				virtual bool visible() const;
				
				/**
					\brief	Shows the modal dialog, blocking user input
					\author	GW
					\date	09/2013
				*/
				virtual void show();

				/**
					\brief	Forcefully dismisses a modal dialog
					\author	GW
					\date	09/2013
				*/
				virtual void dismiss();

				/**
					\brief		Adds a widget (label, button, text field, etc.) to the modal dialog
					\author		GW
					\date		09/2013

					\param		widget_		Widget to place in the dialog
					\returns	true if the widget was added, false otherwise
				*/
				virtual bool addWidget(const Widget& widget_);

				/**
					\brief		Removes a widget (label, button, text field, etc.) from the modal dialog
					\author		GW
					\date		09/2013

					\param		widget_		Widget to remove from the dialog
					\returns	true if the widget was removed, false otherwise
				*/
				virtual bool removeWidget(const Widget& widget_);

				/**
					\brief		Gets which button was used to dismiss the dialog
					\author		GW
					\date		09/2013

					\returns	An enum to describe which button dismissed the dialog
				*/
				virtual ChosenButton chosenButton() const;

				/**
					\brief		Gets the OK button created by the constructor (if the options indicated creation)
					\author		GW
					\date		07/2015

					\returns	A pointer to the OK Button object, or NULL if none
				*/
				std::unique_ptr<Button> okButton() const;

				/**
					\brief		Gets the cancel button created by the constructor (if the options indicated creation)
					\author		GW
					\date		07/2015

					\returns	A pointer to the cancel Button object, or NULL if none
				*/
				std::unique_ptr<Button> cancelButton() const;

				/**
					\brief		Gets the current modal dialog title
					\author		GW
					\date		09/2013
					
					\returns	String value for the modal dialog title, as UTF-8
				*/
				virtual std::string title() const;

				/**
					\brief	Sets the current modal dialog title
					\author	GW
					\date	09/2013
					
					\param	title_	New string value for the modal dialog title, as UTF-8
				*/
				virtual void setTitle(const std::string& title_);

				/**
					\brief		Gets the modal dialog's size
					\author		GW
					\date		11/2013

					\returns	A Size object representing the dialog's dimensions, in 1x resolution coordinates
				*/
				virtual Size size() const;

				/**
					\brief	Sets the modal dialog's size
					\author	GW
					\date	11/2013

					\param	size_	New dialog size, in 1x resolution coordinates
				*/
				virtual void setSize(const Size& size_);

				/**
					\brief	Forces the widgets in the modal dialog to update/redraw
					\author	GW
					\date	09/2013
				*/
				virtual void update() const;
		};
		
		typedef std::unique_ptr<ModalDialog> ModalDialogUP;
		typedef std::shared_ptr<ModalDialog> ModalDialogSP;
		typedef std::weak_ptr<ModalDialog> ModalDialogWP;
	}
}

#endif
// __HDI_CORE_MODAL_DIALOG__
