/**
	\file
	\brief		Header file for a class to create and manipulate Illustrator panels
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_PANEL__
#define __HDI_CORE_PANEL__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreImageIDs.h"
#include "hdicoreRect.h"
#include "hdicoreSVGIDs.h"
#include "hdicoreWindow.h"

namespace hdi
{
	namespace pui
	{
		class Panel;
	}
	
	namespace core
	{
		class Callback;
		class Flyout;
		class Widget;
		
		typedef	AIPanelRef PlatformPanelPtr;

		/**
			\brief	Used to add a panel to Illustrator's UI and contain various UI widgets
		*/
		class Panel : public Window
		{
			public:
				/**
					\brief	Most Illustrator panels are this width
				*/
				static const double typicalWidth;

				/**
					\brief	Constructs an empty Panel object
					\author	GW
					\date	10/2013
					
					\note	To test if a Panel object is empty, call isEmpty() on it
					\note	Empty Panel objects do not relate to any actual UI panel; they are designed to be
							"receivers" of some other Panel object via the overloaded assignment operator. Empty Panel
							objects are useless until such time (though it is safe to call any of their methods).
				*/
				Panel();

				/**
					\brief	Constructs a new Panel object from an existing Panel object (copy constructor)
					\author	GW
					\date	10/2013

					\param	p_	Existing Panel object
				*/
				Panel(const Panel& p_);

				/**
					\brief	Constructs a new Panel object
					\author	GW
					\date	09/2013

					\param	name_			Name/title of the panel (must be unique), as UTF-8
					\param	size_			Size (width and height) of the panel, in 1x resolution coordinates
					\param	resizableW_		Whether the panel width can be resized
					\param	resizableH_		Whether the panel height can be resized
					\param	iconIDs_		PNGI resource IDs for the dock icon

					\note	If resizable_ is true, then the provided width will be the minimum and maximum width, and
							the provided height will be the minimum height (no maximum height is set).
				*/
				Panel(
					const std::string& name_,
					const Size& size_,
					const bool resizableW_,
					const bool resizableH_,
					const ImageIDs& iconIDs_
				);
			
				/**
					\brief	Constructs a new Panel object
					\author	GW
					\date	10/2018

					\param	name_			Name/title of the panel (must be unique), as UTF-8
					\param	size_			Size (width and height) of the panel, in 1x resolution coordinates
					\param	resizableW_		Whether the panel width can be resized
					\param	resizableH_		Whether the panel height can be resized
					\param	iconIDs_		SVG resource IDs for the dock icon

					\note	If resizable_ is true, then the provided width will be the minimum and maximum width, and
							the provided height will be the minimum height (no maximum height is set).
				*/
				Panel(
					const std::string& name_,
					const Size& size_,
					const bool resizableW_,
					const bool resizableH_,
					const SVGIDs& iconIDs_
				);

				/**
					\brief	Destructs a Panel object
					\author	GW
					\date	09/2013
				*/
				virtual ~Panel();

				/**
					\brief		Allows one Panel object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Panel object, but with its value updated to match that of rhs_
				*/
				virtual Panel& operator=(const Panel& rhs_);
				
				/**
					\brief		Convenience method to clone a Panel object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Panel object
					
					\note		If you subclass Panel, you MUST overload this method yourself! If you don't and/or your
								clone() method does not return an instance of your Panel subclass, you will experience
								"object slicing" when e.g. adding the panel to a window collection.

					\warning	The caller must manage the memory for the returned Panel object.
				*/
				virtual Panel* clone() const;

				/**
					\brief	Destroys the UI panel, converting the target object to an empty Panel object (see the
							default constructor for more info on empty Panel objects)
					\author	GW
					\date	10/2013

					\note	This method must be called for any existing panels by the time the plugin is shutting down.
							If you no longer retain any of your own references to a panel then the lib will call this
							method for you as it is shutting down.
				*/
				virtual void destroy();

				/**
					\brief		Gets the platform-specific panel, around which the target object is wrapped
					\author		GW
					\date		11/2013

					\returns	Pointer to platform-specific panel. See PlatformPanelPtr typedef.

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of UI functionality is not handled by this class (or related
								classes), then it should probably be added to the hdi_core library.
				*/
				virtual PlatformPanelPtr platformPanel() const;

				/**
					\brief	Shows the panel
					\author	GW
					\date	09/2013
				*/
				virtual void show();

				/**
					\brief	Hides the panel
					\author	GW
					\date	09/2013
				*/
				virtual void hide();

				/**
					\brief		Gets whether the panel is visible
					\author		GW
					\date		09/2013

					\returns	true if the panel is visible, false otherwise
				*/
				virtual bool visible() const;

				/**
					\brief	Sets panel visibility
					\author	GW
					\date	09/2013

					\param	visible_	true to show the panel, false to hide

					\note	This method simply calls show() or hide() as appropriate
				*/
				virtual void setVisible(const bool visible_);

				/**
					\brief		Gets the panel's frame
					\author		GW
					\date		09/2013

					\returns	A Rect object representing the panel's location and dimensions, in 1x resolution
								coordinates
				*/
				virtual Rect frame() const;

				/**
					\brief	Sets the panel's frame (location and size)
					\author	GW
					\date	09/2013

					\param	frame_	New panel frame, in 1x resolution coordinates
				*/
				virtual void setFrame(const Rect& frame_);

				/**
					\brief	Sets the panel's dimensions
					\author	GW
					\date	09/2015

					\param	s_	New panel size, at 1x resolution
				*/
				virtual void setSize(const Size& s_);
				
				/**
					\brief	Sets the panel's height (keeping the location and width the same)
					\author	GW
					\date	09/2013

					\param	height_		New panel height, at 1x resolution 
				*/
				virtual void setHeight(const double height_);
			
				/**
					\brief		Gets the minimum panel size, if the panel is resizeable in any direction (otherwise, the
				 				minimum matches the current size)
					\author		GW
					\date		08/2018
				 
					\returns	The minimum panel dimensions, at 1x resolution
				*/
				virtual Size minimumSize() const;
			
				/**
					\brief	Sets the minimum panel size, if the panel is resizeable in any direction (otherwise, the
							minimum always matches the current size)
					\author	GW
					\date	08/2018
				 
					\param	min_	The new minimum panel dimensions, at 1x resolution; ignored if the panel is not
									resizeable
				*/
				virtual void setMinimumSize(const Size& min_);
			
				/**
					\brief		Gets the maximum panel size, if the panel is resizeable in any direction (otherwise, the
				 				maximum matches the current size)
					\author		GW
					\date		08/2018
				 
					\returns	The maximum panel dimensions, at 1x resolution
				*/
				virtual Size maximumSize() const;
			
				/**
					\brief	Sets the maximum panel size, if the panel is resizeable in any direction (otherwise, the
							maximum always matches the current size)
					\author	GW
					\date	08/2018
				 
					\param	min_	The new maximum panel dimensions, at 1x resolution; ignored if the panel is not
									resizeable
				*/
				virtual void setMaximumSize(const Size& max_);

				/**
					\brief		Adds a widget (label, button, text field, etc.) to the panel
					\author		GW
					\date		09/2013

					\param		widget_		Widget to place in the panel
					\returns	true if the widget was added successfully
				*/
				virtual bool addWidget(const Widget& widget_);

				/**
					\brief		Removes a widget (label, button, text field, etc.) from the panel
					\author		GW
					\date		09/2013

					\param		widget_		Widget to remove from the panel
					\returns	true if the widget was removed successfully
				*/
				virtual bool removeWidget(const Widget& widget_);

				/**
					\brief		Gets the current panel title
					\author		GW
					\date		12/2015
					
					\returns	String value for the panel title, as UTF-8
				*/
				std::string title() const;

				/**
					\brief	Sets the current panel title
					\author	GW
					\date	12/2015
					
					\param	title_	New string value for the panel title, as UTF-8
				*/
				void setTitle(const std::string& title_);

				/**
					\brief		Gets the current flyout menu for the panel, or NULL if none
					\author		GW
					\date		09/2013

					\returns	A Flyout object pointer, or NULL if none

					\note		The Panel object is not aware of any changes made to the returned Flyout object; you
								must call update() to "register" changes
				*/
				virtual Flyout* const flyoutMenu() const;

				/**
					\brief	Sets/updates the flyout menu for the panel
					\author	GW
					\date	09/2013

					\param	flyout_		New flyout menu
				*/
				virtual void setFlyoutMenu(const Flyout& flyout_);

				/**
					\brief		Gets the registered callback for the panel showing, or NULL if none
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered "panel show" callback
				*/
				virtual Callback* const showCallback() const;

				/**
					\brief	Sets/updates the callback for the panel showing
					\author	GW
					\date	09/2013

					\param	cb_		New callback for a user showing the panel
				*/
				virtual void setShowCallback(const Callback& cb_);

				/**
					\brief		Gets the registered callback for the panel hiding, or NULL if none
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered "panel hide" callback
				*/
				virtual Callback* const hideCallback() const;

				/**
					\brief	Sets/updates the callback for the panel hiding
					\author	GW
					\date	09/2013

					\param	cb_		New callback for a user hiding the panel
				*/
				virtual void setHideCallback(const Callback& cb_);

				/**
					\brief		Gets the callback set for the panel resize, or NULL if none
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered "panel resize" callback
				*/
				virtual Callback* const resizeCallback() const;

				/**
					\brief	Sets/updates the callback for the panel resize
					\author	GW
					\date	09/2013

					\param	cb_		New callback for a user resizing the panel
				*/
				virtual void setResizeCallback(const Callback& cb_);

				/**
					\brief	Forces the widgets in the panel to update/redraw (including the flyout, if any)
					\author	GW
					\date	09/2013
				*/
				virtual void update() const;


			private:
				friend class Flyout;
				friend pui::Panel* __accessImpl(const Panel&);
				friend Panel __accessCtor(pui::Panel* const);
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				void* __impl() const;

				/**
					\brief	Internal use only
					\author	GW
					\date	12/2015
				*/
				Panel(pui::Panel* const);
		};
		
		typedef std::unique_ptr<Panel> PanelUP;
		typedef std::shared_ptr<Panel> PanelSP;
		typedef std::weak_ptr<Panel> PanelWP;
		
		extern pui::Panel* __accessImpl(const Panel&);
		extern Panel __accessCtor(pui::Panel* const);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PANEL__
