/**
	\file
	\brief		Header file for path style-related classes
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_PATH_STYLE__
#define __HDI_CORE_PATH_STYLE__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtColor.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Describes the dash styling of a path
		*/
		class DashStyle
		{
			public:
				/**
					\brief	Maximum number of sublengths that can compose a dash style
				*/
				static const int16_t maxDashComponents = 6;

				/**
					\brief	Acts as a DashStyle value map, indicating which values are "known"
				*/
				class Map
				{
					public:
						/**
							\brief	Whether the dash components are known
						*/
						bool lengths[ maxDashComponents ];
						
						/**
							\brief	Whether the component count is known
						*/
						bool count;
						
						/**
							\brief	Whether the offset is known
						*/
						bool offset;
						
						/**
							\brief	Constructs a new Map object with values set to the given default
							\author	GW
							\date	08/2013
							
							\param	default_	Whether the values are known or unknown
						*/
						Map(const bool default_ = false);

						/**
							\brief	Constructs a new Map object from an existing one
							\author	GW
							\date	12/2013
							
							\param	map_	Existing Map object to copy values from
						*/
						Map(const Map& map_);

						/**
							\brief	Destructs a Map object
							\author	GW
							\date	08/2013
						*/
						virtual ~Map();

						/**
							\brief		Map assignment operator
							\author		GW
							\date		12/2013
							
							\param		rhs_	Existing Map object to copy values from
							\returns	The target Map object, but with its values updated
						*/
						Map& operator=(const Map& rhs_);
				};
		
				typedef std::unique_ptr<Map> MapUP;
				typedef std::shared_ptr<Map> MapSP;
				typedef std::weak_ptr<Map> MapWP;

				/**
					\brief	Lengths of each dash component, in points
				*/
				double lengths[ maxDashComponents ];
			
				/**
					\brief	Number of valid values in the lengths array
				*/
				int16_t count;
				
				/**
					\brief	Dash offset
				*/
				double offset;
				
				/**
					\brief	DashStyle constructor; sets all the lengths to 0.0, the count to 0, and the offset to 0.0
					\author	GW
					\date	08/2013
				*/
				DashStyle();

				/**
					\brief	Constructs a new DashStyle object from an existing one
					\author	GW
					\date	12/2013
					
					\param	style_	Existing DashStyle object to copy values from
				*/
				DashStyle(const DashStyle& style_);

				/**
					\brief	Destructs a DashStyle object
					\author	GW
					\date	08/2013
				*/
				virtual ~DashStyle();

				/**
					\brief		DashStyle assignment operator
					\author		GW
					\date		12/2013
					
					\param		rhs_	Existing DashStyle object to copy values from
					\returns	The target DashStyle object, but with its values updated
				*/
				DashStyle& operator=(const DashStyle& rhs_);
		};
		
		typedef std::unique_ptr<DashStyle> DashStyleUP;
		typedef std::shared_ptr<DashStyle> DashStyleSP;
		typedef std::weak_ptr<DashStyle> DashStyleWP;
		
		/**
			\brief	Describes the fill styling of a path
		*/
		class FillStyle
		{
			public:
				/**
					\brief	Acts as a FillStyle value map, indicating which values are "known"
				*/
				class Map
				{
					public:
						/**
							\brief	Whether the color is known
						*/
						bool color;
						
						/**
							\brief	Whether the overprint setting is known
						*/
						bool overprint;
						
						/**
							\brief	Constructs a new Map object with values set to the given default
							\author	GW
							\date	08/2013
							
							\param	default_	Whether the values are known or unknown
						*/
						Map(const bool default_ = false);

						/**
							\brief	Constructs a new Map object from an existing one
							\author	GW
							\date	12/2013
							
							\param	map_	Existing Map object to copy values from
						*/
						Map(const Map& map_);

						/**
							\brief	Destructs a Map object
							\author	GW
							\date	08/2013
						*/
						virtual ~Map();

						/**
							\brief		Map assignment operator
							\author		GW
							\date		12/2013
							
							\param		rhs_	Existing Map object to copy values from
							\returns	The target Map object, but with its values updated
						*/
						Map& operator=(const Map& rhs_);
				};
		
				typedef std::unique_ptr<Map> MapUP;
				typedef std::shared_ptr<Map> MapSP;
				typedef std::weak_ptr<Map> MapWP;
				
				/**
					\brief	Fill color
				*/
				ArtColor color;

				/**
					\brief		Controls whether fill overprint is enabled
					\details	Let art1 be the art with this fill style, and let art2 be the art below art1. When true,
								art2 will be printed, then art1 will be printed over it. When false, the portion of art2
								below art1 will be "knocked out" (not printed), but art1 will be printed.
				*/
				bool overprint;
				
				/**
					\brief	FillStyle constructor; sets the color to none and overprint to false
					\author	GW
					\date	08/2013
				*/
				FillStyle();

				/**
					\brief	Constructs a new FillStyle object from an existing one
					\author	GW
					\date	12/2013
					
					\param	style_	Existing FillStyle object to copy values from
				*/
				FillStyle(const FillStyle& style_);

				/**
					\brief	Destructs a FillStyle object
					\author	GW
					\date	08/2013
				*/
				virtual ~FillStyle();

				/**
					\brief		FillStyle assignment operator
					\author		GW
					\date		12/2013
					
					\param		rhs_	Existing FillStyle object to copy values from
					\returns	The target FillStyle object, but with its values updated
				*/
				FillStyle& operator=(const FillStyle& rhs_);
		};
		
		typedef std::unique_ptr<FillStyle> FillStyleUP;
		typedef std::shared_ptr<FillStyle> FillStyleSP;
		typedef std::weak_ptr<FillStyle> FillStyleWP;
		
		/**
			\brief	Describes the stroke styling of a path
		*/
		class StrokeStyle
		{
			public:
				/**
					\brief	Indicates the type of cap for a given stroke style
				*/
				enum CapStyle
				{
					ButtCap			= 100,
					RoundCap		= 110,
					ProjectingCap	= 120,
					SquareCap		= ProjectingCap
				};
				
				/**
					\brief	Indicates the type of join for a given stroke style
				*/
				enum JoinStyle
				{
					MiterJoin	= 100,
					RoundJoin	= 110,
					BevelJoin	= 120
				};
				
				/**
					\brief	Acts as a StrokeStyle value map, indicating which values are "known"
				*/
				class Map
				{
					public:
						/**
							\brief	Whether the color is known
						*/
						bool color;
						
						/**
							\brief	Whether the overprint setting is known
						*/
						bool overprint;
						
						/**
							\brief	Whether the width is known
						*/
						bool width;
						
						/**
							\brief	Dash style value mappings
						*/
						DashStyle::Map dash;
						
						/**
							\brief	Whether the cap setting is known
						*/
						bool cap;
						
						/**
							\brief	Whether the join setting is known
						*/
						bool join;
						
						/**
							\brief	Whether the miter limit is known
						*/
						bool miterLimit;
						
						/**
							\brief	Constructs a new Map object with values set to the given default
							\author	GW
							\date	08/2013
							
							\param	default_	Whether the values are known or unknown
						*/
						Map(const bool default_ = false);

						/**
							\brief	Constructs a new Map object from an existing one
							\author	GW
							\date	12/2013
							
							\param	map_	Existing Map object to copy values from
						*/
						Map(const Map& map_);

						/**
							\brief	Destructs a Map object
							\author	GW
							\date	08/2013
						*/
						virtual ~Map();

						/**
							\brief		Map assignment operator
							\author		GW
							\date		12/2013
							
							\param		rhs_	Existing Map object to copy values from
							\returns	The target Map object, but with its values updated
						*/
						Map& operator=(const Map& rhs_);
				};
		
				typedef std::unique_ptr<Map> MapUP;
				typedef std::shared_ptr<Map> MapSP;
				typedef std::weak_ptr<Map> MapWP;

				/**
					\brief	Stroke color
				*/
				ArtColor color;

				/**
					\brief	Controls whether stroke overprint is enabled
				*/
				bool overprint;

				/**
					\brief	Stroke width, in points
				*/
				double width;

				/**
					\brief	Stroke dash style
				*/
				DashStyle dash;

				/**
					\brief	Stroke cap style
				*/
				CapStyle cap;

				/**
					\brief	Stroke join style
				*/
				JoinStyle join;
				
				/**
					\brief	Stroke miter limit
				*/
				double miterLimit;
				
				/**
					\brief	StrokeStyle constructor; sets overprint to false, the width to 0.0, the color to none, the
							cap to ButtCap, the join to MiterJoin, and miterLimit to 0.0
					\author	GW
					\date	08/2013
				*/
				StrokeStyle();

				/**
					\brief	Constructs a new StrokeStyle object from an existing one
					\author	GW
					\date	12/2013
					
					\param	style_	Existing StrokeStyle object to copy values from
				*/
				StrokeStyle(const StrokeStyle& style_);

				/**
					\brief	Destructs a StrokeStyle object
					\author	GW
					\date	08/2013
				*/
				virtual ~StrokeStyle();

				/**
					\brief		StrokeStyle assignment operator
					\author		GW
					\date		12/2013
					
					\param		rhs_	Existing StrokeStyle object to copy values from
					\returns	The target StrokeStyle object, but with its values updated
				*/
				StrokeStyle& operator=(const StrokeStyle& rhs_);
		};
		
		typedef std::unique_ptr<StrokeStyle> StrokeStyleUP;
		typedef std::shared_ptr<StrokeStyle> StrokeStyleSP;
		typedef std::weak_ptr<StrokeStyle> StrokeStyleWP;
		
		/**
			\brief	Describes all styling attributes of a path
		*/
		class PathStyle
		{
			public:
				/**
					\brief	Acts as a StrokeStyle value map, indicating which values are "known"
				*/
				class Map
				{
					public:
						/**
							\brief	Whether the useFill setting is known
						*/
						bool useFill;
						
						/**
							\brief	FillStyle value mappings
						*/
						FillStyle::Map fill;
						
						/**
							\brief	Whether the useStroke setting is known
						*/
						bool useStroke;
						
						/**
							\brief	StrokeStyle value mappings
						*/
						StrokeStyle::Map stroke;
						
						/**
							\brief	Whether the evenodd setting is known
						*/
						bool evenodd;
						
						/**
							\brief	Whether the resolution is known
						*/
						bool resolution;

						/**
							\brief	Constructs a new Map object from an existing one
							\author	GW
							\date	12/2013
							
							\param	map_	Existing Map object to copy values from
						*/
						Map(const Map& map_);
						
						/**
							\brief	Constructs a new Map object with values set to the given default
							\author	GW
							\date	08/2013
							
							\param	default_	Whether the values are known or unknown
						*/
						Map(const bool default_ = false);

						/**
							\brief	Destructs a Map object
							\author	GW
							\date	08/2013
						*/
						virtual ~Map();

						/**
							\brief		Map assignment operator
							\author		GW
							\date		12/2013
							
							\param		rhs_	Existing Map object to copy values from
							\returns	The target Map object, but with its values updated
						*/
						Map& operator=(const Map& rhs_);
				};
		
				typedef std::unique_ptr<Map> MapUP;
				typedef std::shared_ptr<Map> MapSP;
				typedef std::weak_ptr<Map> MapWP;

				/**
					\brief	Whether or not to use fill
				*/
				bool useFill;
				
				/**
					\brief	Fill style
				*/
				FillStyle fill;
				
				/**
					\brief	Whether or not to use stroke
				*/
				bool useStroke;

				/**
					\brief	Stroke style
				*/
				StrokeStyle stroke;
				
				/**
					\brief	Whether to use the even-odd rule to determine path insideness
				*/
				bool evenodd;
				
				/**
					\brief	Resolution for the path
				*/
				double resolution;

				/**
					\brief	PathStyle constructor; sets useFill to false, useStroke to false, evenodd to false, and the
							resolution to 800.0
					\author	GW
					\date	08/2013
				*/
				PathStyle();

				/**
					\brief	Constructs a new PathStyle object from an existing one
					\author	GW
					\date	12/2013
					
					\param	style_	Existing PathStyle object to copy values from
				*/
				PathStyle(const PathStyle& style_);

				/**
					\brief	Destructs a PathStyle object
					\author	GW
					\date	08/2013
				*/
				virtual ~PathStyle();

				/**
					\brief		PathStyle assignment operator
					\author		GW
					\date		12/2013
					
					\param		rhs_	Existing PathStyle object to copy values from
					\returns	The target PathStyle object, but with its values updated
				*/
				PathStyle& operator=(const PathStyle& rhs_);
		};
		
		typedef std::unique_ptr<PathStyle> PathStyleUP;
		typedef std::shared_ptr<PathStyle> PathStyleSP;
		typedef std::weak_ptr<PathStyle> PathStyleWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_PATH_STYLE__
