/**
	\file
	\brief		Header file for creating entry objects to be added to Widget subclasses (that support them)
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_POPUP_ENTRY__
#define __HDI_CORE_POPUP_ENTRY__

#include "hdicoreListEntry.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows combo boxes, popups, and flyouts to store more data than simply strings and indices
		*/
		class PopupEntry : public ListEntry
		{
			public:
				/**
					\brief	Constructs an empty PopupEntry object
					\author	GW
					\date	07/2022
					
					\note	To test if an PopupEntry object is empty, call isEmpty() on it
					\note	Empty PopupEntry objects do not relate to any entry in any widget; they are designed to
							be "receivers" of some other PopupEntry object via the overloaded assignment operator. Empty
							PopupEntry objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				PopupEntry();
			
				/**
					\brief	Constructs a new PopupEntry object from an existing PopupEntry object (copy constructor)
					\author	GW
					\date	07/2022

					\param	e_	Existing PopupEntry object
				*/
				PopupEntry(const PopupEntry& e_);

				/**
					\brief	Constructs a PopupEntry object with some custom data
					\author	GW
					\date	07/2022

					\param	id_			Entry ID string (must be unique across all entries in the target container), as
										UTF-8
					\param	value_		Entry string value, as UTF-8
					\param	userData_	Data to store in the entry
					\param	enabled_	Whether the entry should be enabled
				*/
				PopupEntry(
					const std::string& id_,
					const std::string& value_,
					void* userData_,
					const bool enabled_
				);

				/**
					\brief	Destructs a PopupEntry object
					\author	GW
					\date	07/2022
				*/
				virtual ~PopupEntry();
				
				/**
					\brief		Assigns one PopupEntry object to another
					\author		GW
					\date		07/2022

					\param		rhs_	Existing PopupEntry object to copy values from
					\returns	The target PopupEntry object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual PopupEntry& operator=(const PopupEntry& rhs_);

				/**
					\brief		Gets the editability of the entry (whether the user can edit it inline)
					\author		GW
					\date		07/2022
					
					\returns	true for enabled, false otherwise
				*/
				virtual bool editable() const;

				/**
					\brief	Sets the editability of the entry (whether the user can edit it inline)
					\author	GW
					\date	07/2022
					
					\param	enabled_	true for enabled, false otherwise
				*/
				virtual void setEditable(const bool editable_);
				
				/**
					\brief		Tests whether a given PopupEntry object is the same as another
					\author		GW
					\date		07/2022

					\param		rhs_	PopupEntry to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same entry, false otherwise
				*/
				virtual bool operator==(const PopupEntry& rhs_) const;
				
				/**
					\brief		Tests whether a given PopupEntry object is not the same as another
					\author		GW
					\date		07/2022

					\param		rhs_	PopupEntry to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different entries, false otherwise
				*/
				virtual bool operator!=(const PopupEntry& rhs_) const;
		};
		
		typedef std::unique_ptr<PopupEntry> PopupEntryUP;
		typedef std::shared_ptr<PopupEntry> PopupEntrySP;
		typedef std::weak_ptr<PopupEntry> PopupEntryWP;
	}
}

#endif
// __HDI_CORE_POPUP_ENTRY__
