/**
	\file
	\brief		Header file for plugin preferences storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_PREFERENCES__
#define __HDI_CORE_PREFERENCES__

#include <memory>
#include <vector>

#include "hdicorePathStyle.h"

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Preferences;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class Preferences;
		}
	#endif

	namespace core
	{
		class Color;
		class Font;
		class Point;
		class Rect;
		class Size;

		class PrefData;
		class ArrayPref;
		class BoolPref;
		class ColorPref;
		class DictionaryPref;
		class FontPref;
		class NumberPref;
		class PointPref;
		class RectPref;
		class SizePref;
		class StringPref;
		
		#if defined(HDI_CORE_AIP_MODE)
			class ArtboardPoint;
			class ArtboardRect;
			class ArtColor;
			class TransformMatrix;
		
			class ArtboardPointPref;
			class ArtboardRectPref;
			class ArtColorPref;
			class DashStylePref;
			class DashStyleMapPref;
			class FillStylePref;
			class FillStyleMapPref;
			class PathStylePref;
			class PathStyleMapPref;
			class StrokeStylePref;
			class StrokeStyleMapPref;
			class TransformMatrixPref;
		#endif
		// HDI_CORE_AIP_MODE
	
		/**
			\brief	Aids in preference storage by reading/writing the JSON data in a given file
		*/
		class Preferences
		{
			public:
				typedef std::vector< std::shared_ptr<PrefData> > PrefDataVector;
			
 				/**
					\brief	Allows for iteration through the members of a Preferences object, a la std::vector iteration

					\code
						// Here's an example, iterating through the children of prefs:
						for(hdi::core::Preferences::Iterator iter = prefs.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff with iter.key() or iter.value()
						}
					\endcode
				*/
				class Iterator
				{
					public:
						/**
							\brief	Constructs an Iterator object from an existing Iterator object (copy constructor)
							\author	GW
							\date	05/2018

							\param	iter_	Existing Iterator object
						*/
						Iterator(const Iterator& iter_);
					
						/**
							\brief	Destructs an Iterator object
							\author	GW
							\date	05/2018
						*/
						~Iterator();
					
						/**
							\brief		Assigns one Iterator object to another
							\author		GW
							\date		05/2018

							\param		rhs_	Existing Iterator object to copy values from
							\returns	The target Iterator object, but with its values updated to match that of the
										rhs_ argument
						*/
						Iterator& operator=(const Iterator& rhs_);
					
						/**
							\brief		Tests whether a given Iterator object is the same as another
							\author		GW
							\date		05/2018

							\param		rhs_	Iterator to compare against (righthand side of equality operator)
							\returns	true for the target and rhs_ having the same state, false otherwise
						*/
						bool operator==(const Iterator& rhs_) const;
					
						/**
							\brief		Tests whether a given Iterator object is not the same as another
							\author		GW
							\date		05/2018

							\param		rhs_	Iterator to compare against (righthand side of inequality operator)
							\returns	true for the target and rhs_ having differing states, false otherwise
						*/
						bool operator!=(const Iterator& rhs_) const;

						/**
							\brief		Prefix increment operator for forward iteration
							\author		GW
							\date		05/2018
						 
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator++();

						/**
							\brief		Postfix increment operator for forward iteration
							\author		GW
							\date		05/2018
						 
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator++(int);

						/**
							\brief		Prefix decrement operator for reverse iteration
							\author		GW
							\date		05/2018
						 
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator--();

						/**
							\brief		Postfix decrement operator for reverse iteration
							\author		GW
							\date		05/2018
						 
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator--(int);

						/**
							\brief		Determines whether iteration should stop; works well as the conditional in a
										for() loop
							\author		GW
							\date		05/2018
						 
							\returns	true if at the end of the list of members, false otherwise
						*/
						bool atEnd() const;

						/**
							\brief		Gets the current member's key as pointed to by the iterator
							\author		GW
							\date		05/2018

							\returns	The current member's key, or "" if at the end
						*/
						std::string key() const;

						/**
							\brief		Gets the current member's value as pointed to by the iterator
							\author		GW
							\date		05/2018

							\returns	The current member's value, or NULL if at the end
						*/
						std::unique_ptr<PrefData> value() const;


					private:
						// Only Preferences can construct a new Iterator object
						friend class Preferences;

						/**
							\brief	Private implementation data
						*/
						void* __data;
				
						/**
							\brief	Empty Iterator objects cannot be constructed
							\author	GW
							\date	05/2018
						*/
						Iterator();
					
						/**
							\brief	Creates a new iterator for a Preferences object
							\author	GW
							\date	05/2018

							\param	prefs_	Preferences to be iterated through
						*/
						explicit Iterator(const Preferences& set_);
				};
		
				typedef std::unique_ptr<Iterator> IteratorUP;
				typedef std::shared_ptr<Iterator> IteratorSP;
				typedef std::weak_ptr<Iterator> IteratorWP;
			
				/**
					\brief	Constructs an empty Preferences object
					\author	GW
					\date	09/2013
					
					\note	To test if a Preferences object is empty, call isEmpty() on it
					\note	Empty Preferences objects do not relate to any actual preferences file; they are designed to
							be "receivers" of some other Preferences object via the overloaded assignment operator.
							Empty Preferences objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				Preferences();
			
				/**
					\brief	Constructs a new Preferences object from an existing Preferences object (copy constructor)
					\author	GW
					\date	09/2013

					\param	prefs_	Existing Preferences object
				*/
				Preferences(const Preferences& prefs_);

				/**
					\brief	Destructs a Preferences object
					\author	GW
					\date	09/2013
				*/
				virtual ~Preferences();
				
				/**
					\brief		Assigns one Preferences object to another
					\author		GW
					\date		09/2013

					\param		rhs_	Existing Preferences object to copy values from
					\returns	The target Preferences object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual Preferences& operator=(const Preferences& rhs_);

				/**
					\brief		Gets whether the target Preferences object is empty (constructed with the default ctor)
					\author		GW
					\date		09/2013
					
					\returns	true if the target Preferences object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Convenience method to create a Preferences object from JSON data in a file
					\author		GW
					\date		09/2013

					\param		path_		Path to the preferences file, as UTF-8
					\param		prefs__		Return-by-reference for the Preferences object for the file located at path_
					\returns	true if the file could be created (if necessary) and if the Preferences object was
								acquired, false otherwise

					\note		The file need not exist before calling this method.
				*/
				static bool load(const std::string& path_, Preferences& prefs__);
			
				/**
					\brief		Reloads the preferences data from disk (as it is kept in memory at runtime, this might be
								necessary if another process or plugin manipulates the file)
					\author		GW
					\date		01/2016
					
					\param		path_	Path to the preferences file, as UTF-8
					\returns	true if the prefs could be reloaded from disk, false otherwise
					
					\warning	Any previously acquired PrefData object from the target will become invalid after this
								method has finished (as the internal data is necessarily destroyed and recreated as part
								of reloading).
					
					\note		The given path, in most circumstances, should be the same as given to the load() method!
				*/
				virtual bool reload(const std::string& path_);

				/**
					\brief		Writes JSON preferences data to a file
					\author		GW
					\date		09/2013

					\param		path_	Path to the preferences file, as UTF-8
					\returns	true if the save operation is successful, false otherwise
				*/
				virtual bool save(const std::string& path_) const;

				/**
					\brief		Gets the save count of the prefs file
					\author		GW
					\date		09/2013

					\returns	The number of times the prefs file has been saved
				*/
				virtual uint64_t saveCount() const;

				/**
					\brief		Adds a preference object into the prefs data tree, such that when the prefs are written
								the individual pref data will be present
					\author		GW
					\date		09/2013

					\param		pref_	New preference to add into the top-level of the prefs data tree
					\returns	true if the pref was added, or false otherwise

					\note		The pref is added to the top-most level of the prefs data tree, with any children of its
								own appearing underneath the pref_ member.
				*/
				virtual bool addPref(const PrefData& pref_);
				
				/**
					\brief		Gets a single preference object from the prefs data tree
					\author		GW
					\date		09/2013

					\param		key_	Name of the member to get from the top-level of the prefs data tree, as UTF-8
					\param		pref__	Return-by-reference for the preference found with the given name
					\returns	true if the preference could be found, false otherwise

					\note		The type of the returned object can be determined from its dataType() method, such that
								correct casting can be accomplished.
				*/
				virtual bool getPref(const std::string& key_, std::unique_ptr<PrefData>& pref__) const;

				/**
					\brief		Removes a preference from the prefs data tree
					\author		GW
					\date		09/2013

					\param		pref_	Object for the pref to remove from the top-level of the prefs data tree
					\returns	true if the pref is removed successfully, false otherwise
				*/
				virtual bool removePref(const PrefData& pref_);

				/**
					\brief		Gets an iterator for the Preferences object
					\author		GW
					\date		05/2018
				 
					\returns	An Iterator object, for iterating through the target's members
				 
					\code
						// Here's an example, iterating through the children of prefs:
						for(hdi::core::Preferences::Iterator iter = prefs.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff with iter.key() or iter.value()
						}
					\endcode
				*/
				virtual Iterator begin() const;
				
				/**
					\brief		Gets an array pref from the prefs data tree
					\author		GW
					\date		09/2013

					\param		key_		UTF-8 name/key of the array to get from the top-level and return as a
											neatly-packed PrefData subclass
					\param		array__		Return-by-reference for the array pref identified by key_
					\returns	true if the array could be acquired, false otherwise
				*/
				virtual bool getArrayPref(const std::string& key_, ArrayPref& array__) const;
				
				/**
					\brief		Similar to getArrayPref(), except that, if the pref does not already exist, it will be
								created anew with a series of default elements
					\author		GW
					\date		09/2013

					\param		key_		UTF-8 name/key of the array to get from the top-level and return as a neatly-
											packed PrefData subclass
					\param		elements_	Default elements for the array, in case it didn't already exist (NULL, or an
											empty vector, for none)
					\param		array__		Return-by-reference for the array pref identified by key_
					\returns	true if the array could be acquired or created, false otherwise

					\note		The order of the default elements in the array will match that of the passed elements_
								argument.
				*/
				virtual bool getOrCreateArrayPref(
					const std::string& key_,
					const PrefDataVector* elements_,
					ArrayPref& array__
				);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets an artboard point pref from the prefs data tree
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the artboard point to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		abp__	Return-by-reference for the artboard point pref identified by key_
						\returns	true if the artboard point could be acquired, false otherwise
					*/
					virtual bool getArtboardPointPref(const std::string& key_, ArtboardPointPref& abp__) const;

					/**
						\brief		Similar to getArtboardPointPref(), except that, if the pref does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the artboard point to get from the top-level and return as
												a neatly-packed PrefData subclass
						\param		default_	Default value for the pref, in case it didn't already exist
						\param		abp__		Return-by-reference for the artboard point pref identified by key_
						\returns	true if the pref could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtboardPointPref(
						const std::string& key_,
						const ArtboardPoint& default_,
						ArtboardPointPref& abp__
					);

					/**
						\brief		Gets an artboard rect pref from the prefs data tree
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the artboard rect to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		abr__	Return-by-reference for the artboard rect pref identified by key_
						\returns	true if the artboard rect could be acquired, false otherwise
					*/
					virtual bool getArtboardRectPref(const std::string& key_, ArtboardRectPref& abr__) const;

					/**
						\brief		Similar to getArtboardRectPref(), except that, if the pref does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the artboard rect to get from the top-level and return as
												a  neatly-packed PrefData subclass
						\param		default_	Default value for the pref, in case it didn't already exist
						\param		abr__		Return-by-reference for the artboard rect pref identified by key_
						\returns	true if the pref could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtboardRectPref(
						const std::string& key_,
						const ArtboardRect& default_,
						ArtboardRectPref& abr__
					);

					/**
						\brief		Gets an art color pref from the prefs data tree
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the art color to get from the top-level and return as a neatly-
											packed PrefData subclass
						\param		ac__	Return-by-reference for the art color pref identified by key_
						\returns	true if the art color could be acquired, false otherwise
					*/
					virtual bool getArtColorPref(const std::string& key_, ArtColorPref& ac__) const;

					/**
						\brief		Similar to getArtColorPref(), except that, if the pref does not already exist, it will
									be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the art color to get from the top-level and return as a
												neatly-packed PrefData subclass
						\param		default_	Default value for the pref, in case it didn't already exist
						\param		ac__		Return-by-reference for the art color pref identified by key_
						\returns	true if the pref could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtColorPref(
						const std::string& key_,
						const ArtColor& default_,
						ArtColorPref& ac__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a bool pref from the prefs data tree
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the bool to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		b__		Return-by-reference for the bool pref identified by key_
					\returns	true if the bool could be acquired, false otherwise
				*/
				virtual bool getBoolPref(const std::string& key_, BoolPref& b__) const;

				/**
					\brief		Similar to getBoolPref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the bool to get from the top-level and return as a
											neatly-packed PrefData subclass
					\param		default_	Default value for the pref, in case it didn't already exist
					\param		b__			Return-by-reference for the bool pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateBoolPref(const std::string& key_, const bool default_, BoolPref& b__);

				/**
					\brief		Gets a color pref from the prefs data tree
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the color to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		c__		Return-by-reference for the color pref identified by key_
					\returns	true if the color could be acquired, false otherwise
				*/
				virtual bool getColorPref(const std::string& key_, ColorPref& c__) const;

				/**
					\brief		Similar to getColorPref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the color to get from the top-level and return as a neatly-
											packed PrefData subclass
					\param		default_	Default value for the pref, in case it didn't already exist
					\param		c__			Return-by-reference for the color pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateColorPref(const std::string& key_, const Color& default_, ColorPref& c__);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent dash style object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the dash style to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the dash style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getDashStylePref(const std::string& key_, DashStylePref& ds__);
				
					/**
						\brief		Similar to getDashStylePref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the dash style to get from the top-level and return as a
												neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the dash style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateDashStylePref(
						const std::string& key_,
						const DashStyle& default_,
						DashStylePref& ds__
					);
				
					/**
						\brief		Gets a persistent dash style map object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the dash style map to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the dash style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getDashStyleMapPref(const std::string& key_, DashStyleMapPref& ds__);
				
					/**
						\brief		Similar to getDashStyleMapPref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the dash style map to get from the top-level and return as a
												neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the dash style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateDashStyleMapPref(
						const std::string& key_,
						const DashStyle::Map& default_,
						DashStyleMapPref& ds__
					);
				#endif
				// HDI_CORE_AIP_MODE
				
				/**
					\brief		Gets a dictionary pref from the prefs data tree
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the dictionary to get from the top-level and return as a 
										neatly-packed PrefData subclass
					\param		dict__	Return-by-reference for the dictionary pref identified by key_
					\returns	true if the dictionary could be acquired, false otherwise
				*/
				virtual bool getDictPref(const std::string& key_, DictionaryPref& dict__) const;
				
				/**
					\brief		Similar to getDictPref(), except that, if the pref does not already exist, it will be
								created anew with a series of default members
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the dictionary to get from the top-level and return as a 
											neatly-packed PrefData subclass
					\param		members_	Default members for the dictionary, in case it didn't already exist (NULL,
											or an empty vector, for none)
					\param		dict__		Return-by-reference for the dictionary pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise

					\note		The names of the members in the dictionary will match that of the individual members in
								the passed members_ argument. Provided members with identical names may case undefined
								behavior.
				*/
				virtual bool getOrCreateDictPref(
					const std::string& key_,
					const PrefDataVector* members_,
					DictionaryPref& dict__
				);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent fill style object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the fill style to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the fill style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getFillStylePref(const std::string& key_, FillStylePref& fs__);
				
					/**
						\brief		Similar to getFillStylePref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the fill style to get from the top-level and return as a
												neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the fill style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateFillStylePref(
						const std::string& key_,
						const FillStyle& default_,
						FillStylePref& fs__
					);
				
					/**
						\brief		Gets a persistent fill style map object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the fill style map to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the fill style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getFillStyleMapPref(const std::string& key_, FillStyleMapPref& fs__);
				
					/**
						\brief		Similar to getFillStyleMapPref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the fill style map to get from the top-level and return as
												a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the fill style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateFillStyleMapPref(
						const std::string& key_,
						const FillStyle::Map& default_,
						FillStyleMapPref& fs__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a font pref from the prefs data tree
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the font to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		font__	Return-by-reference for the font pref identified by key_
					\returns	true if the font could be acquired, false otherwise
				*/
				virtual bool getFontPref(const std::string& key_, FontPref& font__) const;

				/**
					\brief		Similar to getFontPref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_			UTF-8 name/key of the font to get from the top-level and return as a 
												neatly-packed PrefData subclass
					\param		defaultFont_	Default font for the pref, in case it didn't already exist
					\param		defaultSize_	Default size for the pref, in case it didn't already exist
					\param		font__			Return-by-reference for the font pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateFontPref(
					const std::string& key_,
					const Font& defaultFont_,
					const double defaultSize_,
					FontPref& font__
				);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a transform matrix pref from the prefs data tree
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the matrix to get from the top-level and return as a neatly-
											packed PrefData subclass
						\param		tm__	Return-by-reference for the transform matrix pref identified by key_
						\returns	true if the transform matrix could be acquired, false otherwise
					*/
					virtual bool getTransformMatrixPref(const std::string& key_, TransformMatrixPref& tm__) const;

					/**
						\brief		Similar to getTransformMatrixPref(), except that, if the pref does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the matrix to get from the top-level and return as a 
												neatly-packed PrefData subclass
						\param		default_	Default value for the pref, in case it didn't already exist
						\param		tm__		Return-by-reference for the transform matrix pref identified by key_
						\returns	true if the pref could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateTransformMatrixPref(
						const std::string& key_,
						const TransformMatrix& default_,
						TransformMatrixPref& tm__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a number pref from the prefs data tree
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the number to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		n__		Return-by-reference for the number pref identified by key_
					\returns	true if the number could be acquired, false otherwise
				*/
				virtual bool getNumberPref(const std::string& key_, NumberPref& n__) const;

				/**
					\brief		Similar to getNumberPref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the number to get from the top-level and return as a 
											neatly-packed PrefData subclass
					\param		default_	Default integer value for the pref, in case it didn't already exist
					\param		n__			Return-by-reference for the number pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateIntPref(const std::string& key_, const int64_t default_, NumberPref& n__);

				/**
					\brief		Similar to getNumberPref(), except that, if the pref does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		05/2012
					
					\param		key_		UTF-8 name/key of the number to get from the top-level and return as a 
											neatly-packed PrefData subclass
					\param		default_	Default double value for the pref, in case it didn't already exist
					\param		n__			Return-by-reference for the number pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateFloatPref(const std::string& key_, const double default_, NumberPref& n__);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent path style object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the path style to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the path style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getPathStylePref(const std::string& key_, PathStylePref& ps__);
				
					/**
						\brief		Similar to getPathStylePref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the path style to get from the top-level and return as a
												neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the path style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreatePathStylePref(
						const std::string& key_,
						const PathStyle& default_,
						PathStylePref& ps__
					);
				
					/**
						\brief		Gets a persistent path style map object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the path style map to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the path style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getPathStyleMapPref(const std::string& key_, PathStyleMapPref& ps__);
				
					/**
						\brief		Similar to getPathStyleMapPref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the path style map to get from the top-level and return as
												a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the path style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreatePathStyleMapPref(
						const std::string& key_,
						const PathStyle::Map& default_,
						PathStyleMapPref& ps__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a point pref from the prefs data tree
					\author		GW
					\date		05/2012
					
					\param		key_	UTF-8 name/key of the point to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		p__		Return-by-reference for the point pref identified by key_
					\returns	true if the point could be acquired, false otherwise
				*/
				virtual bool getPointPref(const std::string& key_, PointPref& p__) const;

				/**
					\brief		Similar to getPointPref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		05/2012
					
					\param		key_		UTF-8 name/key of the point to get from the top-level and return as a neatly-
											packed PrefData subclass
					\param		default_	Default value for the pref, in case it didn't already exist
					\param		p__			Return-by-reference for the point pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreatePointPref(const std::string& key_, const Point& default_, PointPref& p__);

				/**
					\brief		Gets a rect pref from the prefs data tree
					\author		GW
					\date		05/2012
					
					\param		key_	UTF-8 name/key of the rect to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		r__		Return-by-reference for the rectangle pref identified by key_
					\returns	true if the rectangle could be acquired, false otherwise
				*/
				virtual bool getRectPref(const std::string& key_, RectPref& r__) const;

				/**
					\brief		Similar to getRectPref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		05/2012
					
					\param		key_		UTF-8 name/key of the rect to get from the top-level and return as a neatly-
											packed PrefData subclass
					\param		default_	Default value for the pref, in case it didn't already exist
					\param		r__			Return-by-reference for the rect pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateRectPref(const std::string& key_, const Rect& default_, RectPref& r__);

				/**
					\brief		Gets a size pref from the prefs data tree
					\author		GW
					\date		05/2011
					
					\param		key_	UTF-8 name/key of the size to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		s__		Return-by-reference for the size pref identified by key_
					\returns	true if the size could be acquired, false otherwise
				*/
				virtual bool getSizePref(const std::string& key_, SizePref& s__) const;

				/**
					\brief		Similar to getSizePref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		05/2012
					
					\param		key_		UTF-8 name/key of the size to get from the top-level and return as a neatly-
											packed PrefData subclass
					\param		default_	Default value for the pref, in case it didn't already exist
					\param		s__			Return-by-reference for the size pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateSizePref(const std::string& key_, const Size& default_, SizePref& s__);

				/**
					\brief		Gets a string pref from the prefs data tree
					\author		GW
					\date		05/2011
					
					\param		key_	UTF-8 name/key of the string to get from the top-level and return as a neatly-
										packed PrefData subclass
					\param		str__	Return-by-reference for the string pref identified by key_
					\returns	true if the string could be acquired, false otherwise
				*/
				virtual bool getStringPref(const std::string& key_, StringPref& str__) const;

				/**
					\brief		Similar to getStringPref(), except that, if the pref does not already exist, it will be
								created anew with a default value
					\author		GW
					\date		05/2012
					
					\param		key_		UTF-8 name/key of the string to get from the top-level and return as a 
											neatly-packed PrefData subclass
					\param		default_	Default UTF-8 value for the pref, in case it didn't already exist
					\param		str__		Return-by-reference for the string pref identified by key_
					\returns	true if the pref could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateStringPref(
					const std::string& key_,
					const std::string& default_,
					StringPref& str__
				);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent stroke style object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the stroke style to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the stroke style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getStrokeStylePref(const std::string& key_, StrokeStylePref& ss__);
				
					/**
						\brief		Similar to getStrokeStylePref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the stroke style to get from the top-level and return as a
												neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the stroke style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateStrokeStylePref(
						const std::string& key_,
						const StrokeStyle& default_,
						StrokeStylePref& ss__
					);
				
					/**
						\brief		Gets a persistent stroke style map object from the prefs data tree
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the stroke style map to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the stroke style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getStrokeStyleMapPref(const std::string& key_, StrokeStyleMapPref& ss__);
				
					/**
						\brief		Similar to getStrokeStyleMapPref(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the stroke style map to get from the top-level and return as
												a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		c__			Return-by-reference for the stroke style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateStrokeStyleMapPref(
						const std::string& key_,
						const StrokeStyle::Map& default_,
						StrokeStyleMapPref& ss__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets the number version for the preferences data
					\author		GW
					\date		09/2013

					\returns	Stored version number, or 0 if none
				*/
				virtual int32_t versionNumber() const;

				/**
					\brief		Sets the number version for the preferences data
					\author		GW
					\date		09/2013

					\param		version_	New number version value
					\returns	true if the number is successfully updated, or false otherwise
				*/
				virtual bool setVersionNumber(const int32_t version_);

				/**
					\brief		Gets the version string for the preferences data
					\author		GW
					\date		09/2013

					\returns	Stored version string, or "" if none
				*/
				virtual std::string versionString() const;

				/**
					\brief		Sets the version string for the preferences data
					\author		GW
					\date		09/2013

					\param		version_	New string version value
					\returns	true if the string is successfully updated, or false otherwise
				*/
				virtual bool setVersionString(const std::string& version_);


			protected:
				/**
					\brief	Private implementation data
				*/
				void* __data;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Internal use only
						\author	GW
						\date	09/2013
					*/
					Preferences(const aip::Preferences&);
				#elif defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Internal use only
						\author	GW
						\date	09/2013
					*/
					Preferences(const psp::Preferences&);
				#endif
		};
		
		typedef std::unique_ptr<Preferences> PreferencesUP;
		typedef std::shared_ptr<Preferences> PreferencesSP;
		typedef std::weak_ptr<Preferences> PreferencesWP;
	}
}

#endif
// __HDI_CORE_PREFERENCES__
