/**
	\file
	\brief		Header file for plugin user interface radio button widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_RADIOBUTTON__
#define __HDI_CORE_RADIOBUTTON__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class RadioButtonGroup;

		/**
			\brief	Radio button widget, which has both a radio button and label to its right. After instantiation, a
					RadioButton object can be added to a RadioButtonGroup.
		*/
		class RadioButton : public Widget
		{
			public:
				/**
					\brief	Describes the state of a radio button
				*/
				enum StateType
				{
					StateMixed	= -1,
					StateOff	= 0,
					StateOn		= 1
				};
				
				/**
					\brief	Constructs an empty RadioButton object
					\author	GW
					\date	09/2013
					
					\note	To test if a RadioButton object is empty, call isEmpty() on it
					\note	Empty RadioButton objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other RadioButton object via the overloaded assignment operator. Empty
							RadioButton objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				RadioButton();

				/**
					\brief	Constructs a new RadioButton object from an existing RadioButton object (copy constructor)
					\author	GW
					\date	09/2013

					\param	radio_	Existing RadioButton object
				*/
				RadioButton(const RadioButton& radio_);

				/**
					\brief	Constructs a text radio button, using a StateType for the state and a window container type
					\author	GW
					\date	09/2013

					\param	loc_			Position of the radio, in 1x resolution coordinates
					\param	title_			Title of the radio, as UTF-8
					\param	containerType_	Type of window that will contain the widget
					\param	state_			Initial radio button state
					
					\note	The containerType_ argument is necessary for this constructor because it computes the ideal
							width for the new radio button from the initial label string. However, Adobe uses different
							font sizes for different types of windows, and we can't correctly compute the width without
							knowing what font size to use.
				*/
				explicit RadioButton(
					const Point& loc_,
					const std::string& title_,
					const WindowType containerType_,
					const StateType state_ = StateOff
				);

				/**
					\brief	Constructs a text radio button, using a StateType for the state
					\author	GW
					\date	09/2013

					\param	loc_	Position of the radio, in 1x resolution coordinates
					\param	title_	Title of the radio, as UTF-8
					\param	width_	Width of the button, at 1x resolution
					\param	state_	Initial radio button state
				*/
				explicit RadioButton(
					const Point& loc_,
					const std::string& title_,
					const double width_,
					const StateType state_ = StateOff
				);

				/**
					\brief	Constructs an image radio button, using a bool for the state
					\author	GW
					\date	09/2013

					\param	frame_			Location and size of the radio, in 1x resolution coordinates
					\param	imageIDs_		PNGI resource IDs
					\param	hasBorders_		Whether the radio button will be drawn with borders
					\param	chosen_			Whether the radio button is chosen
				*/
				explicit RadioButton(
					const Rect& frame_,
					const ImageWithRolloverIDs& imageIDs_,
					const bool hasBorders_,
					const bool chosen_ = false
				);

				/**
					\brief	Constructs an image radio button, using a bool for the state
					\author	GW
					\date	12/2018

					\param	frame_			Location and size of the radio, in 1x resolution coordinates
					\param	svgIDs_			SVG resource IDs
					\param	hasBorders_		Whether the radio button will be drawn with borders
					\param	chosen_			Whether the radio button is chosen
					\param	dims_			1x dimensions to use when rendering the SVG; the default value causes the widget
											frame dimensions to be used instead
				*/
				explicit RadioButton(
					const Rect& frame_,
					const SVGWithRolloverIDs& svgIDs_,
					const bool hasBorders_,
					const bool chosen_ = false,
					const Size& dims_ = Size()
				);

				/**
					\brief	Radio button destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~RadioButton();

				/**
					\brief		Allows one RadioButton object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target RadioButton object, but with its value updated to match that of rhs_
				*/
				virtual RadioButton& operator=(const RadioButton& rhs_);
				
				/**
					\brief		Convenience method to clone a RadioButton object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new RadioButton object
					
					\note		If you subclass RadioButton, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your RadioButton subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned RadioButton object.
				*/
				virtual RadioButton* clone() const;
				
				/**
					\brief		Convenience method to duplicate a RadioButton object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new RadioButton object (and new UI element)
					
					\warning	The caller must manage the memory for the returned RadioButton object.
				*/
				virtual RadioButton* duplicate() const;
				
				/**
					\brief		Gets the radio button state
					\author		GW
					\date		09/2013

					\returns	Current radio button state
				*/
				virtual StateType state() const;
				
				/**
					\brief	Sets the radio button state
					\author	GW
					\date	09/2013

					\param	state_	New radio button state
				*/
				virtual void setState(const StateType state_);
				
				/**
					\brief	Alternate setter for the radio button state
					\author	GW
					\date	09/2013

					\param	chosen_		true if the radio button is chosen, false otherwise

					\note	If you need to set a mixed state, use the other setter
				*/
				virtual void setChosen(const bool chosen_);

				/**
					\brief		Gets the current text value of the radio button
					\author		GW
					\date		09/2013
					
					\returns	The current text value, as UTF-8

					\note		If the widget is an image radio button, this returns ""
				*/
				virtual std::string text() const;

				/**
					\brief	Sets the current text value of the radio button
					\author	GW
					\date	09/2013

					\param	text_	New text value, as UTF-8

					\note	If the widget is an image radio button, the function will bail early
				*/
				virtual void setText(const std::string& text_);
	
				/**
					\brief		Gets whether an image radio button has borders
					\author		GW
					\date		09/2013

					\returns	true if the button has borders, false otherwise
				*/
				virtual bool hasBorders() const;

				/**
					\brief		Gets the radio button group that the target RadioButton belongs to
					\author		GW
					\date		09/2013

					\returns	A pointer to the radio button group for the target object, or NULL if none
				*/
				virtual RadioButtonGroup* const group() const;

				/**
					\brief	Simulates a click on the button
					\author	GW
					\date	09/2013
				*/
				virtual void click();
				
				/**
					\brief		Gets the current click callback for the button
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered click callback
				*/
				virtual Callback* const clickCallback() const;
				
				/**
					\brief	Sets the click callback for the button
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the button is clicked
				*/
				virtual void setClickCallback(const Callback& callback_);
			
				/**
					\brief		Gets the PNGI resource IDs of the widget
					\author		GW
					\date		03/2015

					\returns	The IDs of the PNGI image resources
				*/
				virtual ImageWithRolloverIDs imageIDs() const;
			
				/**
					\brief	Sets the PNGI resource IDs of the widget
					\author	GW
					\date	03/2015

					\param	ids_	New PNGI image resource IDs
				*/
				virtual void setImageIDs(const ImageWithRolloverIDs& ids_);
			
				/**
					\brief		Gets the SVG resource IDs of the widget
					\author		GW
					\date		12/2018

					\returns	The IDs of the SVG image resources
				*/
				virtual SVGWithRolloverIDs svgIDs() const;
			
				/**
					\brief	Sets the SVG resource IDs of the widget
					\author	GW
					\date	12/2018

					\param	ids_	New SVG image resource IDs
				*/
				virtual void setSVGIDs(const SVGWithRolloverIDs& ids_);
				
				/**
					\brief		Gets whether the background of the button should change in appearance upon mouse-over
					\author		GW
					\date		03/2015
					
					\returns	true for showing a rollover effect, false otherwise
				*/
				virtual bool backgroundRollover() const;
				
				/**
					\brief	Sets whether the background of the button should change in appearance upon mouse-over
					\author	GW
					\date	03/2015
					
					\param	ro_		true to show a rollover effect, false otherwise
				*/
				virtual void setBackgroundRollover(const bool ro_);

				/**
					\brief		Gets the ideal width for a radio button, given its label and a known height
					\author		GW
					\date		09/2013

					\param		text_			Radio button label, as UTF-8
					\param		height_			Known height into which the radio button must fit
					\param		containerType_	Type of window for the widget (because font sizes differ)
					\returns	Ideal width for the given text to fit into the given height, at 1x resolution 
				*/
				static double idealWidth(const std::string& text_, const double height_, const WindowType containerType_);


			private:
				friend class RadioButtonGroup;
		};
		
		typedef std::unique_ptr<RadioButton> RadioButtonUP;
		typedef std::shared_ptr<RadioButton> RadioButtonSP;
		typedef std::weak_ptr<RadioButton> RadioButtonWP;
	}
}

#endif
// __HDI_CORE_RADIOBUTTON__
