/**
	\file
	\brief		Header file for describing rectangles
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_RECT__
#define __HDI_CORE_RECT__

#include "hdicorePoint.h"
#include "hdicoreSize.h"

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class Rect;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class Rect;
		}
	#endif

	namespace core
	{
		/**
			\brief	Contains Point and Size objects to describe a rectangle that exists at a specific point of given
					dimensions
		*/
		class Rect
		{
			public:
				/**
					\brief	Origin point for the rect (top-left, when size.width and size.height are positive)
				*/
				Point origin;

				/**
					\brief	Dimensions of the rect
				*/
				Size size;

				/**
					\brief	Default Rect constructor; calls the default constructors for the origin and size
					\author	GW
					\date	08/2013
				*/
				Rect();

				/**
					\brief	Constructs a Rect from an (x,y) point for the top-left and a width and height
					\author	GW
					\date	08/2013

					\param	x_			Position of the left of the rect along the x-axis
					\param	y_			Position of the top of the rect along the y-axis
					\param	width_		Width of the rect
					\param	height_		Height of the rect
				*/
				Rect(const double x_, const double y_, const double width_, const double height_);

				/**
					\brief	Rect constructor, taking Point and Size arguments
					\author	GW
					\date	08/2013

					\param	pt_		(x,y) point of the rect (top-left)
					\param	size_	Dimensions of the rect
				*/
				Rect(const Point& pt_, const Size& size_);

				/**
					\brief	Rect copy constructor
					\author	GW
					\date	08/2013

					\param	rect_	Existing Rect to copy
				*/
				Rect(const Rect& rect_);
			
				/**
					\brief	Rect destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~Rect();

				/**
					\brief		Sets a Rect object from another Rect object
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; Rect object to pull values from
					\returns	The target ArtboardRect (lefthand side of the = operator) with its values updated
				*/
				virtual Rect& operator=(const Rect& rhs_);
				
				/**
					\brief		Tests whether a given Rect object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Rect to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same rect, false otherwise
				*/
				virtual bool operator==(const Rect& rhs_) const;
				
				/**
					\brief		Tests whether a given Rect object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Rect to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different rects, false otherwise
				*/
				virtual bool operator!=(const Rect& rhs_) const;
				
				/**
					\brief		Allows two rects to have their origin and size objects added using the + operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the + operator
					\returns	A copy of the target ArtboardPoint, with all x and y values summed together
				*/
				virtual Rect operator+(const Rect& rhs_) const;
				
				/**
					\brief		Allows two rects to have their origin and size objects added and assigned using the += operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the += operator
					\returns	The lefthand side of the += operator, with all origin and size objects summed together
				*/
				virtual Rect& operator+=(const Rect& rhs_);

				/**
					\brief		Allows two rects to have their origin and size objects subtracted using the - operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the - operator
					\returns	A copy of the target ArtboardPoint, with rhs_ origin and size objects subtracted from
								this origin and size objects
				*/
				virtual Rect operator-(const Rect& rhs_) const;

				/**
					\brief		Allows two rects to have their origin and size objects subtracted and assigned using the -= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the -= operator
					\returns	The lefthand side of the -= operator, with rhs_ origin and size objects subtracted from
								this origin and size objects
				*/
				virtual Rect& operator-=(const Rect& rhs_);
				
				/**
					\brief		Offsets a rect along the x- and y-axes by given amounts
					\details	This is a convenience method to adjust both the x and y values in one call	
					\author		GW
					\date		08/2013

					\param		tx_		Amount to offset along the x-axis
					\param		ty_		Amount to offset along the y-axis

					\note		Negative values go toward 0.0 if the object to be transformed has a positive x and/or y
								value; positive values go away from 0.0
				*/
				virtual void offset(const double tx_, const double ty_);

				/**
					\brief		Resizes a rect to a given width and height
					\details	This is a convenience method to set both width and height in one call	
					\author		GW
					\date		08/2013

					\param		w_	New rect width
					\param		h_	New rect height
				*/
				virtual void resize(const double w_, const double h_);

				/**
					\brief		Insets a rect by given amounts on the x- and y-axes (or expands if negative)
					\author		GW
					\date		08/2013

					\param		x_	Amount to inset the rect on the left and right sides
					\param		y_	Amount to inset the rect on the top and bottom sides
					\returns	A copy of the target Rect, with its origin and size updated to reflect the provided inset

					\note		Positive values cause an inset, while negative values cause an "outset" (expansion)
				*/
				virtual Rect inset(const double x_, const double y_) const;

				/**
					\brief		Gets the top-left point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the top-left of the rect
					
					\note		The point returned may not be the geometric top-left of the rect in the case of a
								negative width and/or height
				*/
				virtual Point topLeft() const;

				/**
					\brief		Gets the top-center point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the top-center of the rect
					
					\note		The point returned may not be the geometric top-center of the rect in the case of a
								negative width and/or height
				*/
				virtual Point topCenter() const;

				/**
					\brief		Gets the top-right point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the top-right of the rect
					
					\note		The point returned may not be the geometric top-right of the rect in the case of a
								negative width and/or height
				*/
				virtual Point topRight() const;

				/**
					\brief		Gets the mid-left point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the mid-left of the rect
					
					\note		The point returned may not be the geometric mid-left of the rect in the case of a
								negative width and/or height
				*/
				virtual Point midLeft() const;

				/**
					\brief		Gets the mid-center point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the mid-center of the rect
				*/
				virtual Point midCenter() const;

				/**
					\brief		Gets the mid-right point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the mid-right of the rect
					
					\note		The point returned may not be the geometric mid-right of the rect in the case of a
								negative width and/or height
				*/
				virtual Point midRight() const;

				/**
					\brief		Gets the bottom-left point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the bottom-left of the rect
					
					\note		The point returned may not be the geometric bottom-left of the rect in the case of a
								negative width and/or height
				*/
				virtual Point bottomLeft() const;

				/**
					\brief		Gets the bottom-center point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the bottom-center of the rect
					
					\note		The point returned may not be the geometric bottom-center of the rect in the case of a
								negative width and/or height
				*/
				virtual Point bottomCenter() const;

				/**
					\brief		Gets the bottom-right point of the rect
					\author		GW
					\date		08/2013

					\returns	A Point object describing the bottom-right of the rect
					
					\note		The point returned may not be the geometric bottom-right of the rect in the case of a
								negative width and/or height
				*/
				virtual Point bottomRight() const;

				/**
					\brief		Gets a "pin point" from the rect bounds
					\author		GW
					\date		06/2018

					\param		pinPt_	Pin point which will be turned into an absolute point from the rect
					\returns	A Point object for the given pin point
				*/
				virtual Point pointFromPinPoint(const BoundsPinPointType pinPt_) const;

				/**
					\brief		Gets the position of the left bounding edge of the rectangle along the x-axis
					\author		GW
					\date		08/2013

					\returns	The position of the left side of the rectangle
				*/
				virtual double left() const;

				/**
					\brief		Gets the position of the top bounding edge of the rectangle along the y-axis
					\author		GW
					\date		08/2013

					\returns	The position of the top side of the rectangle
				*/
				virtual double top() const;

				/**
					\brief		Gets the position of the right bounding edge of the rectangle along the x-axis
					\author		GW
					\date		08/2013

					\returns	The position of the right side of the rectangle
				*/
				virtual double right() const;

				/**
					\brief		Gets the position of the bottom bounding edge of the rectangle along the y-axis
					\author		GW
					\date		08/2013

					\returns	The position of the bottom side of the rectangle
				*/
				virtual double bottom() const;

				/**
					\brief	Sets the position of the left bounding edge of the rectangle along the x-axis
					\author	GW
					\date	08/2013

					\param	left_	New left edge of the rect

					\note	The right edge will be unaffected (e.g. the width will automatically be adjusted such that
							the right bounding edge stays at the same position)
				*/
				virtual void setLeft(const double left_);

				/**
					\brief	Sets the position of the top bounding edge of the rectangle along the y-axis
					\author	GW
					\date	08/2013

					\param	top_	New top edge of the rect

					\note	The bottom edge will be unaffected (e.g. the height will automatically be adjusted such that
							the bottom bounding edge stays at the same position)
				*/
				virtual void setTop(const double top_);

				/**
					\brief	Sets the position of the right bounding edge of the rectangle along the x-axis
					\author	GW
					\date	08/2013

					\param	right_	New right edge of the rect

					\note	The left edge will be unaffected
				*/
				virtual void setRight(const double right_);

				/**
					\brief	Sets the position of the bottom bounding edge of the rectangle along the y-axis
					\author	GW
					\date	08/2013

					\param	bottom_		New bottom edge of the rect

					\note	The top edge will be unaffected
				*/
				virtual void setBottom(const double bottom_);

				/**
					\brief		Gets the area of the rect
					\author		GW
					\date		08/2013

					\returns	The area of the rect

					\note		The value returned will always be positive
				*/
				virtual double area() const;

				/**
					\brief		Gets the perimeter of the rect
					\author		GW
					\date		08/2013
					
					\returns	The perimeter of the rect
				*/
				virtual double perimeter() const;

				/**
					\brief		Gets whether a given point lies inside the target rect
					\author		GW
					\date		08/2013

					\param		pt_		Point to test
					\returns	true if the point is within the bounds of the rect, false otherwise
				*/
				virtual bool containsPoint(const Point& pt_) const;

				/**
					\brief		Checks whether the target rect overlaps a given rect
					\author		GW
					\date		08/2013

					\param		rect2_	Rect to compare against for overlapping boundaries
					\returns	true if the target rect and the provided rect overlap
				*/
				virtual bool overlaps(const Rect& rect2_) const;

				/**
					\brief		Enlarges the target rect, if necessary, to enclose a given point
					\author		GW
					\date		08/2013

					\param		pt_		Point the rect should enclose
					\returns	true if the rect was enlarged, false otherwise
				*/
				virtual bool enclosePoint(const Point& pt_);

				/**
					\brief		Unions two rects, forming a rect that has bounds sufficient to enclose both initial rects
					\author		GW
					\date		08/2013

					\param		rect2_	Second rect to consider in the union
					\returns	A new ArtboardRect large enough to enclose both the target rect and rect2_
				*/
				virtual Rect unionRects(const Rect& rect2_) const;

				/**
					\brief		Scales the width and height of a rect by given amounts on the x- and y- axes
					\author		GW
					\date		08/2013

					\param		xScaler_	Multiplier for scaling horizontally (e.g. 0.75 is 75% of the current size)
					\param		yScaler_	Multiplier for scaling vertically (e.g. 0.75 is 75% of the current size)
					\returns	A copy of the target ArtboardRect that is scaled as specified
				*/
				virtual Rect scale(const double xScaler_, const double yScaler_) const;

				/**
					\brief		Gets the intersection between two rects
					\author		GW
					\date		08/2013

					\param		rect2_		Second rect to test for an intersection with the target rect
					\param		iRect__		Return-by-reference for the intersection rect
					\returns	true if an intersection exists, false otherwise

					\note		iRect__ will be the rect that describes the intersecting area between rect2_ and the
								target rect.
				*/
				virtual bool intersection(const Rect& rect2_, Rect& iRect__) const;

				/**
					\brief	Gets the width and height of an ellipse that would enclose the rect
					\author	GW
					\date	06/2018

					\param	width__		Return-by-reference for the ellipse width
					\param	height__	Return-by-reference for the ellipse height

					\note	The ellipse will have the same proportions as the rect
				*/
				virtual void enclosingEllipse(double& width__, double& height__) const;

				/**
					\brief		Finds the intersection points on a rectangle that a line has with it
					\author		GW
					\date		08/2013

					\param		a_		Start point for the line
					\param		b_		End point for the line
					\param		iPt1__	Return-by-reference for the first intersection point found
					\param		iPt2__	Return-by-reference for the second intersection point found
					\returns	An enum describing which intersection points were found, if any

					\note		For reference, this function always checks clockwise from the left of the rectangle.
				*/
				virtual RectLineIntersectionType lineIntersectionPoints(
					const Point& a_,
					const Point& b_,
					Point& iPt1__,
					Point& iPt2__
				) const;

				/**
					\brief		Finds the intersection points on a rectangle, were it rotated, that a line has with it
					\author		GW
					\date		08/2013

					\param		a_		Start point for the line
					\param		b_		End point for the line
					\param		angle_	Amount the rectangle is rotated
					\param		iPt1__	Return-by-reference for the first intersection point found
					\param		iPt2__	Return-by-reference for the second intersection point found
					\returns	An enum describing which intersection points were found, if any

					\note		For reference, this function always checks clockwise from the left of the rectangle.
				*/
				virtual RectLineIntersectionType lineIntersectionPoints(
					const Point& a_,
					const Point& b_,
					const Angle& angle_,
					Point& iPt1__,
					Point& iPt2__
				) const;


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::Rect* __accessImpl(const Rect&);
					friend Rect __accessCtor(const aip::Rect&);

					/**
						\brief	Private implementation object
					*/
					aip::Rect* __impl;
					
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Rect(const aip::Rect&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend psp::Rect* __accessImpl(const Rect&);
					friend Rect __accessCtor(const psp::Rect&);

					/**
						\brief	Private implementation object
					*/
					psp::Rect* __impl;
					
					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Rect(const psp::Rect&);
				#endif
		};
		
		typedef std::unique_ptr<Rect> RectUP;
		typedef std::shared_ptr<Rect> RectSP;
		typedef std::weak_ptr<Rect> RectWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::Rect* __accessImpl(const Rect&);
			extern Rect __accessCtor(const aip::Rect&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern psp::Rect* __accessImpl(const Rect&);
			extern Rect __accessCtor(const psp::Rect&);
		#endif
	}
}

#endif
// __HDI_CORE_RECT__
