/**
	\file
	\brief		Header file for rectangle preference data storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_RECT_PREF__
#define __HDI_CORE_RECT_PREF__

#include "hdicoreRect.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of rects/bounds
		*/
		class RectPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty RectPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a RectPref object is empty, call isEmpty() on it
					\note	Empty RectPref objects do not relate to any actual preference data; they are designed to be
							"receivers" of some other RectPref object via the overloaded assignment operator. Empty
							RectPref objects are useless until such time (though it is safe to call any of their methods)
				*/
				RectPref();
			
				/**
					\brief	Constructs a new RectPref object from an existing RectPref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	r_	Existing RectPref object
				*/
				RectPref(const RectPref& r_);

				/**
					\brief	Constructs a RectPref object, with new data, to be added to a container at a later
							time
					\author	GW
					\date	09/2013

					\param	value_	Initial rect value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				RectPref(const Rect& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a RectPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~RectPref();

				/**
					\brief		Allows one RectPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target RectPref object, but with its value updated to match that of rhs_
				*/
				virtual RectPref& operator=(const RectPref& rhs_);
				
				/**
					\brief		Convenience method to clone an RectPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new RectPref object
					
					\note		If you subclass RectPref, you MUST overload this method yourself! If you don't and/or 
								your clone() method does not return an instance of your RectPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned RectPref object.
				*/
				virtual RectPref* clone() const;

				/**
					\brief		Gets the rect for this object
					\author		GW
					\date		09/2013

					\returns	The Rect object being described by this wrapper
				*/
				virtual Rect rect() const;

				/**
					\brief		Sets the rect for this object
					\author		GW
					\date		09/2013

					\param		rect_	New rect value
					\returns	true if the rect is successfully updated, or false otherwise
				*/
				virtual bool setRect(const Rect& rect_);
		};
		
		typedef std::unique_ptr<Rect> RectPrefUP;
		typedef std::shared_ptr<Rect> RectPrefSP;
		typedef std::weak_ptr<Rect> RectPrefWP;
	}
}

#endif
// __HDI_CORE_RECT_PREF__
