/**
	\file
	\brief		Header file for general runtime-related functions
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_RUNTIME__
#define __HDI_CORE_RUNTIME__

#include "hdicoreSize.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		
		/**
			\brief	Indicates a specific version of the hdi_core library
		*/
		enum LibVersion
		{
			LibVersion_Unknown	= 0,
			LibVersion_0_5_0	= 500,	// 0.5.0,	2014-01-27
			LibVersion_0_5_1	= 501,	// 0.5.1,	2014-03-14
			LibVersion_0_5_2	= 502,	// 0.5.2,	2014-03-18
			LibVersion_0_5_3	= 503,	// 0.5.3,	2014-05-19
			LibVersion_0_5_4	= 504,	// 0.5.4,	2014-05-30
			LibVersion_0_5_5	= 505,	// 0.5.5,	2014-07-17
			LibVersion_0_5_6	= 506,	// 0.5.6,	2014-10-16
			LibVersion_0_5_7	= 507,	// 0.5.7,	2014-12-14
			LibVersion_0_5_8	= 508,	// 0.5.8,	2015-03-23
			LibVersion_0_5_9	= 509,	// 0.5.9,	2015-06-18
			LibVersion_0_6_0	= 600,	// 0.6.0,	2015-10-28
			LibVersion_0_6_1	= 601,	// 0.6.1,	2016-02-16
			LibVersion_0_6_2	= 602,	// 0.6.2,	2016-07-23
			LibVersion_0_6_3	= 603,	// 0.6.3,	2016-11-14
			LibVersion_0_7_0	= 700,	// 0.7.0,	2017-04-11
			LibVersion_0_7_1	= 701,	// 0.7.1,	2017-10-30
			LibVersion_0_7_2	= 702,	// 0.7.2,	2018-10-17
			LibVersion_0_7_3	= 703,	// 0.7.3,	2018-11-19
			LibVersion_0_7_4	= 704,	// 0.7.4,	2019-11-05
			LibVersion_0_7_5	= 705,	// 0.7.5,	2020-09-02
			LibVersion_0_7_6	= 706,	// 0.7.6,	2020-10-22
			LibVersion_0_7_7	= 707,	// 0.7.7,	2021-01-18
			LibVersion_0_7_8	= 708,	// 0.7.8,	2021-03-07
			LibVersion_0_8_0	= 800,	// 0.8.0,	2022-01-14
			LibVersion_0_8_1	= 801,	// 0.8.1,	2022-11-16
			LibVersion_0_8_2	= 802,	// 0.8.2,	2023-11-07
			LibVersion_0_8_3	= 803,	// 0.8.3,	2024-12-30
			LibVersion_0_8_4	= 804	// 0.8.4,	2025-11-05
		};

		/**
			\brief		Gets the version of the hdi_core library against which the plugin was built
			\author		GW
			\date		12/2013
			
			\returns	Version enum value for the hdi_core lib
		*/
		LibVersion version();
		
		/**
			\brief		Gets the version of the hdi_core library as a string
			\author		GW
			\date		12/2013
			
			\returns	A stringified variation of the version() function
		*/
		std::string versionString();
		
		/**
			\brief	Describes the OS version that the plug-in is running within
		*/
		enum SystemVersion
		{
			UnknownSystemVersion	= 0,
			
			#if defined(MAC_ENV)
				MacOSX_10_11			= 101100,
				MacOSX_10_11_OrEarlier	= MacOSX_10_11,
				MacOSX_10_12			= 101200,
				MacOSX_10_13			= 101300,
				MacOSX_10_14			= 101400,
				MacOSX_10_15			= 101500,
				MacOSX_11_00			= 110000,
				MacOSX_11_00_OrLater	= MacOSX_11_00
			#elif defined(WIN_ENV)
				Windows_Vista			= 60000,
				Windows_Vista_OrEarlier	= Windows_Vista,
				Windows_Vista_SP1		= 60001,
				Windows_Vista_SP2		= 60002,
				Windows_7				= 60100,
				Windows_7_SP1			= 60101,
				Windows_8				= 60200,
				Windows_8_1				= 60300,
				Windows_10				= 100000,
				Windows_10_OrLater		= Windows_10
			#endif
		};
		
		/**
			\brief		Gets the version of the OS in which the plug-in is being run
			\author		GW
			\date		05/2014
			
			\returns	Platform-specific enum value for the OS version
			
			\note		On macOS you can get more specific version info by calling the systemVersionNumbers() function.
		*/
		SystemVersion systemVersion();
		
		#if defined(MAC_ENV)
			/**
				\brief		Gets the individual version numbers on macOS in which the plug-in is being run
				\author		GW
				\date		12/2024
				
				\param		major__		Return-by-reference for the major OS version number (e.g. "11" in "11.2.3")
				\param		minor__		Return-by-reference for the minor OS version number (e.g. "2" in "11.2.3")
				\param		revision__	Return-by-reference for the revision OS version number (e.g. "3" in "11.2.3")
			*/
			void systemVersionNumbers(int16_t& major__, int16_t& minor__, int16_t& revision__);
		#endif
		
		/**
			\brief		Gets the UUID for the machine on which the plug-in is being run
			\author		GW
			\date		09/2013
			
			\returns	A UUID string for the user's machine, or "" if it cannot be acquired
			
			\note		This function is not guaranteed to return the same value for all versions of the app. However,
						within the same version of the app and on the same platform, this function will always return
						the same value for a given machine (barring the user replacing some component which the OS uses
						for the UUID calculation, but that is out of our control).
		*/
		std::string machineUUID();

		/**
			\brief		Gets the dimensions of the user's primary screen
			\author		GW
			\date		08/2013

			\returns	The dimensions of the user's primary display, in pixels
		*/
		Size screenDimensions();

		/**
			\brief		Gets whether the OS has a browser available
			\author		GW
			\date		08/2013

			\returns	true if a browser is available, false otherwise
		*/
		bool browserAvailable();
		
		/**
			\brief	Used to contain data when writing/reading to/from the clipboard
		*/
		class ClipboardData
		{
			public:
				/**
					\brief	Describes the type of data contained in a ClipboardData object
				*/
				enum DataType
				{
					DataTypeUnknown	= 0,
					DataTypeString	= 10,
					DataTypeImage	= 20	// Reserved for future use
				};
				
				/**
					\brief	Default ClipboardData constructor, setting the object to have default values
					\author	GW
					\date	02/2015
				*/
				ClipboardData();
				
				/**
					\brief	ClipboardData copy constructor to set the new object to have the same values as an existing
							object
					\author	GW
					\date	02/2015
					
					\param	data_	Existing ClipboardData object
				*/
				ClipboardData(const ClipboardData& data_);
				
				/**
					\brief	Constructs a new ClipboardData object from a string value
					\author	GW
					\date	02/2015
					
					\param	str_	String value that the ClipboardData object should hold, as UTF-8
				*/
				ClipboardData(const std::string& str_);
				
				/**
					\brief	Destructs an existing ClipboardData object
					\author	GW
					\date	02/2015
				*/
				virtual ~ClipboardData();
				
				/**
					\brief		Assigns one ClipboardData object to another
					\author		GW
					\date		02/2015

					\param		rhs_	Existing ClipboardData object to copy values from
					\returns	The target ClipboardData object, but with its values updated to match that of the rhs_
								argument
				*/
				ClipboardData& operator=(const ClipboardData& rhs_);
				
				/**
					\brief		Tests whether a given ClipboardData object is the same as another
					\author		GW
					\date		02/2015

					\param		rhs_	ClipboardData to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same clipboard data, false otherwise
				*/
				bool operator==(const ClipboardData& rhs_) const;
				
				/**
					\brief		Tests whether a given ClipboardData object is not the same as another
					\author		GW
					\date		02/2015

					\param		rhs_	ClipboardData to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different clipboard data, false otherwise
				*/
				bool operator!=(const ClipboardData& rhs_) const;
				
				/**
					\brief		Gets the type of data held in the target
					\author		GW
					\date		02/2015
					
					\returns	The type of the contained data, or DataTypeUnknown if the default ctor was used
				*/
				DataType dataType() const;
				
				/**
					\brief		Gets the string data contained in the target object, if the type is DataTypeString
					\author		GW
					\date		02/2015
					
					\returns	The string value held in the target (as UTF-8), or "" if the type is not
								DataTypeString
				*/
				std::string stringData() const;
				
				/**
					\brief	Sets the string data contained in the target object, converting the type to DataTypeString
					\author	GW
					\date	02/2015
					
					\param	str_	New string value to hold in the target, as UTF-8
				*/
				void setStringData(const std::string& str_);


			private:
				/**
					\brief	Private implementation data
				*/
				void* __data;
		};
		
		/**
			\brief		Stores the current clipboard data for later restoration
			\author		GW
			\date		01/2018
			
			\returns	true for success, false otherwise
			
			\warning	The restoreClipboardData() function must be called after this function, within the same message
						context from the parent app!
			
			\note		This function is useful when needing to temporarily override data in the clipboard; call this
						function, change whatever you need in the clipboard, then call restoreClipboardData() thereafter.
		*/
		bool storeClipboardData();
		
		/**
			\brief		Restores the clipboard data (from the stored data from an earlier call to the storeClipboardData()
						function)
			\author		GW
			\date		01/2018
			
			\returns	true for success, false otherwise
			
			\warning	This function must be called after the storeClipboardData() function, within the same message
						context from the parent app!
		*/
		bool restoreClipboardData();
		
		/**
			\brief		Gets the data currently on the clipboard
			\author		GW
			\date		02/2015
			
			\param		data__		Return-by-reference for the clipboard data
			\returns	true if the data could be acquired, false otherwise
		*/
		bool clipboardData(ClipboardData& data__);
		
		/**
			\brief		Sets the data currently on the clipboard
			\author		GW
			\date		02/2015
			
			\param		data_	New clipboard data
			\returns	true if the data could be set, false otherwise
		*/
		bool setClipboardData(const ClipboardData& data_);
		
		#if defined(HDI_CORE_PSP_MODE)
			/**
				\brief		Pastes the contents of the clipboard to the document, using Photoshop suite calls
				\author		GW
				\date		01/2018
				
				\returns	true for success, false otherwise
				
				\note		This function bails early for non-action plugins.
			*/
			bool pasteClipboard();
		#endif
		// HDI_CORE_PSP_MODE
		
		/**
			\brief		Gets which mouse buttons are currently pressed (can be called anytime)
			\author		GW
			\date		07/2014
			
			\returns	A bitwise OR of MouseButton enum values for the currently pressed buttons
		*/
		MouseButton mouseButtonsPressed();
		
		/**
			\brief		Gets which modifier keys are currently pressed (can be called anytime)
			\author		GW
			\date		04/2014
			
			\returns	A bitwise OR of ModifierKey enum values for the currently pressed keys
		*/
		ModifierKey modifierKeysPressed();
		
		/**
			\brief	Describes the language for which the current (keyboard) input source is designed
		*/
		enum KeyboardLanguage
		{
			UnknownKeyLang				= 0,
			AfghanPashtoKeyLang			= 10,
			ArabicKeyLang				= 20,
			ArmenianKeyLang				= 30,
			AzeriKeyLang				= 40,
			BanglaKeyLang				= 50,
			BelgianKeyLang				= 60,
			BulgarianKeyLang			= 70,
			BelarusianKeyLang			= 80,
			CherokeeKeyLang				= 90,
			ChineseSimplifiedKeyLang	= 100,
			ChineseTraditionalKeyLang	= 110,
			CroatianKeyLang				= 120,
			CzechKeyLang				= 130,
			DanishKeyLang				= 140,
			DevanagariKeyLang			= 150,
			DutchKeyLang				= 160,
			EnglishAmericanKeyLang		= 170,
			EnglishAustralianKeyLang	= 180,
			EnglishBritishKeyLang		= 190,
			EnglishCanadianKeyLang		= 200,
			EnglishOtherKeyLang			= 210,
			EstonianKeyLang				= 220,
			FaroeseKeyLang				= 230,
			FinnishKeyLang				= 240,
			FrenchKeyLang				= 250,
			FrenchCanadianKeyLang		= 260,
			FrenchSwissKeyLang			= 270,
			GeorgianKeyLang				= 280,
			GermanKeyLang				= 290,
			GermanAustrianKeyLang		= 300,
			GermanSwissKeyLang			= 310,
			GreekKeyLang				= 320,
			GujaratiKeyLang				= 330,
			GurmukhiKeyLang				= 340,
			HawaiianKeyLang				= 350,
			HebrewKeyLang				= 360,
			HungarianKeyLang			= 370,
			InuktitutKeyLang			= 380,
			IrishKeyLang				= 390,
			ItalianKeyLang				= 400,
			JapaneseKeyLang				= 410,
			KannadaKeyLang				= 420,
			KazakhKeyLang				= 430,
			KhmerKeyLang				= 440,
			KoreanKeyLang				= 450,
			KurdishSoraniKeyLang		= 460,
			LatvianKeyLang				= 470,
			LithuanianKeyLang			= 480,
			MacedonianKeyLang			= 490,
			MalayalamKeyLang			= 500,
			MalteseKeyLang				= 510,
			MaoriKeyLang				= 520,
			NepaliKeyLang				= 530,
			NorwegianKeyLang			= 540,
			OriyaKeyLang				= 550,
			PersianKeyLang				= 560,
			PersianDariKeyLang			= 570,
			PolishKeyLang				= 580,
			PortugueseKeyLang			= 590,
			PortugueseBrazilianKeyLang	= 600,
			RomanianKeyLang				= 610,
			RussianKeyLang				= 620,
			SamiKeyLang					= 630,
			SerbianKeyLang				= 640,
			SinhalaKeyLang				= 650,
			SlovakKeyLang				= 660,
			SlovenianKeyLang			= 670,
			SpanishKeyLang				= 680,
			SwedishKeyLang				= 690,
			TamilKeyLang				= 700,
			TeluguKeyLang				= 710,
			ThaiKeyLang					= 720,
			TibetanKeyLang				= 730,
			TurkishKeyLang				= 740,
			UkrainianKeyLang			= 750,
			UrduKeyLang					= 760,
			UyghurKeyLang				= 770,
			VietnameseKeyLang			= 780,
			WelshKeyLang				= 790
		};

		/**
			\brief	Describes the layout of the current (keyboard) input source

			\note	Unfortunately, Mac and Windows describe layouts in completely different manners, so there is no
					good way to "marry" the two of them in cross-platform values
		*/
		enum KeyboardLayout
		{
			KeyLayout_Unknown = 0,

			#if defined(MAC_ENV)
				KeyLayout_Korean_2Set = 100,
				KeyLayout_Korean_3Set = 200,
				KeyLayout_Korean_390Sebulshik = 300,
				KeyLayout_Korean_GongjinCheongRomaja = 400,

				KeyLayout_Japanese_Ainu = 500,
				KeyLayout_Japanese_Hiragana = 600,
				KeyLayout_Japanese_Katakana = 700,
				KeyLayout_Japanese_KatakanaHalfWidth = 800,
				KeyLayout_Japanese_Romaji = 900,
				KeyLayout_Japanese_RomajiFullWidth = 1000,

				KeyLayout_Chinese_Simplified_Pinyin = 1100,
				KeyLayout_Chinese_Simplified_WubiHua = 1200,
				KeyLayout_Chinese_Simplified_WubiXing = 1300,

				KeyLayout_Tamil_Anjal = 1400,
				KeyLayout_Tamil_Tamil99 = 1500,

				KeyLayout_Chinese_Traditional_Cangjie = 1600,
				KeyLayout_Chinese_Traditional_Sucheng = 1700,
				KeyLayout_Chinese_Traditional_Pinyin = 1800,
				KeyLayout_Chinese_Traditional_Zhuyin = 1900,
				KeyLayout_Chinese_Traditional_ZhuyinEten = 2000,

				KeyLayout_Vietnamese_Telex = 2100,
				KeyLayout_Vietnamese_TelexSimple = 2200,
				KeyLayout_Vietnamese_VIQR = 2300,
				KeyLayout_Vietnamese_VNI = 2400,

				KeyLayout_Afghan_Dari = 2500,
				KeyLayout_Afghan_Pashto = 2600,
				KeyLayout_Afghan_Uzbek = 2700,

				KeyLayout_Arabic = 2800,
				KeyLayout_Arabic_QWERTY = 2900,
				KeyLayout_Arabic_PC = 3000,

				KeyLayout_Armenian_HM_QWERTY = 3100,
				KeyLayout_Armenian_WesternQWERTY = 3200,

				KeyLayout_Australian = 3300,

				KeyLayout_Austrian = 3400,
				
				KeyLayout_Azeri = 3500,
				
				KeyLayout_Bangla = 3600,
				KeyLayout_Bangla_QWERTY = 3700,

				KeyLayout_Belgian = 3800,

				KeyLayout_Brazilian = 3900,

				KeyLayout_British = 4000,
				KeyLayout_British_PC = 4100,

				KeyLayout_Bulgarian = 4200,
				KeyLayout_Bulgarian_Phonetic = 4300,

				KeyLayout_Belarusian = 4400,

				KeyLayout_CanadianEnglish = 4500,
				KeyLayout_CanadianFrench_CSA = 4600,

				KeyLayout_Cherokee_Nation = 4700,
				KeyLayout_Cherokee_QWERTY = 4800,

				KeyLayout_Colemak = 4900,
				
				KeyLayout_Croatian = 5000,
				KeyLayout_Croatian_PC = 5100,

				KeyLayout_Czech = 5200,
				KeyLayout_Czech_QWERTY = 5300,

				KeyLayout_Danish = 5400,

				KeyLayout_Devanagari = 5500,
				KeyLayout_Devanagari_QWERTY = 5600,

				KeyLayout_Dutch = 5700,

				KeyLayout_Dvorak = 5800,
				KeyLayout_Dvorak_Left = 5900,
				KeyLayout_Dvorak_QWERTY = 6000,
				KeyLayout_Dvorak_Right = 6100,

				KeyLayout_Estonian = 6200,

				KeyLayout_Faroese = 6300,

				KeyLayout_Finnish = 6400,
				KeyLayout_Finnish_Extended = 6500,
				KeyLayout_Finnish_SamiPC = 6600,

				KeyLayout_French = 6700,
				KeyLayout_French_Numerical = 6800,

				KeyLayout_Georgian_QWERTY = 6900,

				KeyLayout_German = 7000,

				KeyLayout_Greek = 7100,
				KeyLayout_Greek_Polytonic = 7200,

				KeyLayout_Gujarati = 7300,
				KeyLayout_Gujarati_QWERTY = 7400,

				KeyLayout_Gurmukhi = 7500,
				KeyLayout_Gurmukhi_QWERTY = 7600,

				KeyLayout_Hawaiian = 7700,
				
				KeyLayout_Hebrew = 7800,
				KeyLayout_Hebrew_PC = 7900,
				KeyLayout_Hebrew_QWERTY = 8000,

				KeyLayout_Hungarian = 8100,

				KeyLayout_Inuktitut_Nunavik = 8200,
				KeyLayout_Inuktitut_Nunavut = 8300,
				KeyLayout_Inuktitut_Nutaaq = 8400,
				KeyLayout_Inuktitut_QWERTY = 8500,

				KeyLayout_Irish = 8600,
				KeyLayout_Irish_Extended = 8700,

				KeyLayout_Italian_Typewriter = 8800,
				KeyLayout_Italian = 8900,

				KeyLayout_Jawi_QWERTY = 9000,

				KeyLayout_Kannada = 9100,
				KeyLayout_Kannada_QWERTY = 9200,

				KeyLayout_Kazakh = 9300,

				KeyLayout_Khmer = 9400,

				KeyLayout_Kurdish_Sorani = 9500,

				KeyLayout_Latvian = 9600,

				KeyLayout_Lithuanian = 9700,

				KeyLayout_Macedonian = 9800,

				KeyLayout_Malayalam = 9900,
				KeyLayout_Malayalam_QWERTY = 10000,

				KeyLayout_Maltese = 10100,

				KeyLayout_Maori = 10200,

				KeyLayout_Myanmar_QWERTY = 10300,

				KeyLayout_Nepali = 10400,

				KeyLayout_NorthernSami = 10500,

				KeyLayout_Norwegian = 10600,
				KeyLayout_Norwegian_Extended = 10700,
				KeyLayout_Norwegian_SamiPC = 10800,

				KeyLayout_Oriya = 10900,
				KeyLayout_Oriya_QWERTY = 11000,

				KeyLayout_Persian = 11100,
				KeyLayout_Persian_ISIRI = 11200,
				KeyLayout_Persian_QWERTY = 11300,

				KeyLayout_Polish = 11400,
				KeyLayout_Polish_Pro = 11500,

				KeyLayout_Portuguese = 11600,

				KeyLayout_Romanian = 11700,
				KeyLayout_Romanian_Standard = 11800,

				KeyLayout_Russian = 11900,
				KeyLayout_Russian_Phonetic = 12000,
				KeyLayout_Russian_PC = 12100,

				KeyLayout_Sami_PC = 12200,

				KeyLayout_Serbian = 12300,
				KeyLayout_Serbian_Latin = 12400,

				KeyLayout_Sinhala = 12500,
				KeyLayout_Sinhala_QWERTY = 12600,

				KeyLayout_Slovak = 12700,
				KeyLayout_Slovak_QWERTY = 12800,

				KeyLayout_Slovenian = 12900,

				KeyLayout_Spanish = 13000,
				KeyLayout_Spanish_ISO = 13100,

				KeyLayout_Swedish = 13200,
				KeyLayout_Swedish_Pro = 13300,
				KeyLayout_Swedish_SamiPC = 13400,

				KeyLayout_SwissFrench = 13500,
				KeyLayout_SwissGerman = 13600,

				KeyLayout_Telugu = 13700,
				KeyLayout_Telugu_QWERTY = 13800,

				KeyLayout_Thai = 13900,
				KeyLayout_Thai_PattaChote = 14000,

				KeyLayout_Tibetan_Otani = 14100,
				KeyLayout_Tibetan_QWERTY = 14200,
				KeyLayout_Tibetan_Wylie = 14300,

				KeyLayout_Turkish = 14400,
				KeyLayout_Turkish_QWERTY = 14500,
				KeyLayout_Turkish_QWERTY_PC = 14600,

				KeyLayout_Ukrainian = 14700,

				KeyLayout_Urdu = 14800,

				KeyLayout_US = 14900,
				KeyLayout_US_Extended = 15000,
				KeyLayout_US_InternationalPC = 15100,

				KeyLayout_Uyghur = 15200,

				KeyLayout_Vietnamese = 15300,

				KeyLayout_Welsh = 15400
			#elif defined(WIN_ENV)
				KeyLayout_Afrikaans = 150,

				KeyLayout_Albanian = 250,

				KeyLayout_Alsatian = 350,

				KeyLayout_Amharic = 450,

				KeyLayout_Arabic_Algeria = 550,
				KeyLayout_Arabic_Bahrain = 650,
				KeyLayout_Arabic_Egypt = 750,
				KeyLayout_Arabic_Iraq = 850,
				KeyLayout_Arabic_Jordan = 950,
				KeyLayout_Arabic_Kuwait = 1050,
				KeyLayout_Arabic_Lebanon = 1150,
				KeyLayout_Arabic_Libya = 1250,
				KeyLayout_Arabic_Morocco = 1350,
				KeyLayout_Arabic_Oman = 1450,
				KeyLayout_Arabic_Qatar = 1550,
				KeyLayout_Arabic_SaudiArabia = 1650,
				KeyLayout_Arabic_Syria = 1750,
				KeyLayout_Arabic_Tunisia = 1850,
				KeyLayout_Arabic_Uae = 1950,
				KeyLayout_Arabic_Yemen = 2050,

				KeyLayout_Armenian = 2150,

				KeyLayout_Assamese = 2250,

				KeyLayout_Azeri_Cyrillic = 2350,
				KeyLayout_Azeri_Latin = 2450,

				KeyLayout_Bangla = 2550,
				KeyLayout_Bangla_India = 2650,

				KeyLayout_Bashkir = 2750,

				KeyLayout_Basque = 2850,

				KeyLayout_Belarusian = 2950,

				KeyLayout_Bosnian = 3050,
				KeyLayout_Bosnian_Cyrillic = 3150,
				KeyLayout_Bosnian_Latin = 3250,

				KeyLayout_Breton = 3350,

				KeyLayout_Bulgarian = 3450,

				KeyLayout_Catalan = 3550,

				KeyLayout_Chinese_Hongkong = 3850,
				KeyLayout_Chinese_Macau = 3950,
				KeyLayout_Chinese_Singapore = 4050,
				KeyLayout_Chinese_Simplified = 4150,
				KeyLayout_Chinese_Traditional = 4250,

				KeyLayout_Corsican = 4350,

				KeyLayout_Croatian = 4450,
				KeyLayout_Croatian_BosniaHerzegovina = 4650,

				KeyLayout_Czech = 4750,

				KeyLayout_Danish = 4850,

				KeyLayout_Dari = 4950,

				KeyLayout_Divehi = 5050,

				KeyLayout_Dutch = 5150,
				KeyLayout_Dutch_Belgian = 5250,

				KeyLayout_English_Australia = 5350,
				KeyLayout_English_Belize = 5450,
				KeyLayout_English_Canada = 5550,
				KeyLayout_English_Caribbean = 5650,
				KeyLayout_English_India = 5750,
				KeyLayout_English_Ireland = 5950,
				KeyLayout_English_Jamaica = 6050,
				KeyLayout_English_Malaysia = 6150,
				KeyLayout_English_NewZealand = 6250,
				KeyLayout_English_Philippines = 6350,
				KeyLayout_English_Singapore = 6450,
				KeyLayout_English_SouthAfrica = 6550,
				KeyLayout_English_Trinidad = 6650,
				KeyLayout_English_UK = 6750,
				KeyLayout_English_US = 6850,
				KeyLayout_English_Zimbabwe = 6950,

				KeyLayout_Estonian = 7050,

				KeyLayout_Faeroese = 7150,

				KeyLayout_Filipino = 7250,

				KeyLayout_Finnish = 7350,

				KeyLayout_French = 7450,
				KeyLayout_French_Belgian = 7550,
				KeyLayout_French_Canadian = 7650,
				KeyLayout_French_Luxembourg = 7750,
				KeyLayout_French_Monaco = 7850,
				KeyLayout_French_Swiss = 7950,

				KeyLayout_Frisian = 8050,

				KeyLayout_Galician = 8150,

				KeyLayout_Georgian = 8250,

				KeyLayout_German = 8350,
				KeyLayout_German_Austrian = 8450,
				KeyLayout_German_Liechtenstein = 8550,
				KeyLayout_German_Luxembourg = 8650,
				KeyLayout_German_Swiss = 8750,

				KeyLayout_Greek = 8850,

				KeyLayout_Greenlandic = 8950,

				KeyLayout_Gujarati = 9050,

				KeyLayout_Hausa = 9150,

				KeyLayout_Hebrew = 9350,

				KeyLayout_Hindi = 9450,

				KeyLayout_Hungarian = 9550,

				KeyLayout_Icelandic = 9650,

				KeyLayout_Igbo = 9750,

				KeyLayout_Indonesian = 9850,

				KeyLayout_Inuktitut_Canada = 9950,
				KeyLayout_Inuktitut_CanadaLatin = 10050,

				KeyLayout_Irish = 10150,

				KeyLayout_Italian = 10250,
				KeyLayout_Italian_Swiss = 10350,

				KeyLayout_Japanese = 10450,

				KeyLayout_Kannada = 10550,

				KeyLayout_Kazakh = 10650,

				KeyLayout_Khmer = 10750,

				KeyLayout_Kiche = 10850,

				KeyLayout_Kinyarwanda = 10950,

				KeyLayout_Konkani = 11050,

				KeyLayout_Korean = 11150,

				KeyLayout_Kyrgyz = 11250,

				KeyLayout_Lao = 11350,

				KeyLayout_Latvian = 11450,

				KeyLayout_Lithuanian = 11550,

				KeyLayout_Sorbian = 11650,

				KeyLayout_Luxembourgish = 11750,

				KeyLayout_Macedonian = 11850,

				KeyLayout_Malay = 11950,
				KeyLayout_Malay_BruneiDarussalam = 12050,

				KeyLayout_Malayalam = 12150,

				KeyLayout_Maltese = 12250,

				KeyLayout_Maori = 12350,

				KeyLayout_Mapudungun = 12450,

				KeyLayout_Marathi = 12550,

				KeyLayout_Mohawk = 12650,

				KeyLayout_Mongolian_Cyrillic = 12750,
				KeyLayout_Mongolian_PRC = 12850,

				KeyLayout_Nepali = 12950,
				KeyLayout_Nepali_India = 13050,

				KeyLayout_Norwegian_Bokmal = 13150,
				KeyLayout_Norwegian_Nynorsk = 13250,

				KeyLayout_Occitan = 13350,

				KeyLayout_Oriya = 13450,

				KeyLayout_Pashto = 13550,

				KeyLayout_Persian = 13650,

				KeyLayout_Polish = 13750,

				KeyLayout_Portuguese = 13850,
				KeyLayout_Portuguese_Brazilian = 13950,

				KeyLayout_Punjabi = 14150,

				KeyLayout_Quechua_Bolivia = 14350,
				KeyLayout_Quechua_Ecuador = 14450,
				KeyLayout_Quechua_Peru = 14550,

				KeyLayout_Romanian = 14650,

				KeyLayout_Romansh = 14750,

				KeyLayout_Russian = 14850,

				KeyLayout_Sami_Inari_Finland = 15050,
				KeyLayout_Sami_Lule_Norway = 15150,
				KeyLayout_Sami_Lule_Sweden = 15250,
				KeyLayout_Sami_Northern_Finland = 15350,
				KeyLayout_Sami_Northern_Norway = 15450,
				KeyLayout_Sami_Northern_Sweden = 15550,
				KeyLayout_Sami_Skolt_Finland = 15650,
				KeyLayout_Sami_Southern_Norway = 15750,
				KeyLayout_Sami_Southern_Sweden = 15850,

				KeyLayout_Sanskrit = 15950,

				KeyLayout_Serbian = 16050,
				KeyLayout_Serbian_Cyrillic = 16150,
				KeyLayout_Serbian_Latin = 16250,
				KeyLayout_Serbian_BosniaHerzegovina_Cyrillic = 16350,
				KeyLayout_Serbian_BosniaHerzegovina_Latin = 16450,

				KeyLayout_Sindhi = 16650,

				KeyLayout_Sinhalese = 16750,

				KeyLayout_Slovak = 16850,

				KeyLayout_Slovenian = 16950,

				KeyLayout_Sotho = 17050,

				KeyLayout_Spanish = 17150,
				KeyLayout_Spanish_Argentina = 17250,
				KeyLayout_Spanish_Bolivia = 17350,
				KeyLayout_Spanish_Chile = 17450,
				KeyLayout_Spanish_Colombia = 17550,
				KeyLayout_Spanish_CostaRica = 17650,
				KeyLayout_Spanish_DominicanRepublic = 17750,
				KeyLayout_Spanish_Ecuador = 17850,
				KeyLayout_Spanish_ElSalvador = 17950,
				KeyLayout_Spanish_Guatemala = 18050,
				KeyLayout_Spanish_Honduras = 18150,
				KeyLayout_Spanish_Mexican = 18250,
				KeyLayout_Spanish_Nicaragua = 18350,
				KeyLayout_Spanish_Panama = 18450,
				KeyLayout_Spanish_Paraguay = 18550,
				KeyLayout_Spanish_Peru = 18650,
				KeyLayout_Spanish_PuertoRico = 18750,
				KeyLayout_Spanish_Modern = 18850,
				KeyLayout_Spanish_US = 18950,
				KeyLayout_Spanish_Uruguay = 19050,
				KeyLayout_Spanish_Venezuela = 19150,

				KeyLayout_Swahili = 19250,

				KeyLayout_Swedish = 19350,
				KeyLayout_Swedish_Finland = 19450,

				KeyLayout_Syriac = 19650,

				KeyLayout_Tajik = 19750,

				KeyLayout_Tamazight = 19850,

				KeyLayout_Tamil_India = 19950,

				KeyLayout_Tatar = 20150,

				KeyLayout_Telugu = 20250,

				KeyLayout_Thai = 20350,

				KeyLayout_Tibetan = 20450,

				KeyLayout_Tigrinya_Eritrea = 20550,

				KeyLayout_Tswana_SouthAfrica = 20850,

				KeyLayout_Turkish = 20950,

				KeyLayout_Turkmen = 21050,

				KeyLayout_Uighur = 21150,

				KeyLayout_Ukrainian = 21250,

				KeyLayout_Urdu_India = 21450,
				KeyLayout_Urdu_Pakistan = 21550,

				KeyLayout_Uzbek_Cyrillic = 21650,
				KeyLayout_Uzbek_Latin = 21750,

				KeyLayout_Vietnamese = 21950,

				KeyLayout_Welsh = 22050,

				KeyLayout_Wolof = 22150,

				KeyLayout_Xhosa = 22250,

				KeyLayout_Yakut = 22350,

				KeyLayout_Yi = 22450,

				KeyLayout_Yoruba = 22550,

				KeyLayout_Zulu = 22650
			#endif
		};

		/**
			\brief		Gets the best-fit keyboard language for the current input source (keyboard layout)
			\author		GW
			\date		04/2014

			\returns	Best-fit keyboard language for the current layout
			
			\note		Not all values from the Windows portion of the KeyboardLayout enum are processed internally
						by this function, due to there being no equivalent Mac values. The same goes for the Mac
						portion being translated to Windows values. For these cases, call the keyboardLayout()
						function yourself and process the output.
		*/
		KeyboardLanguage keyboardLanguage();

		/**
			\brief		Gets the current keyboard layout
			\author		GW
			\date		04/2014

			\returns	The current OS-specific keyboard layout enum value
		*/
		KeyboardLayout keyboardLayout();
	}
}

#endif
// __HDI_CORE_RUNTIME__
