/**
	\file
	\brief		Header file for plugin user interface stepper widgets
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_STEPPER__
#define __HDI_CORE_STEPPER__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;

		/**
			\brief	Stepper widget, which has a up/down button interface to allow for increasing/decreasing values of
					other nearby widgets
		*/
		class Stepper : public Widget
		{
			public:
				/**
					\brief	Constructs an empty Stepper object
					\author	GW
					\date	02/2017
					
					\note	To test if a Stepper object is empty, call isEmpty() on it
					\note	Empty Stepper objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other Stepper object via the overloaded assignment operator. Empty
							Stepper objects are useless until such time (though it is safe to call any of their methods).
				*/
				Stepper();

				/**
					\brief	Constructs a new Stepper object from an existing Stepper object (copy constructor)
					\author	GW
					\date	02/2017

					\param	stepper_	Existing Stepper object
				*/
				Stepper(const Stepper& stepper_);

				/**
					\brief	Constructs a stepper
					\author	GW
					\date	02/2017

					\param	loc_	Position of the stepper, in 1x resolution coordinates
				*/
				Stepper(const Point& loc_);
				
				/**
					\brief	Stepper destructor
					\author	GW
					\date	02/2017
				*/
				virtual ~Stepper();

				/**
					\brief		Allows one Stepper object to be assigned from another
					\author		GW
					\date		02/2017

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Stepper object, but with its value updated to match that of rhs_
				*/
				virtual Stepper& operator=(const Stepper& rhs_);
				
				/**
					\brief		Convenience method to clone a Stepper object on the heap
					\author		GW
					\date		02/2017

					\returns	A pointer to the new Stepper object
					
					\note		If you subclass Stepper, you MUST overload this method yourself! If you don't and/or your
								clone() method does not return an instance of your Stepper subclass, you will experience
								"object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned Stepper object.
				*/
				virtual Stepper* clone() const;
				
				/**
					\brief		Convenience method to duplicate a Stepper object, creating a new and identical UI element
								to the target (but not belonging to the same containing window)
					\author		GW
					\date		02/2017
					
					\returns	A pointer to the new Stepper object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Stepper object.
				*/
				virtual Stepper* duplicate() const;
				
				/**
					\brief		Gets the up-click callback for the stepper
					\author		GW
					\date		02/2017

					\returns	A pointer to the currently registered up-click callback
				*/
				virtual Callback* const upClickCallback() const;
				
				/**
					\brief	Sets the up-click callback
					\author	GW
					\date	02/2017

					\param	callback_	New callback for when the stepper's up-button is clicked
				*/
				virtual void setUpClickCallback(const Callback& callback_);
				
				/**
					\brief		Gets the down-click callback for the stepper
					\author		GW
					\date		02/2017

					\returns	A pointer to the currently registered down-click callback
				*/
				virtual Callback* const downClickCallback() const;
				
				/**
					\brief	Sets the down-click callback
					\author	GW
					\date	02/2017

					\param	callback_	New callback for when the stepper's down-button is clicked
				*/
				virtual void setDownClickCallback(const Callback& callback_);
		};
		
		typedef std::unique_ptr<Stepper> StepperUP;
		typedef std::shared_ptr<Stepper> StepperSP;
		typedef std::weak_ptr<Stepper> StepperWP;
	}
}

#endif
// __HDI_CORE_STEPPER__
