/**
	\file
	\brief		Header file for string preference data storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_STRING_PREF__
#define __HDI_CORE_STRING_PREF__

#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for the storage of strings
		*/
		class StringPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty StringPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a StringPref object is empty, call isEmpty() on it
					\note	Empty StringPref objects do not relate to any actual preference data; they are designed to
							be "receivers" of some other StringPref object via the overloaded assignment operator. Empty
							StringPref objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				StringPref();
			
				/**
					\brief	Constructs a new StringPref object from an existing StringPref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	str_	Existing StringPref object
				*/
				StringPref(const StringPref& str_);

				/**
					\brief	Constructs a StringPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	09/2013

					\param	value_	Initial string value, as UTF-8
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				StringPref(const std::string& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a StringPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~StringPref();

				/**
					\brief		Allows one StringPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target StringPref object, but with its value updated to match that of rhs_
				*/
				virtual StringPref& operator=(const StringPref& rhs_);
				
				/**
					\brief		Convenience method to clone an StringPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new StringPref object
					
					\note		If you subclass StringPref, you MUST overload this method yourself! If you don't and/or 
								your clone() method does not return an instance of your StringPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned StringPref object.
				*/
				virtual StringPref* clone() const;

				/**
					\brief		Gets the string value for this object
					\author		GW
					\date		09/2013

					\returns	The string value being described by this wrapper, as UTF-8
				*/
				virtual std::string stringValue() const;

				/**
					\brief		Sets the string value for this object
					\author		GW
					\date		09/2013

					\param		value_	New string value, as UTF-8
					\returns	true if the string is successfully updated, or false otherwise
				*/
				virtual bool setStringValue(const std::string& value_);
		};
		
		typedef std::unique_ptr<StringPref> StringPrefUP;
		typedef std::shared_ptr<StringPref> StringPrefSP;
		typedef std::weak_ptr<StringPref> StringPrefWP;
	}
}

#endif
// __HDI_CORE_STRING_PREF__
