/**
	\file
	\brief		Header file for stroke style preference data storage
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_STROKE_STYLE_PREF__
#define __HDI_CORE_STROKE_STYLE_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace aip
	{
		class PersistentDictionary;
	}

	namespace core
	{
		/**
			\brief	Allows for storage of stroke styles
		*/
		class StrokeStylePref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty StrokeStylePref object
					\author	GW
					\date	11/2015
					
					\note	To test if a StrokeStylePref object is empty, call isEmpty() on it
					\note	Empty StrokeStylePref objects do not relate to any actual preference data; they are designed to
							be "receivers" of some other StrokeStylePref object via the overloaded assignment operator.
							Empty StrokeStylePref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				StrokeStylePref();
			
				/**
					\brief	Constructs a new StrokeStylePref object from an existing StrokeStylePref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	ss_		Existing StrokeStylePref object
				*/
				StrokeStylePref(const StrokeStylePref& ss_);

				/**
					\brief	Constructs a StrokeStylePref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial stroke style value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				StrokeStylePref(const StrokeStyle& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a StrokeStylePref object
					\author	GW
					\date	11/2015
				*/
				virtual ~StrokeStylePref();

				/**
					\brief		Allows one StrokeStylePref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target StrokeStylePref object, but with its value updated to match that of rhs_
				*/
				virtual StrokeStylePref& operator=(const StrokeStylePref& rhs_);
				
				/**
					\brief		Convenience method to clone an StrokeStylePref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new StrokeStylePref object
					
					\note		If you subclass StrokeStylePref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your StrokeStylePref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned StrokeStylePref object.
				*/
				virtual StrokeStylePref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	StrokeStyleType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the stroke style value for this object
					\author		GW
					\date		11/2015

					\returns	The stroke style value being described by this wrapper
				*/
				virtual StrokeStyle strokeStyle() const;

				/**
					\brief		Sets the stroke style value for this object
					\author		GW
					\date		11/2015

					\param		value_	New stroke style value
					\returns	true if the stroke style is successfully updated, or false otherwise
				*/
				virtual bool setStrokeStyle(const StrokeStyle& value_);


			private:
				friend class PathStylePref;

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2015
				*/
				StrokeStylePref(aip::PersistentDictionary*&);
		};
		
		typedef std::unique_ptr<StrokeStylePref> StrokeStylePrefUP;
		typedef std::shared_ptr<StrokeStylePref> StrokeStylePrefSP;
		typedef std::weak_ptr<StrokeStylePref> StrokeStylePrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_STROKE_STYLE_PREF__
