/**
	\file
	\brief		Header file for general Illustrator text line manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_LINE__
#define __HDI_CORE_TEXT_LINE__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextLine;
	}
	
	namespace core
	{
		class ArtboardPoint;
		class TextArt;
		class TextGlyphRun;
		class TextRange;
	
		class TextLine
		{
			public:
				typedef std::vector< std::shared_ptr<TextGlyphRun> > TextGlyphRunVector;
			
				/**
					\brief	Creates a new TextLine object
					\author	GW
					\date	08/2014
					
					\note	New TextLine objects do not relate to any text line on the Illustrator artboard; they are
							designed to be "receivers" of some other TextLine object via the overloaded assignment
							operator or assigned to some other existent text art.
				*/
				TextLine();
			
				/**
					\brief	TextLine copy constructor
					\author	GW
					\date	08/2014
					
					\param	tl_	TextLine object to copy values from
				*/
				TextLine(const TextLine& tl_);

				/**
					\brief	Destructs a TextLine object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextLine();
			
				/**
					\brief	Overloaded assignment operator for TextLine objects
					\author	GW
					\date	08/2014
				*/
				virtual TextLine& operator=(const TextLine& rhs_);
				
				/**
					\brief		Tests whether a given TextLine object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextLine to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text line, false otherwise
				*/
				virtual bool operator==(const TextLine& rhs_) const;
				
				/**
					\brief		Tests whether a given TextLine object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextLine to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text line, false otherwise
				*/
				virtual bool operator!=(const TextLine& rhs_) const;

				/**
					\brief		Gets the ATE text line ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::ITextLineRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ITextLine> ateTextLine() const;

				/**
					\brief		Gets whether the target TextLine object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextLine object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Retrieves all glyph-runs for the target text line
					\author		GW
					\date		08/2014
					
					\returns	A vector of glyph-run objects
				*/
				virtual TextGlyphRunVector glyphRuns() const;

				/**
					\brief		Retrieves the parent text frame of this line
					\author		GW
					\date		08/2014
					
					\returns	The text frame object
				*/
				virtual std::unique_ptr<TextArt> textFrame() const;

				/**
					\brief		Retrieves the text range for this line
					\author		GW
					\date		08/2014
					
					\returns	The text range object
				*/
				virtual std::unique_ptr<TextRange> textRange() const;
			
				/**
					\brief		Retrieves the count of baselines
					\author		GW
					\date		08/2014
					
					\returns	The baseline count, or -1 for error
				*/
				virtual int32_t baselineCount() const;

				/**
					\brief		Retrieves a baseline with a start and end point in the coordinate space of the containing text frame
					\author		GW
					\date		08/2014
					
					\param		index_	Index of the baseline in question
					\param		start__	Return-by-reference for the baseline start point
					\param		end__	Return-by-reference for the baseline end point
					\returns	true if the baseline could be acquired, false otherwise
				*/
				virtual bool baseline(const int32_t index_, ArtboardPoint& start__, ArtboardPoint& end__) const;


			private:
				friend aip::TextLine* __accessImpl(const TextLine&);
				friend TextLine __accessCtor(const aip::TextLine&);

				/**
					\brief	Private implementation object
				*/
				aip::TextLine* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextLine(const aip::TextLine&);
		};
		
		typedef std::unique_ptr<TextLine> TextLineUP;
		typedef std::shared_ptr<TextLine> TextLineSP;
		typedef std::weak_ptr<TextLine> TextLineWP;
		
		extern aip::TextLine* __accessImpl(const TextLine&);
		extern TextLine __accessCtor(const aip::TextLine&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_LINE__
