/**
	\file
	\brief		Header file for general Illustrator text paragraph feature manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_PARA_FEATURES__
#define __HDI_CORE_TEXT_PARA_FEATURES__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextParaFeatures;
	}

	namespace core
	{
		class TextCharFeatures;
		class TextTabStopSet;
	
		/**
			\brief	Allows for easy manipulation of Illustrator text paragraph features
		*/
		class TextParaFeatures
		{
			public:
				/**
					\brief	Creates a new TextParaFeatures object
					\author	GW
					\date	08/2014
					
					\note	New TextParaFeatures objects do not relate to any text paragraphs on the Illustrator
							artboard; they are designed to be "receivers" of some other TextParaFeatures object via the
							overloaded assignment operator or assigned to some other existent text art.
				*/
				TextParaFeatures();
			
				/**
					\brief	TextParaFeatures copy constructor
					\author	GW
					\date	08/2014
					
					\param	pf_	TextParaFeatures object to copy values from
				*/
				TextParaFeatures(const TextParaFeatures& pf_);

				/**
					\brief	Destructs a TextParaFeatures object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextParaFeatures();
			
				/**
					\brief	Overloaded assignment operator for TextParaFeatures objects
					\author	GW
					\date	08/2014
				*/
				virtual TextParaFeatures& operator=(const TextParaFeatures& rhs_);
				
				/**
					\brief		Tests whether a given TextParaFeatures object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextParaFeatures to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text paragraph features, false otherwise
				*/
				virtual bool operator==(const TextParaFeatures& rhs_) const;
				
				/**
					\brief		Tests whether a given TextParaFeatures object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextParaFeatures to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text paragraph features, false otherwise
				*/
				virtual bool operator!=(const TextParaFeatures& rhs_) const;

				/**
					\brief		Gets the ATE paragraph features ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IParaFeatures for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IParaFeatures> ateParaFeatures() const;

				/**
					\brief		Gets whether the target TextParaFeatures object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextParaFeatures object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Duplicates the target object
					\author		GW
					\date		08/2014
					
					\returns	A duplicate of the target object, or NULL for error
				*/
				virtual std::unique_ptr<TextParaFeatures> duplicate() const;

				/**
					\brief		Retrieves the justification attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		just__	Return-by-reference for the justification enum value
					\returns	true if the attribute could be acquired and just__ was set, false otherwise
				*/
				virtual bool justification(ParagraphJustification& just__) const;

				/**
					\brief		Retrieves the first-line indent attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		indent__	The first-line indent value in document points
					\returns	true if the attribute could be acquired and indent__ was set, false otherwise
				*/
				virtual bool firstLineIndent(double& indent__) const;

				/**
					\brief		Retrieves the paragraph start indent attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		indent__	Return-by-reference for the start indent value in document points
					\returns	true if the attribute could be acquired and indent__ was set, false otherwise
				*/
				virtual bool startIndent(double& indent__) const;

				/**
					\brief		Retrieves the paragraph end indent attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		indent__	Return-by-reference for the end indent value in document points
					\returns	true if the attribute could be acquired and indent__ was set, false otherwise
				*/
				virtual bool endIndent(double& indent__) const;

				/**
					\brief		Retrieves the line space before attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the line space before value in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool spaceBefore(double& space__) const;

				/**
					\brief		Retrieves the line space after attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the line space after value in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool spaceAfter(double& space__) const;

				/**
					\brief		Retrieves the automatic hyphenation attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		hyphen__	Return-by-reference for true if automatic hyphenation is on, false if it is
											off
					\returns	true if the attribute could be acquired and hyphen__ was set, false otherwise
				*/
				virtual bool autoHyphenate(bool& hyphen__) const;

				/**
					\brief		Retrieves the hyphenation word size attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		size__	Return-by-reference for the hyphenation word size value (number of characters)
					\returns	true if the attribute could be acquired and size__ was set, false otherwise
				*/
				virtual bool hyphenatedWordSize(int32_t& size__) const;

				/**
					\brief		Retrieves the pre-hyphen size attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		size__	Return-by-reference for the pre-hyphen size value (number of characters)
					\returns	true if the attribute could be acquired and size__ was set, false otherwise
				*/
				virtual bool preHyphenSize(int32_t& size__) const;

				/**
					\brief		Retrieves the post-hyphen size attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		size__	Return-by-reference for the post-hyphen size value (number of characters)
					\returns	true if the attribute could be acquired and size__ was set, false otherwise
				*/
				virtual bool postHyphenSize(int32_t& size__) const;

				/**
					\brief		Retrieves the consecutive hyphen limit attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		limit__	Return-by-reference for the consecutive hyphen limit value (number of hyphens)
					\returns	true if the attribute could be acquired and limit__ was set, false otherwise
				*/
				virtual bool consecutiveHyphenLimit(int32_t& limit__) const;

				/**
					\brief		Retrieves the hyphenation zone attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		zone__	Return-by-reference for the hyphenation zone value in document points
					\returns	true if the attribute could be acquired and zone__ was set, false otherwise
				*/
				virtual bool hyphenationZone(double& zone__) const;

				/**
					\brief		Retrieves the hyphenation capitalization attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		cap__	Return-by-reference for true if hyphenation capitalization is on, false if it is
										off
					\returns	true if the attribute could be acquired and cap__ was set, false otherwise
				*/
				virtual bool hyphenateCapitalized(bool& cap__) const;

				/**
					\brief		Retrieves the hyphenation preference attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		pref__	Return-by-reference for the hyphenation preference value
					\returns	true if the attribute could be acquired and pref__ was set, false otherwise
				*/
				virtual bool hyphenationPreference(double& pref__) const;

				/**
					\brief		Retrieves the desired word spacing attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the desired word spacing value, in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool desiredWordSpacing(double& space__) const;

				/**
					\brief		Retrieves the minimum word spacing attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the minimum word spacing value, in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool minWordSpacing(double& space__) const;

				/**
					\brief		Retrieves the maximum word spacing attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the maximum word spacing value, in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool maxWordSpacing(double& space__) const;

				/**
					\brief		Retrieves the desired letter spacing attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the desired letter spacing value, in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool desiredLetterSpacing(double& space__) const;

				/**
					\brief		Retrieves the minimum letter spacing attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the minimum letter spacing value, in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool minLetterSpacing(double& space__) const;

				/**
					\brief		Retrieves the maximum letter spacing attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		space__	Return-by-reference for the maximum letter spacing value, in document points
					\returns	true if the attribute could be acquired and space__ was set, false otherwise
				*/
				virtual bool maxLetterSpacing(double& space__) const;

				/**
					\brief		Retrieves the desired glyph scaling attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		scale__	Return-by-reference for the desired glyph scaling value, where 1.0 is 100.0%
					\returns	true if the attribute could be acquired and scale__ was set, false otherwise
				*/
				virtual bool desiredGlyphScaling(double& scale__) const;

				/**
					\brief		Retrieves the minimum glyph scaling attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		scale__	Return-by-reference for the minimum glyph scaling value, where 1.0 is 100.0%
					\returns	true if the attribute could be acquired and scale__ was set, false otherwise
				*/
				virtual bool minGlyphScaling(double& scale__) const;

				/**
					\brief		Retrieves the maximum glyph scaling attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		scale__	Return-by-reference for the maximum glyph scaling value, where 1.0 is 100.0%
					\returns	true if the attribute could be acquired and scale__ was set, false otherwise
				*/
				virtual bool maxGlyphScaling(double& scale__) const;

				/**
					\brief		Retrieves the single-word justification attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		just__	Return-by-reference for the single-word justification type
					\returns	true if the attribute could be acquired and just__ was set, false otherwise
				*/
				virtual bool singleWordJustification(ParagraphJustification& just__) const;

				/**
					\brief		Retrieves the automatic leading percentage attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		perc__	Return-by-reference for the automatic leading percentage value, where 1.0 is
										100.0%
					\returns	true if the attribute could be acquired and perc__ was set, false otherwise
				*/
				virtual bool autoLeadingPercentage(double& perc__) const;

				/**
					\brief		Retrieves the leading type attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		type__	Return-by-reference for the leading type
					\returns	true if the attribute could be acquired and type__ was set, false otherwise
				*/
				virtual bool leadingType(LeadingType& type__) const;

				/**
					\brief		Retrieves the tab stops attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		stops__	Return-by-reference for the tab stops object
					\returns	true if the attribute could be acquired and stops__ was set, false otherwise
				*/
				virtual bool tabStops(TextTabStopSet& stops__) const;

				/**
					\brief		Retrieves the default tab width attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		width__	Return-by-reference for the default tab width value, in document points
					\returns	true if the attribute could be acquired and width__ was set, false otherwise
				*/
				virtual bool defaultTabWidth(double& width__) const;

				/**
					\brief		Retrieves the every-line composer attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		comp__	Return-by-reference for true if every-line composer is on, false if it is off
					\returns	true if the attribute could be acquired and comp__ was set, false otherwise
				*/
				virtual bool everyLineComposer(bool& comp__) const;

				/**
					\brief		Retrieves the default character features attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		feat__	Return-by-reference for the character features object
					\returns	true if the attribute could be acquired and feat__ was set, false otherwise
				*/
				virtual bool defaultCharFeatures(TextCharFeatures& feat__) const;

				/**
					\brief		Retrieves the composer engine attribute of this paragraph
					\author		GW
					\date		08/2014
					
					\param		engine__	Return-by-reference for the composer engine object
					\returns	true if the attribute could be acquired and engine__ was set, false otherwise
				*/
				virtual bool composerEngine(ComposerEngine& engine__) const;
	
				/**
					\brief	Sets the local justification attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	just_	The new justification type
				*/
				virtual void setJustification(const ParagraphJustification just_);

				/**
					\brief	Sets the local first-line indent attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	indent_	The new first-line indent value, in document points
				*/
				virtual void setFirstLineIndent(const double indent_);

				/**
					\brief	Sets the local start indent attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	indent_	The new start indent value, in document points
				*/
				virtual void setStartIndent(const double indent_);

				/**
					\brief	Sets the local end indent attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	indent_	The new end indent value, in document points
				*/
				virtual void setEndIndent(const double indent_);

				/**
					\brief	Sets the local line space before attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new line space before value, in document points
				*/
				virtual void setSpaceBefore(const double space_);

				/**
					\brief	Sets the local line space after attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new line space after value, in document points
				*/
				virtual void setSpaceAfter(const double space_);

				/**
					\brief	Sets the local automatic hyphenation attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	hyphen_	true to turn automatic hyphenation on, false to turn it off
				*/
				virtual void setAutoHyphenate(const bool hyphen_);

				/**
					\brief	Sets the local hyphenated word size attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	size_	The new hyphenated word size value, in characters
				*/
				virtual void setHyphenatedWordSize(const int32_t size_);

				/**
					\brief	Sets the local pre-hyphen size attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	size_	The new pre-hyphen size value, in characters
				*/
				virtual void setPreHyphenSize(const int32_t size_);

				/**
					\brief	Sets the local post-hyphen size attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	size_	The new post-hyphen size value, in characters
				*/
				virtual void setPostHyphenSize(const int32_t size_);

				/**
					\brief	Sets the local consecutive hyphen limit attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	limit_	The new consecutive hyphen limit value, a number of hyphens
				*/
				virtual void setConsecutiveHyphenLimit(const int32_t limit_);

				/**
					\brief	Sets the local hyphenation zone attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	zone_	The new hyphenation zone value, in document points
				*/
				virtual void setHyphenationZone(const double zone_);

				/**
					\brief	Sets the local hyphenation capitalized attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	cap_	true to turn hyphenation capitalized on, false to turn it off
				*/
				virtual void setHyphenateCapitalized(const bool cap_);

				/**
					\brief	Sets the local hyphenation preference attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	pref_	The new hyphenation preference value
				*/
				virtual void setHyphenationPreference(const double pref_);

				/**
					\brief	Sets the local desired word spacing attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new desired word spacing value, in document points
				*/
				virtual void setDesiredWordSpacing(const double space_);

				/**
					\brief	Sets the local minimum word spacing attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new minimum word spacing value, in document points
				*/
				virtual void setMinWordSpacing(const double space_);

				/**
					\brief	Sets the local maximum word spacing attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new maximum word spacing value, in document points
				*/
				virtual void setMaxWordSpacing(const double space_);

				/**
					\brief	Sets the local desired letter spacing attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new desired letter spacing value, in document points
				*/
				virtual void setDesiredLetterSpacing(const double space_);

				/**
					\brief	Sets the local minimum letter spacing attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new minimum letter spacing value, in document points
				*/
				virtual void setMinLetterSpacing(const double space_);

				/**
					\brief	Sets the local maximum letter spacing attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	space_	The new maximum letter spacing value, in document points
				*/
				virtual void setMaxLetterSpacing(const double space_);

				/**
					\brief	Sets the local desired glyph scaling attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	scale_	The new desired glyph scaling value, where 1.0 is 100.0%
				*/
				virtual void setDesiredGlyphScaling(const double scale_);

				/**
					\brief	Sets the local minimum glyph scaling attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	scale_	The new minimum glyph scaling value, where 1.0 is 100.0%
				*/
				virtual void setMinGlyphScaling(const double scale_);

				/**
					\brief	Sets the local maximum glyph scaling attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	scale_	The new maximum glyph scaling value, where 1.0 is 100.0%
				*/
				virtual void setMaxGlyphScaling(const double scale_);

				/**
					\brief	Sets the local single-word justification attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	just_	The new single-word justification type
				*/
				virtual void setSingleWordJustification(const ParagraphJustification just_);

				/**
					\brief	Sets the local automatic leading percentage attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	perc_	The new automatic leading percentage value, where 1.0 is 100.0%
				*/
				virtual void setAutoLeadingPercentage(const double perc_);

				/**
					\brief	Sets the local leading type attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	type_	The new leading type
				*/
				virtual void setLeadingType(const LeadingType type_);

				/**
					\brief	Sets the local tab stops attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	stops_	The new tab stops object
				*/
				virtual void setTabStops(const TextTabStopSet& stops_);

				/**
					\brief	Sets the local default tab width attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	width_	The new default tab width value, in document points
				*/
				virtual void setDefaultTabWidth(const double width_);

				/**
					\brief	Sets the local every-line composer attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	comp_	true to turn every-line composer on, false to turn it off
				*/
				virtual void setEveryLineComposer(const bool comp_);

				/**
					\brief	Sets the local default character features attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	feat_	The new default character features object
				*/
				virtual void setDefaultCharFeatures(const TextCharFeatures& feat_);

				/**
					\brief	Sets the composer engine attribute of this paragraph
					\author	GW
					\date	08/2014

					\param	engine_	The new composer engine object
				*/
				virtual void setComposerEngine(const ComposerEngine engine_);

				/**
					\brief	Removes the local justification attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearJustification();

				/**
					\brief	Removes the local first-line indent attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearFirstLineIndent();

				/**
					\brief	Removes the local start indent attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearStartIndent();

				/**
					\brief	Removes the local end indent attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearEndIndent();

				/**
					\brief	Removes the local line space before attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearSpaceBefore();

				/**
					\brief	Removes the local line space after attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearSpaceAfter();

				/**
					\brief	Removes the local automatic hyphenation attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearAutoHyphenate();

				/**
					\brief	Removes the local hyphenation word size attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearHyphenatedWordSize();

				/**
					\brief	Removes the local pre-hyphen size attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearPreHyphenSize();

				/**
					\brief	Removes the local post-hyphen size attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearPostHyphenSize();

				/**
					\brief	Removes the local consecutive hyphen limit attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearConsecutiveHyphenLimit();

				/**
					\brief	Removes the local hyphenation zone attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearHyphenationZone();

				/**
					\brief	Removes the local hyphenate capitalized attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearHyphenateCapitalized();

				/**
					\brief	Removes the local hyphenation preference attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearHyphenationPreference();

				/**
					\brief	Removes all of the local word spacing attribute values of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearWordSpacing();

				/**
					\brief	Removes all of the local letter spacing attribute values of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearLetterSpacing();

				/**
					\brief	Removes all of the local glyph scaling attribute values of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearGlyphScaling();

				/**
					\brief	Removes the local single-word justification attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearSingleWordJustification();

				/**
					\brief	Removes the local automatic leading percentage attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearAutoLeadingPercentage();

				/**
					\brief	Removes the local leading type attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearLeadingType();

				/**
					\brief	Removes the local tab stops attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearTabStops();

				/**
					\brief	Removes the local every-line composer attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearEveryLineComposer();

				/**
					\brief	Removes the local default character features attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearDefaultCharFeatures();

				/**
					\brief	Removes the composer engine attribute value of this paragraph
					\author	GW
					\date	08/2014
				*/
				virtual void clearComposerEngine();


			private:
				friend aip::TextParaFeatures* __accessImpl(const TextParaFeatures&);
				friend TextParaFeatures __accessCtor(const aip::TextParaFeatures&);

				/**
					\brief	Private implementation object
				*/
				aip::TextParaFeatures* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextParaFeatures(const aip::TextParaFeatures&);
		};
		
		typedef std::unique_ptr<TextParaFeatures> TextParaFeaturesUP;
		typedef std::shared_ptr<TextParaFeatures> TextParaFeaturesSP;
		typedef std::weak_ptr<TextParaFeatures> TextParaFeaturesWP;
		
		extern aip::TextParaFeatures* __accessImpl(const TextParaFeatures&);
		extern TextParaFeatures __accessCtor(const aip::TextParaFeatures&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_PARA_FEATURES__
