/**
	\file
	\brief		Header file for Illustrator text paragraph inspection
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_PARA_INSPECTOR__
#define __HDI_CORE_TEXT_PARA_INSPECTOR__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextParaInspector;
	}

	namespace core
	{
		/**
			\brief	Allows for easy inspection of Illustrator text characters
		*/
		class TextParaInspector
		{
			public:
				typedef std::vector<bool> BoolVector;
				typedef std::vector<double> DoubleVector;
				typedef std::vector<int32_t> IntVector;
			
				typedef std::vector<ComposerEngine> ComposerEngineVector;
				typedef std::vector<LeadingType> LeadingTypeVector;
				typedef std::vector<ParagraphJustification> ParagraphJustificationVector;
			
				/**
					\brief	Creates a new TextParaInspector object
					\author	GW
					\date	12/2013
					
					\note	New TextParaInspector objects do not relate to any text characters on the Illustrator
							artboard; they are designed to be "receivers" of some other TextParaInspector object via the
							overloaded assignment operator or assigned to some other existent text art.
				*/
				TextParaInspector();
			
				/**
					\brief	TextParaInspector copy constructor
					\author	GW
					\date	12/2013
					
					\param	pi_	TextParaInspector object to copy values from
				*/
				TextParaInspector(const TextParaInspector& pi_);

				/**
					\brief	Destructs a TextParaInspector object
					\author	GW
					\date	12/2013
				*/
				virtual ~TextParaInspector();
			
				/**
					\brief	Overloaded assignment operator for TextParaInspector objects
					\author	GW
					\date	12/2013
				*/
				virtual TextParaInspector& operator=(const TextParaInspector& rhs_);
				
				/**
					\brief		Tests whether a given TextParaInspector object is the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	TextParaInspector to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text char inspector, false otherwise
				*/
				virtual bool operator==(const TextParaInspector& rhs_) const;
				
				/**
					\brief		Tests whether a given TextParaInspector object is not the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	TextParaInspector to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text char inspectors, false otherwise
				*/
				virtual bool operator!=(const TextParaInspector& rhs_) const;

				/**
					\brief		Gets the ATE char inspector ref around which the target object is wrapped
					\author		GW
					\date		12/2013
					
					\returns	ATE::IParaInspector for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IParaInspector> ateParaInspector() const;

				/**
					\brief		Gets whether the target TextParaInspector object is empty
					\author		GW
					\date		12/2013
					
					\returns	true if the target TextParaInspector object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Retrieves the justification values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of justification settings for each paragraph being inspected
				*/
				virtual ParagraphJustificationVector justification() const;
				
				/**
					\brief		Retrieves the first-line indent values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of first-line indent settings for each paragraph being inspected
				*/
				virtual DoubleVector firstLineIndent() const;
				
				/**
					\brief		Retrieves the start-indent values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of start-indent settings for each paragraph being inspected
				*/
				virtual DoubleVector startIndent() const;
				
				/**
					\brief		Retrieves the end-indent values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of end-indent settings for each paragraph being inspected
				*/
				virtual DoubleVector endIndent() const;
				
				/**
					\brief		Retrieves the space-before values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of space-before settings for each paragraph being inspected
				*/
				virtual DoubleVector spaceBefore() const;
				
				/**
					\brief		Retrieves the space-after values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of space-after settings for each paragraph being inspected
				*/
				virtual DoubleVector spaceAfter() const;
				
				/**
					\brief		Retrieves the auto-hyphenate values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of auto-hyphenate settings for each paragraph being inspected
				*/
				virtual BoolVector autoHyphenate() const;
				
				/**
					\brief		Retrieves the hyphenated word size values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of hyphenated word sizes for each paragraph being inspected
				*/
				virtual IntVector hyphenatedWordSize() const;
				
				/**
					\brief		Retrieves the pre-hyphen size values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of pre-hyphen sizes for each paragraph being inspected
				*/
				virtual IntVector preHyphenSize() const;
				
				/**
					\brief		Retrieves the post-hyphen size values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of post-hyphen sizes for each paragraph being inspected
				*/
				virtual IntVector postHyphenSize() const;
				
				/**
					\brief		Retrieves the consecutive hyphen limit values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of consecutive hyphen limits for each paragraph being inspected
				*/
				virtual IntVector consecutiveHyphenLimit() const;
				
				/**
					\brief		Retrieves the hyphenation zone values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of hyphenation zones for each paragraph being inspected
				*/
				virtual DoubleVector hyphenationZone() const;
				
				/**
					\brief		Retrieves the hyphenate capitalized settings from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of hyphenate capitalized settings for each paragraph being inspected
				*/
				virtual BoolVector hyphenateCapitalized() const;
				
				/**
					\brief		Retrieves the hyphenation pref values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of hyphenation prefs for each paragraph being inspected
				*/
				virtual DoubleVector hyphenationPreference() const;
				
				/**
					\brief		Retrieves the desired word spacing values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of desired word spacings for each paragraph being inspected
				*/
				virtual DoubleVector desiredWordSpacing() const;
				
				/**
					\brief		Retrieves the min word spacing values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of min word spacings for each paragraph being inspected
				*/
				virtual DoubleVector minWordSpacing() const;
				
				/**
					\brief		Retrieves the max word spacing values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of max word spacings for each paragraph being inspected
				*/
				virtual DoubleVector maxWordSpacing() const;
				
				/**
					\brief		Retrieves the desired letter spacing values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of desired letter spacings for each paragraph being inspected
				*/
				virtual DoubleVector desiredLetterSpacing() const;
				
				/**
					\brief		Retrieves the min letter spacing values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of min letter spacings for each paragraph being inspected
				*/
				virtual DoubleVector minLetterSpacing() const;
				
				/**
					\brief		Retrieves the max letter spacing values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of max letter spacings for each paragraph being inspected
				*/
				virtual DoubleVector maxLetterSpacing() const;
				
				/**
					\brief		Retrieves the desired glyph scaling values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of desired glyph scalings for each paragraph being inspected
				*/
				virtual DoubleVector desiredGlyphScaling() const;
				
				/**
					\brief		Retrieves the min glyph scaling values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of min glyph scalings for each paragraph being inspected
				*/
				virtual DoubleVector minGlyphScaling() const;
				
				/**
					\brief		Retrieves the max glyph scaling values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of max glyph scalings for each paragraph being inspected
				*/
				virtual DoubleVector maxGlyphScaling() const;
				
				/**
					\brief		Retrieves the single-word justification values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of single-word justifications for each paragraph being inspected
				*/
				virtual ParagraphJustificationVector singleWordJustification() const;
				
				/**
					\brief		Retrieves the auto-leading percentage values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of auto-leading percentages for each paragraph being inspected
				*/
				virtual DoubleVector autoLeadingPercentage() const;
				
				/**
					\brief		Retrieves the leading types from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of leading types for each paragraph being inspected
				*/
				virtual LeadingTypeVector leadingType() const;
				
				/**
					\brief		Retrieves the default tab width values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of default tab widths for each paragraph being inspected
				*/
				virtual DoubleVector defaultTabWidth() const;
				
				/**
					\brief		Retrieves the every-line composer values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of every-line composer values for each paragraph being inspected
				*/
				virtual BoolVector everyLineComposer() const;
				
				/**
					\brief		Retrieves the composer engine values from the paragraph set
					\author		GW
					\date		12/2013
					
					\returns	A vector of composer engines for each paragraph being inspected
				*/
				virtual ComposerEngineVector composerEngine() const;


			private:
				friend aip::TextParaInspector* __accessImpl(const TextParaInspector&);
				friend TextParaInspector __accessCtor(const aip::TextParaInspector&);

				/**
					\brief	Private implementation object
				*/
				aip::TextParaInspector* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	12/2013
				*/
				TextParaInspector(const aip::TextParaInspector&);
		};
		
		typedef std::unique_ptr<TextParaInspector> TextParaInspectorUP;
		typedef std::shared_ptr<TextParaInspector> TextParaInspectorSP;
		typedef std::weak_ptr<TextParaInspector> TextParaInspectorWP;
		
		extern aip::TextParaInspector* __accessImpl(const TextParaInspector&);
		extern TextParaInspector __accessCtor(const aip::TextParaInspector&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_PARA_INSPECTOR__
