/**
	\file
	\brief		Header file for general Illustrator paragraph styles manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_PARA_STYLE__
#define __HDI_CORE_TEXT_PARA_STYLE__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextParaStyle;
	}
	
	namespace core
	{
		class TextParaFeatures;
	
		class TextParaStyle
		{
			public:
				/**
					\brief	Creates a new TextParaStyle object
					\author	GW
					\date	08/2014
					
					\note	New TextParaStyle objects do not relate to any text paragraph style on the Illustrator
							artboard; they are designed to be "receivers" of some other TextParaStyle object via the
							overloaded assignment operator or assigned to some other existent text art.
				*/
				TextParaStyle();
			
				/**
					\brief	TextParaStyle copy constructor
					\author	GW
					\date	08/2014
					
					\param	ps_	TextParaStyle object to copy values from
				*/
				TextParaStyle(const TextParaStyle& ps_);

				/**
					\brief	Destructs a TextParaStyle object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextParaStyle();
			
				/**
					\brief	Overloaded assignment operator for TextParaStyle objects
					\author	GW
					\date	08/2014
				*/
				virtual TextParaStyle& operator=(const TextParaStyle& rhs_);
				
				/**
					\brief		Tests whether a given TextParaStyle object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextParaStyle to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text paragraph style, false otherwise
				*/
				virtual bool operator==(const TextParaStyle& rhs_) const;
				
				/**
					\brief		Tests whether a given TextParaStyle object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextParaStyle to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text paragraph styles, false otherwise
				*/
				virtual bool operator!=(const TextParaStyle& rhs_) const;

				/**
					\brief		Gets the ATE paragraph style ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IParaStyleRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IParaStyle> ateParaStyle() const;

				/**
					\brief		Gets whether the target TextParaStyle object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextParaStyle object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets the name of the paragraph style
					\author		GW
					\date		08/2014
					
					\returns	The paragraph style name, as UTF-8
				*/
				virtual std::string name() const;

				/**
					\brief		Sets the name of the paragraph style
					\author		GW
					\date		08/2014
					
					\param		name_	New paragraph style name, as UTF-8
					\returns	true if the name could be set, false otherwise
				*/
				virtual bool setName(const std::string& name_);

				/**
					\brief		Gets the parent style of the target paragraph style object
					\author		GW
					\date		08/2014
					
					\returns	The TextParaStyle object for the parent, or NULL for none
				*/
				virtual std::unique_ptr<TextParaStyle> parent() const;

				/**
					\brief		Sets the parent style of the target paragraph style object
					\author		GW
					\date		08/2014
					
					\param		parent_		New TextParaStyle parent
					\returns	true if the parent could be set, false otherwise
				*/
				virtual bool setParent(const TextParaStyle& parent_);

				/**
					\brief		Gets whether the target object has a parent
					\author		GW
					\date		08/2014
					
					\returns	true if a parent exists, false otherwise
				*/
				virtual bool hasParent() const;

				/**
					\brief		Gets the features defined for the target style
					\author		GW
					\date		08/2014
					
					\returns	The TextParaFeatures object for the style, or NULL for error
				*/
				virtual std::unique_ptr<TextParaFeatures> features() const;

				/**
					\brief	Sets the features defined for the target style
					\author	GW
					\date	08/2014
					
					\param	feat_	New TextParaFeatures object for the style
				*/
				virtual void setFeatures(const TextParaFeatures& feat_);

				/**
					\brief	Replaces existing features in the target style, or adds them anew if they were not already
							present (values that are unassigned in the replacement set remain unchanged in the target 
							style).
					\author	GW
					\date	08/2014
					
					\param	feat_	New/replacement TextParaFeatures object for the style
				*/
				virtual void replaceOrAddFeatures(const TextParaFeatures& feat_);


			private:
				friend aip::TextParaStyle* __accessImpl(const TextParaStyle&);
				friend TextParaStyle __accessCtor(const aip::TextParaStyle&);

				/**
					\brief	Private implementation object
				*/
				aip::TextParaStyle* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextParaStyle(const aip::TextParaStyle&);
		};
		
		typedef std::unique_ptr<TextParaStyle> TextParaStyleUP;
		typedef std::shared_ptr<TextParaStyle> TextParaStyleSP;
		typedef std::weak_ptr<TextParaStyle> TextParaStyleWP;
		
		extern aip::TextParaStyle* __accessImpl(const TextParaStyle&);
		extern TextParaStyle __accessCtor(const aip::TextParaStyle&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_PARA_STYLE__
