/**
	\file
	\brief		Header file for Illustrator text range manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_RANGE__
#define __HDI_CORE_TEXT_RANGE__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextRange;
	}

	namespace core
	{
		class TextArt;
		class TextCharFeatures;
		class TextCharInspector;
		class TextCharStyleSet;
		class TextPara;
		class TextParaFeatures;
		class TextParaInspector;
		class TextParaStyleSet;
		class TextRangeSet;
		class TextStory;
		class TextWord;
	
		/**
			\brief	Allows for easy manipulation of Illustrator text ranges
		*/
		class TextRange
		{
			public:
				typedef std::vector< std::shared_ptr<TextArt> > TextArtVector;
				typedef std::vector< std::shared_ptr<TextPara> > TextParaVector;
				typedef std::vector< std::shared_ptr<TextRange> > TextRangeVector;
				typedef std::vector< std::shared_ptr<TextWord> > TextWordVector;
			
				enum CollapseDirection
				{
					UnknownCollapseDir	= 0,
					CollapseEndDir		= 10,
					CollapseStartDir	= 20
				};

				/**
					\brief	Creates a new TextRange object
					\author	GW
					\date	12/2013
					
					\note	New TextRange objects do not relate to any text range on the Illustrator artboard; they
							are designed to be "receivers" of some other TextRange object via the overloaded assignment
							operator or assigned to some other existent text art.
				*/
				TextRange();
			
				/**
					\brief	TextRange copy constructor
					\author	GW
					\date	12/2013
					
					\param	tr_	TextRange object to copy values from
				*/
				TextRange(const TextRange& tr_);

				/**
					\brief	Destructs a TextRange object
					\author	GW
					\date	12/2013
				*/
				virtual ~TextRange();
			
				/**
					\brief	Overloaded assignment operator for TextRange objects
					\author	GW
					\date	12/2013
				*/
				virtual TextRange& operator=(const TextRange& rhs_);
				
				/**
					\brief		Tests whether a given TextRange object is the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	TextRange to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text range, false otherwise
				*/
				virtual bool operator==(const TextRange& rhs_) const;
				
				/**
					\brief		Tests whether a given TextRange object is not the same as another
					\author		GW
					\date		12/2013

					\param		rhs_	TextRange to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text ranges, false otherwise
				*/
				virtual bool operator!=(const TextRange& rhs_) const;

				/**
					\brief		Gets the ATE text range ref around which the target object is wrapped
					\author		GW
					\date		12/2013
					
					\returns	ATE::ITextRange for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ITextRange> ateTextRange() const;

				/**
					\brief		Gets whether the target TextRange object is empty
					\author		GW
					\date		12/2013
					
					\returns	true if the target TextRange object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Duplicates the target text range, returning a new unique TextRange object
					\author		GW
					\date		12/2013
					
					\returns	A new TextRange object for the duplicate
				*/
				virtual std::unique_ptr<TextRange> duplicate() const;
				
				/**
					\brief		Gets the start offset of the target text range
					\author		GW
					\date		12/2013
					
					\returns	The 0-based index of the start offset from the beginning of the containing story
				*/
				virtual int32_t start() const;
				
				/**
					\brief	Sets the start offset of the target text range
					\author	GW
					\date	12/2013
					
					\param	end_	The 0-based index of the start offset from the beginning of the containing story
				*/
				virtual void setStart(const int32_t start_);
				
				/**
					\brief		Gets the end offset of the target text range
					\author		GW
					\date		12/2013
					
					\returns	The 0-based index of the end offset from the beginning of the containing story
				*/
				virtual int32_t end() const;
				
				/**
					\brief	Sets the end offset of the target text range
					\author	GW
					\date	12/2013
					
					\param	end_	The 0-based index of the end offset from the beginning of the containing story
				*/
				virtual void setEnd(const int32_t end_);

				/**
					\brief		Retrieves the number of characters in the target text range
					\author		GW
					\date		12/2013
					
					\returns	The size of the text range
				*/
				virtual int32_t size() const;
				
				/**
					\brief	Convenience method to set both the start and end points of the target text range
					\author	GW
					\date	12/2013
					
					\param	start_	The 0-based index of the start offset from the beginning of the containing story
					\param	end_	The 0-based index of the end offset from the beginning of the containing story
				*/
				virtual void setRange(const int32_t start_, const int32_t end_);
				
				/**
					\brief	Resets start or end point of the target range so that it contains only one character, either
							the first or the last
					\author	GW
					\date	12/2013
					
					\param	dir_	The direction constant to collapse; CollapseEndDir sets the start offset to the end
									offset, while CollapseStartDir sets the end offset to the start offset
				*/
				virtual void collapse(const CollapseDirection dir_ = CollapseEndDir);
				
				/**
					\brief		Moves the target text range by adding or subtracting a number of characters to or from
								the start and end offsets
					\author		GW
					\date		12/2013
					
					\param		unit_	The number of characters, positive to move the range toward the end, negative to
										move it toward the beginning of the story
					\returns	The number of characters by which the range was translated, or 0 if the translation
								could not be made within the story bounds

					\note		Does not move the range if the result would be out of the story bounds.
				*/
				virtual int32_t move(const int32_t unit_);

				/**
					\brief		Retrieves the story that contains this text range
					\author		GW
					\date		08/2014
					
					\returns	The story object
				*/
				virtual std::unique_ptr<TextStory> story() const;

				/**
					\brief	Sets the story that contains this text range
					\author	GW
					\date	08/2014
					
					\param	New containing story
				*/
				virtual void setStory(const TextStory& story_);

				/**
					\brief		Retrieves the selected text within this text range
					\author		GW
					\date		08/2014
					
					\returns	The text ranges object containing the selected text
				*/
				virtual std::unique_ptr<TextRangeSet> selections() const;

				/**
					\brief		Gets a vector of the text frames contained in this text range
					\author		GW
					\date		08/2014
					
					\returns	The vector of text frames
				*/
				virtual TextArtVector textFrames() const;

				/**
					\brief		Gets a vector of the paragraphs contained in this text range
					\author		GW
					\date		08/2014
					
					\returns	The vector of paragraphs
				*/
				virtual TextParaVector paragraphs() const;

				/**
					\brief		Gets a vector of the words contained in this text range
					\author		GW
					\date		08/2014
					
					\returns	The vector of words
				*/
				virtual TextWordVector words() const;

				/**
					\brief		Gets a vector of the text ranges (that share character attributes) contained in this
								text range
					\author		GW
					\date		08/2014
					
					\returns	The vector of text ranges
				*/
				virtual TextRangeVector runs() const;

				/**
					\brief		Creates an inspector with which to access the character features of all characters in
								the target text range
					\author		GW
					\date		12/2013
					
					\returns	The new character inspector object
				*/
				virtual std::unique_ptr<TextCharInspector> charInspector() const;

				/**
					\brief		Creates an inspector with which to access the paragraph features of all paragraphs in
								the target text range
					\author		GW
					\date		12/2013
					
					\returns	The new paragraph inspector object
				*/
				virtual std::unique_ptr<TextParaInspector> paraInspector() const;

				/**
					\brief		Retrieves a set of all named character styles used in the target text range
					\author		GW
					\date		08/2014
					
					\returns	The character style set object
				*/
				virtual std::unique_ptr<TextCharStyleSet> namedCharStyles() const;
				
				/**
					\brief		Associates a named character style to the target text range. The inherited values can be
								overridden by styles or features specified locally in contained text ranges or
								individual characters.
					\author		GW
					\date		08/2014
					
					\param		The name of the style, as UTF-8
					\returns	true if the style is successfully applied, false if there is no style with the specified 
								name
				*/
				virtual bool setNamedCharStyle(const std::string& name_);
				
				/**
					\brief	Removes the association of the target text range and its character style. Copies the feature
							values of the character style into local override values in the contained characters.
					\author	GW
					\date	08/2014
				*/
				virtual void clearNamedCharStyle();

				/**
					\brief		Retrieves a set of all named paragraph styles used in the target text range
					\author		GW
					\date		08/2014
					
					\returns	The paragraph style set object
				*/
				virtual std::unique_ptr<TextParaStyleSet> namedParaStyles() const;
				
				/**
					\brief		Associates a named paragraph style to the target text range. The inherited values can be
								overridden by styles or features specified locally in contained text ranges or
								individual paragraphs.
					\author		GW
					\date		08/2014
					
					\param		The name of the style, as UTF-8
					\returns	true if the style is successfully applied, false if there is no style with the specified 
								name
				*/
				virtual bool setNamedParaStyle(const std::string& name_);
				
				/**
					\brief	Removes the association of the target text range and its character style. Copies the feature
							values of the character style into local override values in the contained characters.
					\author	GW
					\date	08/2014
				*/
				virtual void clearNamedParaStyle();

				/**
					\brief		Retrieves the unique character features used in the target text range. Unique features
								are those which have the same value in all text runs in the range.
					\author		GW
					\date		08/2014
					
					\returns	The character features object containing the unique feature values. Other features are
								unassigned.
				*/
				virtual std::unique_ptr<TextCharFeatures> uniqueCharFeatures() const;

				/**
					\brief		Retrieves the unique paragraph features used in the target text range. Unique features
								are those which have the same value in all text runs in the range.
					\author		GW
					\date		08/2014
					
					\returns	The paragraph features object containing the unique feature values. Other features are
								unassigned.
				*/
				virtual std::unique_ptr<TextParaFeatures> uniqueParaFeatures() const;
				
				/**
					\brief		Reports whether there any local character feature overrides for characters contained in
								the target text range
					\author		GW
					\date		08/2014
					
					\returns	true if there are local overrides, false otherwise
				*/
				virtual bool hasLocalCharFeatures() const;
				
				/**
					\brief		Retrieves the character features that have local overrides in the target text range, and
								whose local values are the same in all text runs in the range
					\author		GW
					\date		08/2014
					
					\returns	The character features object containing the unique local feature values. Other features
								are unassigned. If all features are unassigned, either there are no local overrides, or
								the local overrides have no common values.
				*/
				virtual std::unique_ptr<TextCharFeatures> localCharFeatures() const;
				
				/**
					\brief	Replaces all of the local overrides for all characters in the target text range with a new
							set of feature values. All values that are assigned become local values, replacing any
							previous local value. These local values override any specified in a style associated with
							the character or the text range. All values that are unassigned remove any previous local
							values, so that those values are inherited.
					\author	GW
					\date	08/2014
					
					\param	feat_	The new feature set object
					
					\note	Triggers a reflow operation that can cause previously obtained text runs to become invalid.
				*/
				virtual void setLocalCharFeatures(const TextCharFeatures& feat_);
				
				/**
					\brief	Modifies the local overrides for all characters	in the target text range. All values that
							are assigned become local values, replacing any previous local value. Values that are
							unassigned leave any previous local values unchanged.
					\author	GW
					\date	08/2014
					
					\param	The new feature set object
					
					\note	Triggers a reflow operation that can cause previously obtained text runs to become invalid.
				*/
				virtual void replaceOrAddLocalCharFeatures(const TextCharFeatures& feat_);
				
				/**
					\brief	Removes all local overrides for all characters in the target text range. All character
							features are then inherited from styles associated with the character or text range, or from
							the normal style.
					\author	GW
					\date	08/2014
				*/
				virtual void clearLocalCharFeatures();
				
				/**
					\brief		Reports whether there any local paragraph feature overrides for paragraphs contained in
								the target text range
					\author		GW
					\date		08/2014
					
					\returns	true if there are local overrides, false otherwise
				*/
				virtual bool hasLocalParaFeatures() const;
				
				/**
					\brief		Retrieves the paragraph features that have local overrides in the target text range, and
								whose local values are the same in all text runs in the range
					\author		GW
					\date		08/2014
					
					\returns	The paragraph features object containing the unique local feature values. Other features
								are unassigned. If all features are unassigned, either there are no local overrides, or
								the local overrides have no common values.
				*/
				virtual std::unique_ptr<TextParaFeatures> localParaFeatures() const;
				
				/**
					\brief	Replaces all of the local overrides for all paragraphs in the target text range with a new
							set of feature values. All values that are assigned become local values, replacing any
							previous local value. These local values override any specified in a style associated with
							the paragraph or the text range. All values that are unassigned remove any previous local
							values, so that those values are inherited.
					\author	GW
					\date	08/2014
					
					\param	feat_	The new feature set object
					
					\note	Triggers a reflow operation that can cause previously obtained text runs to become invalid.
				*/
				virtual void setLocalParaFeatures(const TextParaFeatures& feat_);
				
				/**
					\brief	Modifies the local overrides for all paragraphs	in the target text range. All values that
							are assigned become local values, replacing any previous local value. Values that are
							unassigned leave any previous local values unchanged.
					\author	GW
					\date	08/2014
					
					\param	feat_	The new feature set object
					
					\note	Triggers a reflow operation that can cause previously obtained text runs to become invalid.
				*/
				virtual void replaceOrAddLocalParaFeatures(const TextParaFeatures& feat_);
				
				/**
					\brief	Removes all local overrides for all paragraphs in the target text range. All paragraph
							features are then inherited from styles associated with the paragraph or text range, or from
							the normal style.
					\author	GW
					\date	08/2014
				*/
				virtual void clearLocalParaFeatures();
				
				/**
					\brief	Inserts text into the target text range before the current start point
					\author	GW
					\date	12/2013
					
					\param	text_	The text to be inserted, as UTF-8
					
					\note	Triggers a reflow operation that resets the start and end points of the target and any other
							affected ranges to include both the old and new text.
				*/
				virtual void insertBefore(const std::string& text_);
				
				/**
					\brief	Inserts text into the target text range after the current end point
					\author	GW
					\date	12/2013
					
					\param	text_	The text to be inserted, as UTF-8
					
					\note	Triggers a reflow operation that resets the start and end points of the target and any other
							affected ranges to include both the old and new text.
				*/
				virtual void insertAfter(const std::string& text_);
				
				/**
					\brief	Inserts text into the target text range before the current start point
					\author	GW
					\date	12/2013
					
					\param	otherRange_		A text range object containing the text
					
					\note	Triggers a reflow operation that resets the start and end points of the target and any other
							affected ranges to include both the old and new text.
				*/
				virtual void insertBefore(const TextRange& otherRange_);
				
				/**
					\brief	Inserts text into the target text range after the current end point
					\author	GW
					\date	12/2013
					
					\param	otherRange_		A text range object containing the text
					
					\note	Triggers a reflow operation that resets the start and end points of the target and any other
							affected ranges to include both the old and new text.
				*/
				virtual void insertAfter(const TextRange& otherRange_);

				/**
					\brief		Gets the value (contents) of the text range
					\author		GW
					\date		12/2013
					
					\param		le_		Indicates the desired line ending for the output string
					\returns	A string for the value of the text art, as UTF-8
					
					\note		Illustrator forces text art to utilize the "\r" character for line endings; if this
								behavior is unexpected or undesired for your application (as most systems use either
								"\n" or "\r\n" these days), you can specify what you'd prefer with the optional le_
								argument.
				*/
				virtual std::string contents(const std::string& le_ = "\n") const;

				/**
					\brief	Selects the text in the target text range
					\author	GW
					\date	12/2013
					
					\param	addToCurrSelection_		true to add the target's text to the current selection, false to
													clear the current selection before selecting this text
				*/
				virtual void select(const bool addToCurrSelection_ = false);

				/**
					\brief	Deselects the text in the target text range; this can result in a discontiguous selection,
							if this text range is a subset of the selected text
					\author	GW
					\date	12/2013
				*/
				virtual void deselect();
				
				/**
					\brief	Changes the case of the text in the target text range
					\author	GW
					\date	12/2013
					
					\param	type_	Desired case type
				*/
				virtual void changeCase(const CaseChangeType type_);

				/**
					\brief	Adjusts the tracking of the text in the target range to fit on one line spanning the width
							of the area text object
					\author	GW
					\date	12/2013
					
					\note	This is the equivalent of choosing the Type > Fit Headline menu item with the text range
							selected.
				*/
				virtual void fitHeadlines();
				
				/**
					\brief	Deletes all of the characters in the target text range
					\author	GW
					\date	12/2013
					
					\note	Triggers a reflow operation that resets the start and end points of any other affected
							ranges.
				*/
				virtual void remove();


			private:
				friend aip::TextRange* __accessImpl(const TextRange&);
				friend TextRange __accessCtor(const aip::TextRange&);

				/**
					\brief	Private implementation object
				*/
				aip::TextRange* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	12/2013
				*/
				TextRange(const aip::TextRange&);
		};
		
		typedef std::unique_ptr<TextRange> TextRangeUP;
		typedef std::shared_ptr<TextRange> TextRangeSP;
		typedef std::weak_ptr<TextRange> TextRangeWP;
		
		extern aip::TextRange* __accessImpl(const TextRange&);
		extern TextRange __accessCtor(const aip::TextRange&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_RANGE__
