/**
	\file
	\brief		Header file for general Illustrator text range set manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_RANGE_SET__
#define __HDI_CORE_TEXT_RANGE_SET__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextRangeSet;
	}
	
	namespace core
	{
		class TextCharFeatures;
		class TextCharInspector;
		class TextPara;
		class TextParaFeatures;
		class TextParaInspector;
		class TextRange;
		class TextWord;
	
		class TextRangeSet
		{
			public:
				typedef std::vector< std::shared_ptr<TextPara> > TextParaVector;
				typedef std::vector< std::shared_ptr<TextRange> > TextRangeVector;
				typedef std::vector< std::shared_ptr<TextWord> > TextWordVector;
				
				/**
					\brief	Creates a new TextRangeSet object
					\author	GW
					\date	08/2014
					
					\note	New TextRangeSet objects do not relate to any text range set on the Illustrator artboard;
							they are designed to be "receivers" of some other TextRangeSet object via the overloaded
							assignment operator or assigned to some other existent text art.
				*/
				TextRangeSet();
			
				/**
					\brief	TextRangeSet copy constructor
					\author	GW
					\date	08/2014
					
					\param	rs_	TextRangeSet object to copy values from
				*/
				TextRangeSet(const TextRangeSet& rs_);

				/**
					\brief	Destructs a TextRangeSet object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextRangeSet();
			
				/**
					\brief	Overloaded assignment operator for TextRangeSet objects
					\author	GW
					\date	08/2014
				*/
				virtual TextRangeSet& operator=(const TextRangeSet& rhs_);
				
				/**
					\brief		Tests whether a given TextRangeSet object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextRangeSet to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text range set, false otherwise
				*/
				virtual bool operator==(const TextRangeSet& rhs_) const;
				
				/**
					\brief		Tests whether a given TextRangeSet object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextRangeSet to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text range set, false otherwise
				*/
				virtual bool operator!=(const TextRangeSet& rhs_) const;

				/**
					\brief		Gets the ATE range set ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::ITextRangesRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ITextRanges> ateTextRangeSet() const;

				/**
					\brief		Gets whether the target TextRangeSet object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextRangeSet object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Retrieves the number of members of the target set
					\author		GW
					\date		08/2014

					\returns	The number of members, or -1 for error
				*/
				virtual int32_t size() const;
				
				/**
					\brief		Retrieves the selected text within the target text range set
					\author		GW
					\date		08/2014

					\returns	The text ranges object containing the selected text
				*/
				virtual std::unique_ptr<TextRangeSet> selections() const;
				
				/**
					\brief		Retrieves a vector of paragraphs contained in the target text range set
					\author		GW
					\date		08/2014

					\returns	The paragraph vector object
				*/
				virtual TextParaVector paragraphs() const;
				
				/**
					\brief		Retrieves a vector of words contained in the target text range set
					\author		GW
					\date		08/2014

					\returns	The words vector object
				*/
				virtual TextWordVector words() const;
				
				/**
					\brief		Retrieves a vector of runs (ranges) contained in the target text range set
					\author		GW
					\date		08/2014

					\returns	The ranges vector object
				*/
				virtual TextRangeVector runs() const;

				/**
					\brief		Creates an inspector with which to access the character features of all characters in
								the target text range set
					\author		GW
					\date		08/2014

					\returns	The character inspector object
				*/
				virtual std::unique_ptr<TextCharInspector> charInspector() const;

				/**
					\brief		Creates an inspector with which to access the paragraph features of all paragraphs in
								the target text range set
					\author		GW
					\date		08/2014

					\returns	The paragraph inspector object
				*/
				virtual std::unique_ptr<TextParaInspector> paraInspector() const;
				
				/**
					\brief	Removes the association of the target text range set and its character style. Copies the
							feature values of the character style into local override values in the contained characters.
					\author	GW
					\date	08/2014
				*/
				virtual void clearNamedCharStyle();

				/**
					\brief	Removes the association of the target text range set and its paragraph style. Copies the
							feature values of the paragraph style into local override values in the contained paragraphs.
					\author	GW
					\date	08/2014
				*/
				virtual void clearNamedParaStyle();

				/**
					\brief		Retrieves the unique character features used in the target text range set. Unique
								features are those which have the same value in all text runs in the ranges.
					\author		GW
					\date		08/2014

					\returns	The character features object containing the unique feature values
				*/
				virtual std::unique_ptr<TextCharFeatures> uniqueCharFeatures() const;
				
				/**
					\brief		Retrieves the unique paragraph features used in the target text range set. Unique
								features are those which have the same value in all text runs in the ranges.
					\author		GW
					\date		08/2014

					\returns	The paragraph features object containing the unique feature values
				*/
				virtual std::unique_ptr<TextParaFeatures> uniqueParaFeatures() const;

				/**
					\brief		Reports whether there any local character feature overrides for characters contained in
								the target text range set
					\author		GW
					\date		08/2014

					\returns	true if there are local overrides
				*/
				virtual bool hasLocalCharFeatures();

				/**
					\brief		Reports whether there any local paragraph feature overrides for paragraphs contained in
								the target text range set
					\author		GW
					\date		08/2014

					\returns	true if there are local overrides
				*/
				virtual bool hasLocalParaFeatures();

				/**
					\brief		Retrieves the character features that have local overrides in the target text range set,
								and whose local values are the same in all text runs in the ranges
					\author		GW
					\date		08/2014

					\returns	The character features object containing the unique local feature values. Other features
								are unassigned. If all features are unassigned, either there are no local overrides, or
								the local overrides have no common values.
				*/
				virtual std::unique_ptr<TextCharFeatures> localCharFeatures();

				/**
					\brief	Replaces all of the local overrides for all characters in the target text range set with a
							new set of feature values. All values that are assigned become local values, replacing any
							previous local value. These local values override any specified in a style associated with a
							character or a text range. All values that are unassigned remove any previous local values,
							so that those values are inherited.
					\author	GW
					\date	08/2014
					
					\param	feat_	The new feature set object
				*/
				virtual void setLocalCharFeatures(const TextCharFeatures& feat_);
				
				/**
					\brief	Modifies the local overrides for all characters	in the target text range set. All values 
							that are assigned become local values, replacing any previous local value. Values that are
							unassigned leave any previous local values unchanged.
					\author	GW
					\date	08/2014
					
					\param	feat_	The new feature set object
				*/
				virtual void replaceOrAddLocalCharFeatures(const TextCharFeatures& feat_);
				
				/**
					\brief	Removes all local overrides for all characters in the target text range set. All character
							features are then inherited from styles associated with the character or text range, or from
							the Normal style.
					\author	GW
					\date	08/2014
				*/
				virtual void clearLocalCharFeatures();

				/**
					\brief		Retrieves the paragraph features that have local overrides in the target text range set,
								and whose local values are the same in all text runs in the ranges
					\author		GW
					\date		08/2014

					\returns	The paragraph features object containing the unique local feature values. Other features
								are unassigned. If all features are unassigned, either there are no local overrides, or
								the local overrides have no common values.
				*/
				virtual std::unique_ptr<TextParaFeatures> localParaFeatures();
				
				/**
					\brief	Replaces all of the local overrides for all paragraphs in the target text range set with a
							new set of feature values. All values that are assigned become local values, replacing any
							previous local value. These local values override any specified in a style associated with a
							paragraph or a text range. All values that are unassigned remove any previous local values,
							so that those values are inherited.
					\author	GW
					\date	08/2014
					
					\param	feat_	The new feature set object
				*/
				virtual void setLocalParaFeatures(const TextParaFeatures& feat_);
				
				/**
					\brief	Modifies the local overrides for all paragraphs	in the target text range set. All values
							that are assigned become local values, replacing any previous local value. Values that are
							unassigned leave any previous local values unchanged.
					\author	GW
					\date	08/2014
					
					\param	feat_	The new feature set object
				*/
				virtual void replaceOrAddLocalParaFeatures(const TextParaFeatures& feat_);

				/**
					\brief	Removes all local overrides for all paragraphs in this text range set. All paragraph
							features are then inherited from styles associated with the paragraph or text range, or from
							the Normal style.
					\author	GW
					\date	08/2014
				*/
				virtual void clearLocalParaFeatures();

				/**
					\brief	Selects the text in the target text range set
					\author	GW
					\date	08/2014
					
					\param	addToCurrSelection_	true to add this text to the current selection, false to clear the
							current selection before selecting this text
				*/
				virtual void select(const bool addToCurrSelection_ = false);

				/**
					\brief	Deselects the text in the target text range set. This can result in a discontiguous
							selection, if this text is a subset of the selected text.
					\author	GW
					\date	08/2014
				*/
				virtual void deselect();

				/**
					\brief		Gets the value (contents) of the text range
					\author		GW
					\date		12/2013
					
					\param		le_		Indicates the desired line ending for the output string
					\returns	A string for the value of the text art, as UTF-8
					
					\note		Illustrator forces text art to utilize the "\r" character for line endings; if this
								behavior is unexpected or undesired for your application (as most systems use either
								"\n" or "\r\n" these days), you can specify what you'd prefer with the optional le_
								argument.
				*/
				virtual std::string contents(const std::string& le_ = "\n") const;
				
				/**
					\brief	Changes the case of the text in the target text range set
					\author	GW
					\date	08/2014
					
					\param	The case type enum value
				*/
				virtual void changeCase(const CaseChangeType type_);
				
				/**
					\brief	Adds a text range as a member of this set
					\author	GW
					\date	08/2014
					
					\param	range_	The text range object to add
				*/
				virtual void add(const TextRange& range_);

				/**
					\brief		Retrieves a member of this set by position index
					\author		GW
					\date		08/2014

					\param		index_	The 0-based position index
					\returns	The member object
				*/
				virtual std::unique_ptr<TextRange> textRange(const int32_t index_) const;
				
				/**
					\brief	Removes a member from the set
					\author	GW
					\date	08/2014
					
					\param	index_	The 0-based position index of the member to remove
				*/
				virtual void remove(const int32_t index_);
				
				/**
					\brief	Removes all members from the set
					\author	GW
					\date	08/2014
				*/
				virtual void removeAll();


			private:
				friend aip::TextRangeSet* __accessImpl(const TextRangeSet&);
				friend TextRangeSet __accessCtor(const aip::TextRangeSet&);

				/**
					\brief	Private implementation object
				*/
				aip::TextRangeSet* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextRangeSet(const aip::TextRangeSet&);
		};
		
		typedef std::unique_ptr<TextRangeSet> TextRangeSetUP;
		typedef std::shared_ptr<TextRangeSet> TextRangeSetSP;
		typedef std::weak_ptr<TextRangeSet> TextRangeSetWP;
		
		extern aip::TextRangeSet* __accessImpl(const TextRangeSet&);
		extern TextRangeSet __accessCtor(const aip::TextRangeSet&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_RANGE_SET__
