/**
	\file
	\brief		Header file for general Illustrator text story manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_STORY__
#define __HDI_CORE_TEXT_STORY__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextStory;
	}
	
	namespace core
	{
		class TextArt;
		class TextPara;
		class TextRange;
		class TextRangeSet;
		class TextStorySet;
		class TextWord;
	
		class TextStory
		{
			public:
				typedef std::vector< std::shared_ptr<TextArt> > TextArtVector;
				typedef std::vector< std::shared_ptr<TextPara> > TextParaVector;
				typedef std::vector< std::shared_ptr<TextRange> > TextRangeVector;
				typedef std::vector< std::shared_ptr<TextWord> > TextWordVector;
				
				/**
					\brief	Creates a new TextStory object
					\author	GW
					\date	08/2014
					
					\note	Empty TextStory objects do not relate to any text story on the Illustrator artboard; they
							are designed to be "receivers" of some other TextStory object via the overloaded assignment
							operator or assigned to some other existent text art.
				*/
				TextStory();
			
				/**
					\brief	TextStory copy constructor
					\author	GW
					\date	08/2014
					
					\param	s_	TextStory object to copy values from
				*/
				TextStory(const TextStory& s_);

				/**
					\brief	Destructs a TextStory object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextStory();
			
				/**
					\brief	Overloaded assignment operator for TextStory objects
					\author	GW
					\date	08/2014
				*/
				virtual TextStory& operator=(const TextStory& rhs_);
				
				/**
					\brief		Tests whether a given TextStory object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextStory to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text story, false otherwise
				*/
				virtual bool operator==(const TextStory& rhs_) const;
				
				/**
					\brief		Tests whether a given TextStory object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextStory to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text story, false otherwise
				*/
				virtual bool operator!=(const TextStory& rhs_) const;

				/**
					\brief		Gets the ATE story ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::IStoryRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::IStory> ateStory() const;

				/**
					\brief		Gets whether the target TextStory object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextStory object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Duplicates the target object
					\author		GW
					\date		08/2014
					
					\returns	A duplicate of the target object, or NULL for error
				*/
				virtual std::unique_ptr<TextStory> duplicate();

				/**
					\brief		Retrieves the index position of the target story in its collection
					\author		GW
					\date		08/2014
					
					\returns	The 0-based index, or -1 for error
				*/
				virtual int32_t index() const;

				/**
					\brief		Retrieves the number of characters in the target story
					\author		GW
					\date		08/2014
					
					\returns	The number of characters, or -1 for error
				*/
				virtual int32_t charCount() const;

				/**
					\brief		Retrieves the text range that contains the entire text of the target story
					\author		GW
					\date		08/2014
					
					\returns	The text range object
				*/
				virtual std::unique_ptr<TextRange> textRange() const;

				/**
					\brief		Retrieves an arbitrary text range from this story
					\author		GW
					\date		08/2014
					
					\param		start_	The 0-based position index of the first character in the range
					\param		end_	The 0-based position index of the last character in the range
					\returns	The text range object
					
					\note		If end_ is greater than the size of the story, the last character of the story is used.
				*/
				virtual std::unique_ptr<TextRange> textRange(const int32_t start_, const int32_t end_) const;

				/**
					\brief		Retrieves the set of text ranges that contains the selected text of the target story
					\author		GW
					\date		08/2014
					
					\returns	The text range set object
				*/
				virtual std::unique_ptr<TextRangeSet> selections() const;
				
				/**
					\brief		Retrieves the story collection that contains the target story
					\author		GW
					\date		08/2014
					
					\returns	The story collection object
				*/
				virtual std::unique_ptr<TextStorySet> stories() const;

				/**
					\brief		Retrieves a vector of text frames contained in the target story
					\author		GW
					\date		08/2014
					
					\returns	The text frames vector object
				*/
				virtual TextArtVector textFrames() const;

				/**
					\brief		Retrieves a vector of paragraphs contained in the target story
					\author		GW
					\date		08/2014
					
					\returns	The paragraphs vector object
				*/
				virtual TextParaVector paragraphs() const;

				/**
					\brief		Retrieves a vector of words contained in the target story
					\author		GW
					\date		08/2014
					
					\returns	The words vector object
				*/
				virtual TextWordVector words() const;

				/**
					\brief		Retrieves a vector of runs contained in the target story
					\author		GW
					\date		08/2014
					
					\returns	The runs (text ranges) vector object
				*/
				virtual TextRangeVector runs() const;

				/**
					\brief		Retrieves a specific text frame from the target story
					\author		GW
					\date		08/2014
					
					\param		index_	The 0-based position index
					\returns	The text frame object, or NULL for error
				*/
				virtual std::unique_ptr<TextArt> textFrame(const int32_t index_) const;

				/**
					\brief		Retrieves the kerning type and value for a text range in the target story
					\author		GW
					\date		08/2014
					
					\param		range_		The text range object
					\param		type__		Return-by-reference for the kerning type enum value
					\param		kerning__	Return-by-reference for the kerning value
					\returns	true if the kerning could be acquired, false otherwise
				*/
				virtual bool kerningForRange(const TextRange& range_, AutoKernType& type__, int32_t& kerning__) const;
				
				/**
					\brief	Sets automatic or optical kerning for a text range in the target story
					\author	GW
					\date	08/2014
					
					\param	range_	The text range to change
					\param	type_	The kerning type enum value
				*/
				virtual void setAutoKerningForRange(const TextRange& range_, const AutoKernType type_);

				/**
					\brief		Retrieves the kerning type and value for a character in the target story
					\author		GW
					\date		08/2014
					
					\param		index_		The 0-based position index of the character
					\param		type__		Return-by-reference for the kerning type enum value
					\param		kerning__	Return-by-reference for the kerning value
					\returns	true if the kerning could be acquired, false otherwise
				*/
				virtual bool kerningForChar(const int32_t index_, AutoKernType& type__, int32_t& kerning__) const;
				
				/**
					\brief	Sets a specific kern value in the target story
					\author	GW
					\date	08/2014
					
					\param	index_		The 0-based position index of the character to kern
					\param	kerning_	The kerning value
				*/
				virtual void setKerningForChar(const int32_t index_, const int32_t kerning_);
				
				/**
					\brief	Suspends reflow calculation for the target story. Speeds up calls that cause reflow, such as
							TextRange insertion methods
					\author	GW
					\date	08/2014
					
					\note	Use resumeReflow() to restore normal reflow calculation
				*/
				virtual void suspendReflow();

				/**
					\brief	Resumes normal reflow calculation after a call to suspendReflow()
					\author	GW
					\date	08/2014
				*/
				virtual void resumeReflow();


			private:
				friend aip::TextStory* __accessImpl(const TextStory&);
				friend TextStory __accessCtor(const aip::TextStory&);

				/**
					\brief	Private implementation object
				*/
				aip::TextStory* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextStory(const aip::TextStory&);
		};
		
		typedef std::unique_ptr<TextStory> TextStoryUP;
		typedef std::shared_ptr<TextStory> TextStorySP;
		typedef std::weak_ptr<TextStory> TextStoryWP;
		
		extern aip::TextStory* __accessImpl(const TextStory&);
		extern TextStory __accessCtor(const aip::TextStory&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_STORY__
