/**
	\file
	\brief		Header file for general Illustrator tab stop manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_TAB_STOP__
#define __HDI_CORE_TEXT_TAB_STOP__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextTabStop;
	}
	
	namespace core
	{
		class TextTabStop
		{
			public:
				enum TabType
				{
					UnknownTabType	= 0,
					LeftTabType		= 10,
					CenterTabType	= 20,
					RightTabType	= 30,
					DecimalTabType	= 40
				};
			
				/**
					\brief	Creates a new TextTabStop object
					\author	GW
					\date	08/2014
					
					\note	New TextTabStop objects do not relate to any text tab stop on the Illustrator artboard;
							they are designed to be "receivers" of some other TextTabStop object via the overloaded
							assignment operator or assigned to some other existent text art.
				*/
				TextTabStop();
			
				/**
					\brief	TextTabStop copy constructor
					\author	GW
					\date	08/2014
					
					\param	ts_	TextTabStop object to copy values from
				*/
				TextTabStop(const TextTabStop& ts_);

				/**
					\brief	Destructs a TextTabStop object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextTabStop();
			
				/**
					\brief	Overloaded assignment operator for TextTabStop objects
					\author	GW
					\date	08/2014
				*/
				virtual TextTabStop& operator=(const TextTabStop& rhs_);
				
				/**
					\brief		Tests whether a given TextTabStop object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextTabStop to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text tab stop, false otherwise
				*/
				virtual bool operator==(const TextTabStop& rhs_) const;
				
				/**
					\brief		Tests whether a given TextTabStop object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextTabStop to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text tab stops, false otherwise
				*/
				virtual bool operator!=(const TextTabStop& rhs_) const;

				/**
					\brief		Gets the ATE tab stop ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::ITabStopRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ITabStop> ateTabStop() const;

				/**
					\brief		Gets whether the target TextTabStop object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextTabStop object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Retrieves the type of the target's tab stop
					\author		GW
					\date		08/2014

					\returns	The tab type constant
				*/
				virtual TabType tabType() const;

				/**
					\brief	Sets the type of the target's tab stop
					\author	GW
					\date	08/2014

					\param	type_	The tab type constant
				*/
				virtual void setTabType(const TabType type_);

				/**
					\brief		Retrieves the position of the target tab stop
					\author		GW
					\date		08/2014

					\returns	The tab position in document points
				*/
				virtual double position() const;

				/**
					\brief	Sets the position of the target tab stop
					\author	GW
					\date	08/2014

					\param	pos_	The tab position in document points
				*/
				virtual void setPosition(const double pos_);

				/**
					\brief		Reports whether the target tab stop uses a leader
					\author		GW
					\date		08/2014

					\returns	true if the tab uses a leader, false otherwise
				*/
				virtual bool hasLeader() const;

				/**
					\brief		Retrieves the leader string for the target tab stop
					\author		GW
					\date		08/2014

					\returns	The leader contents, as UTF-8
				*/
				virtual std::string leader() const;

				/**
					\brief	Sets the leader string for the target tab stop
					\author	GW
					\date	08/2014

					\param	leader_		The new leader content string, as UTF-8
				*/
				virtual void setLeader(const std::string& leader_);

				/**
					\brief		Retrieves the decimal character for the target tab stop
					\author		GW
					\date		08/2014

					\returns	A string containing the character, as UTF-8
				*/
				virtual std::string decimalChar() const;

				/**
					\brief	Sets the decimal character for the target tab stop
					\author	GW
					\date	08/2014

					\param	char_	A string containing the character, as UTF-8
				*/
				virtual void setDecimalChar(const std::string& char_);


			private:
				friend aip::TextTabStop* __accessImpl(const TextTabStop&);
				friend TextTabStop __accessCtor(const aip::TextTabStop&);

				/**
					\brief	Private implementation object
				*/
				aip::TextTabStop* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextTabStop(const aip::TextTabStop&);
		};
		
		typedef std::unique_ptr<TextTabStop> TextTabStopUP;
		typedef std::shared_ptr<TextTabStop> TextTabStopSP;
		typedef std::weak_ptr<TextTabStop> TextTabStopWP;
		
		extern aip::TextTabStop* __accessImpl(const TextTabStop&);
		extern TextTabStop __accessCtor(const aip::TextTabStop&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_TAB_STOP__
