/**
	\file
	\brief		Header file for general Illustrator tab stop set manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_TEXT_TAB_STOP_SET__
#define __HDI_CORE_TEXT_TAB_STOP_SET__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class TextTabStopSet;
	}
	
	namespace core
	{
		class TextTabStop;

		class TextTabStopSet
		{
			public:
				/**
					\brief	Allows for iteration through the children of a tab stop set, a la std::vector iteration

					\code
						// Here's an example, iterating through the children of someSet:
						for(hdi::core::TextTabStopSet::Iterator iter = someSet.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff to iter.tabStop()
						}
					\endcode
				*/
				class Iterator
				{
					public:
						/**
							\brief	Constructs an Iterator object from an existing Iterator object (copy constructor)
							\author	GW
							\date	08/2014

							\param	iter_	Existing Iterator object
						*/
						Iterator(const Iterator& iter_);
					
						/**
							\brief	Destructs an Iterator object
							\author	GW
							\date	08/2014
						*/
						~Iterator();
						
						/**
							\brief		Assigns one Iterator object to another
							\author		GW
							\date		08/2014

							\param		rhs_	Existing Iterator object to copy values from
							\returns	The target Iterator object, but with its values updated to match that of the
										rhs_ argument
						*/
						Iterator& operator=(const Iterator& iter_);
						
						/**
							\brief		Tests whether a given Iterator object is the same as another
							\author		GW
							\date		08/2014

							\param		rhs_	Iterator to compare against (righthand side of equality operator)
							\returns	true for the target and rhs_ having the same state, false otherwise
						*/
						bool operator==(const Iterator& iter_) const;
						
						/**
							\brief		Tests whether a given Iterator object is not the same as another
							\author		GW
							\date		08/2014

							\param		rhs_	Iterator to compare against (righthand side of inequality operator)
							\returns	true for the target and rhs_ having differing states, false otherwise
						*/
						bool operator!=(const Iterator& iter_) const;

						/**
							\brief		Prefix increment operator for forward iteration
							\author		GW
							\date		08/2014
							
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator++();

						/**
							\brief		Postfix increment operator for forward iteration
							\author		GW
							\date		08/2014
							
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator++(int);

						/**
							\brief		Determines whether iteration should stop; works well as the conditional in a
										for() loop
							\author		GW
							\date		08/2014
							
							\returns	true if at the end of the list of tab stops, false otherwise
						*/
						bool atEnd() const;

						/**
							\brief		Gets the current paragraph style pointed to by the iterator
							\author		GW
							\date		08/2014

							\returns	The current style, or an empty TextTabStop object if at the end (see default
										TextTabStop constructor for more information)
						*/
						std::unique_ptr<TextTabStop> tabStop() const;


					private:
						// Only TextTabStopSet can construct a new Iterator object
						friend class TextTabStopSet;

						/**
							\brief	Private implementation data
						*/
						void* __data;
				
						/**
							\brief	Empty Iterator objects cannot be constructed
							\author	GW
							\date	08/2014
						*/
						Iterator();
					
						/**
							\brief	Creates a new iterator for a TextTabStopSet
							\author	GW
							\date	08/2014

							\param	set_	Set to be iterated through
						*/
						explicit Iterator(const TextTabStopSet& set_);

						/**
							\brief	Not possible to decrement a tab stop iterator
							\author	GW
							\date	08/2014
						*/
						Iterator& operator--();

						/**
							\brief	Not possible to decrement a tab stop iterator
							\author	GW
							\date	08/2014
						*/
						Iterator operator--(int);
				};
		
				typedef std::unique_ptr<Iterator> IteratorUP;
				typedef std::shared_ptr<Iterator> IteratorSP;
				typedef std::weak_ptr<Iterator> IteratorWP;

				/**
					\brief	Creates a new TextTabStopSet object
					\author	GW
					\date	08/2014
					
					\note	New TextTabStopSet objects do not relate to any text tab stop set on the Illustrator
							artboard; they are designed to be "receivers" of some other TextTabStopSet object via the
							overloaded assignment operator or assigned to some other existent text art.
				*/
				TextTabStopSet();
			
				/**
					\brief	TextTabStopSet copy constructor
					\author	GW
					\date	08/2014
					
					\param	tss_	TextTabStopSet object to copy values from
				*/
				TextTabStopSet(const TextTabStopSet& tss_);

				/**
					\brief	Destructs a TextTabStopSet object
					\author	GW
					\date	08/2014
				*/
				virtual ~TextTabStopSet();
			
				/**
					\brief	Overloaded assignment operator for TextTabStopSet objects
					\author	GW
					\date	08/2014
				*/
				virtual TextTabStopSet& operator=(const TextTabStopSet& rhs_);
				
				/**
					\brief		Tests whether a given TextTabStopSet object is the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextTabStopSet to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same text tab stop set, false otherwise
				*/
				virtual bool operator==(const TextTabStopSet& rhs_) const;
				
				/**
					\brief		Tests whether a given TextTabStopSet object is not the same as another
					\author		GW
					\date		08/2014

					\param		rhs_	TextTabStopSet to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different text tab stop sets, false otherwise
				*/
				virtual bool operator!=(const TextTabStopSet& rhs_) const;

				/**
					\brief		Gets the ATE tab stop set ref around which the target object is wrapped
					\author		GW
					\date		08/2014
					
					\returns	ATE::ITabStopsRef for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual std::unique_ptr<ATE::ITabStops> ateTabStopSet() const;

				/**
					\brief		Gets whether the target TextTabStopSet object is empty
					\author		GW
					\date		08/2014
					
					\returns	true if the target TextTabStopSet object is empty, false otherwise
					
					\note		A TextTabStopSet object being empty is not the same as it having a size of 0.
				*/
				virtual bool isEmpty() const;
	
				/**
					\brief		Retrieves the number of members of the target set
					\author		GW
					\date		08/2014

					\returns	The number of members, or -1 is isEmpty() returns true
				*/
				virtual int32_t size() const;

				/**
					\brief		Gets an iterator for the set
					\author		GW
					\date		08/2014
					
					\returns	An Iterator object, for iterating through the target's children
					
					\code
						// Here's an example, iterating through the children of someSet:
						for(hdi::core::TextTabStopSet::Iterator iter = someSet.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff to iter.tabStop()
						}
					\endcode
				*/
				virtual Iterator begin() const;
	
				/**
					\brief	Replaces or adds a tab stop to the target set
					\author	GW
					\date	08/2014

					\param	ts_		The replacement or new tab stop object
				*/
				virtual void replaceOrAdd(const TextTabStop& ts_);
	
				/**
					\brief		Retrieves a member of the target set by position index
					\author		GW
					\date		08/2014

					\param		index_	The 0-based position index
					\returns	The tab stop member object
				*/
				virtual std::unique_ptr<TextTabStop> tabStop(const int32_t index_) const;
	
				/**
					\brief	Removes a member from the set
					\author	GW
					\date	08/2014

					\param	index_	The 0-based position index of the member to remove
				*/
				virtual void remove(const int32_t index_);
	
				/**
					\brief	Removes all members from the set
					\author	GW
					\date	08/2014
				*/
				virtual void removeAll();


			private:
				friend aip::TextTabStopSet* __accessImpl(const TextTabStopSet&);
				friend TextTabStopSet __accessCtor(const aip::TextTabStopSet&);

				/**
					\brief	Private implementation object
				*/
				aip::TextTabStopSet* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2014
				*/
				TextTabStopSet(const aip::TextTabStopSet&);
		};
		
		typedef std::unique_ptr<TextTabStopSet> TextTabStopSetUP;
		typedef std::shared_ptr<TextTabStopSet> TextTabStopSetSP;
		typedef std::weak_ptr<TextTabStopSet> TextTabStopSetWP;
		
		extern aip::TextTabStopSet* __accessImpl(const TextTabStopSet&);
		extern TextTabStopSet __accessCtor(const aip::TextTabStopSet&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_TEXT_TAB_STOP_SET__
