/**
	\file
	\brief		Header file for querying information about a third-party plugin
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_3RD_PARTY_PLUGIN__
#define __HDI_CORE_3RD_PARTY_PLUGIN__

#include "hdicoreTypes.h"

namespace hdi
{
	#if defined(HDI_CORE_AIP_MODE)
		namespace aip
		{
			class ThirdPartyPlugin;
		}
	#elif defined(HDI_CORE_PSP_MODE)
		namespace psp
		{
			class ThirdPartyPlugin;
		}
	#endif

	namespace core
	{
		/**
			\brief	Wraps around a third-party plugin, such that it can be messaged or otherwise manipulated
		*/
		class ThirdPartyPlugin
		{
			public:
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief	Identifies the configuration of various plug-in behaviors
					*/
					enum Options
					{
						NoOptions							= 0,
						AutoSelectsResultsOption			= 1 << 0,
						RetainsPartialPathSelectionOption	= 1 << 1
					};
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief	Constructs an empty ThirdPartyPlugin object
					\author	GW
					\date	04/2014
					
					\note	To test if a ThirdPartyPlugin object is empty, call isEmpty() on it
					\note	Empty ThirdPartyPlugin objects do not relate to any actual plugin; they are designed to be
							"receivers" of some other ThirdPartyPlugin object via the overloaded assignment operator.
							Empty ThirdPartyPlugin objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				ThirdPartyPlugin();
			
				/**
					\brief	Constructs a new ThirdPartyPlugin object from an existing ThirdPartyPlugin object (copy
							constructor)
					\author	GW
					\date	04/2014

					\param	p_	Existing ThirdPartyPlugin object
				*/
				ThirdPartyPlugin(const ThirdPartyPlugin& p_);

				/**
					\brief	Destructs a ThirdPartyPlugin object
					\author	GW
					\date	04/2014
				*/
				virtual ~ThirdPartyPlugin();
			
				/**
					\brief		Assigns one ThirdPartyPlugin object to another
					\author		GW
					\date		04/2014

					\param		rhs_	Existing ThirdPartyPlugin object to copy values from
					\returns	The target ThirdPartyPlugin object, but with its values updated to match that of the
								rhs_ argument
				*/
				virtual ThirdPartyPlugin& operator=(const ThirdPartyPlugin& rhs_);
				
				/**
					\brief		Tests whether a given ThirdPartyPlugin object is the same as another
					\author		GW
					\date		04/2014

					\param		rhs_	ThirdPartyPlugin to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same plugin, false otherwise
				*/
				virtual bool operator==(const ThirdPartyPlugin& rhs_) const;
				
				/**
					\brief		Tests whether a given ThirdPartyPlugin object is not the same as another
					\author		GW
					\date		04/2014

					\param		rhs_	ThirdPartyPlugin to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different plugins, false otherwise
				*/
				virtual bool operator!=(const ThirdPartyPlugin& rhs_) const;
			
				/**
					\brief		Gets the plugin ref around which this object is wrapped
					\author		GW
					\date		04/2014
					
					\returns	SPPluginRef for the target ThirdPartyPlugin object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by the app is not handled by this
								class (or related classes), then it should probably be added to the library.
				*/
				virtual SPPluginRef spPluginRef() const;

				/**
					\brief		Gets whether the target ThirdPartyPlugin object is empty (constructed with the default
								ctor)
					\author		GW
					\date		04/2014
					
					\returns	true if the target ThirdPartyPlugin object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the full path of the plugin file at runtime
					\author		GW
					\date		04/2014
					
					\returns	Plugin file path as UTF-8, e.g. "/Applications/Adobe Illustrator CC/Plug-ins/MyPlugin.aip"
				*/
				virtual std::string filePath() const;

				/**
					\brief		Gets the name of the plugin file at runtime
					\author		GW
					\date		04/2014
					
					\returns	Plugin file name as UTF-8, e.g. "MyPlugin.aip"
				*/
				virtual std::string fileName() const;

				/**
					\brief		Gets the plugin name
					\author		GW
					\date		04/2014
					
					\returns	The plugin name, as UTF-8
				*/
				virtual std::string name() const;

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the plugin's options
						\author		GW
						\date		04/2014
						
						\returns	Current plugin options
					*/
					virtual Options options() const;
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets whether the plugin has started
					\author		GW
					\date		04/2014
					
					\returns	true if the plugin has started, false otherwise
				*/
				virtual bool started() const;

				/**
					\brief		Gets whether the plugin has reported an error condition that makes it unavailable
					\author		GW
					\date		04/2014
					
					\returns	true if the plugin is broken, false otherwise
				*/
				virtual bool broken() const;
				
				/**
					\brief		Initializes a message to be sent to the target plugin with message()
					\author		GW
					\date		04/2014
					
					\param		message__	A reference to an existent SPMessageData object that needs to be initialized
					\returns	true if message__ could be initialized, false otherwise
					
					\note		The message__ argument MUST already be allocated in the app's shared memory using
								the Illustrator::alloc() or Photoshop::alloc() method.
				*/
				virtual bool prepMessageData(SPMessageData& message__) const;
			
				/**
					\brief		Sends a message to the target plugin
					\author		GW
					\date		04/2014
					
					\param		caller_		Caller to send to the plugin
					\param		selector_	Selector for caller_ to send to the plugin
					\param		message_	Message data to send to the plugin. See notes.
					\param		result__	Return-by-reference for the output error from the target plugin
					\returns	Whether an error occurred when attempting to message the plugin
					
					\warning	For Photoshop, this method is only suitable for use with action plugins.
					
					\note		The message_ argument MUST be initialized by the prepMessageData() method, and destroyed
								by the cleanMessageData() method.
					\note		You are not limited in sending merely an SPMessageData object to the plugin; you can
								send any other type of structure, but the first member of said structure MUST be a
								SPMessageData object (so you can legally cast the other structure to SPMessageData*). If
								you choose to go this route, the entire structure instance must be allocated using the
								Illustrator::alloc() or Photoshop::alloc() method.
				*/
				virtual SPErr message(
					const std::string& caller_,
					const std::string& selector_,
					SPMessageData* const message_,
					SPErr& result__
				) const;
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Sends a message to the target plugin
						\author		GW
						\date		12/2017
						
						\param		selID_		Selector ID number to send to the plugin
						\param		message_	Message data to send to the plugin. See notes.
						\param		result__	Return-by-reference for the output error from the target plugin
						\returns	Whether an error occurred when attempting to message the plugin
					
						\warning	This method is only suitable for use with non-action Photoshop plugins.
						
						\note		If messaging an action plugin, the message_ argument MUST be initialized by the
									prepMessageData() method, and destroyed by the cleanMessageData() method.
					*/
					virtual SPErr message(
						const int16_t selID_,
						void* const message_,
						SPErr& result__
					) const;
				#endif
				
				/**
					\brief		Destroys a message that has already been sent to the target plugin with message()
					\author		GW
					\date		04/2014
					
					\param		message__	A reference to an existent SPMessageData object that needs to be cleaned up
					\returns	true if the cleanup was successful, false otherwise
					
					\note		Since you should have allocated the message__ argument using the Illustrator::alloc() or
								Photoshop::alloc() method in the first place (right!?), you should now deallocate it
								using the Illustrator::free() or Photoshop::free() method.
				*/
				virtual bool cleanMessageData(SPMessageData& message__) const;


			private:
				#if defined(HDI_CORE_AIP_MODE)
					friend aip::ThirdPartyPlugin* __accessImpl(const ThirdPartyPlugin&);
					friend ThirdPartyPlugin __accessCtor(aip::ThirdPartyPlugin*&);

					/**
						\brief	Internal use only
						\author	GW
						\date	04/2014
					*/
					ThirdPartyPlugin(aip::ThirdPartyPlugin*&);
				#elif defined(HDI_CORE_PSP_MODE)
					friend psp::ThirdPartyPlugin* __accessImpl(const ThirdPartyPlugin&);
					friend ThirdPartyPlugin __accessCtor(psp::ThirdPartyPlugin*&);

					/**
						\brief	Internal use only
						\author	GW
						\date	04/2014
					*/
					ThirdPartyPlugin(psp::ThirdPartyPlugin*&);
				#endif
					
				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	04/2014
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<ThirdPartyPlugin> ThirdPartyPluginUP;
		typedef std::shared_ptr<ThirdPartyPlugin> ThirdPartyPluginSP;
		typedef std::weak_ptr<ThirdPartyPlugin> ThirdPartyPluginWP;
		
		#if defined(HDI_CORE_AIP_MODE)
			extern aip::ThirdPartyPlugin* __accessImpl(const ThirdPartyPlugin&);
			extern ThirdPartyPlugin __accessCtor(aip::ThirdPartyPlugin*&);
		#elif defined(HDI_CORE_PSP_MODE)
			extern psp::ThirdPartyPlugin* __accessImpl(const ThirdPartyPlugin&);
			extern ThirdPartyPlugin __accessCtor(psp::ThirdPartyPlugin*&);
		#endif
	}
}

#if defined(HDI_CORE_AIP_MODE)
	hdi::core::ThirdPartyPlugin::Options operator|(
		const hdi::core::ThirdPartyPlugin::Options lhs_,
		const hdi::core::ThirdPartyPlugin::Options rhs_
	);

	hdi::core::ThirdPartyPlugin::Options& operator|=(
		hdi::core::ThirdPartyPlugin::Options& lhs__,
		const hdi::core::ThirdPartyPlugin::Options rhs_
	);
#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_3RD_PARTY_PLUGIN__
