/**
	\file
	\brief		Header file for Illustrator array manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_AI_ARRAY__
#define __HDI_CORE_AI_ARRAY__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class IllustratorArray;
	}

	namespace core
	{
		class Art;
		class UID;

		namespace ai
		{
			class Entry;
			class Dictionary;
		
			/**
				\brief	Embodies Illustrator's array storage containers; can be placed inside a dictionary or another array
			*/
			class Array
			{
				public:
					/**
						\brief	Constructs an Array object as a new data container
						\author	GW
						\date	08/2013
					*/
					Array();

					/**
						\brief	Array copy constructor
						\author	GW
						\date	08/2013
						
						\param	a_	Array object to clone
					*/
					Array(const Array& a_);
				
					/**
						\brief	Constructs a new Array object from an AIArrayRef
						\author	GW
						\date	08/2015
						
						\param	aiArray_	Raw Illustrator AIArrayRef
					*/
					Array(const AIArrayRef aiArray_);

					/**
						\brief	Destructs an Array object
						\author	GW
						\date	08/2013
					*/
					virtual ~Array();
			
					/**
						\brief		Overloaded assignment operator for Array objects
						\author		GW
						\date		08/2013

						\param		rhs_	Righthand side of the assignment operator; existing Array object that the
											target should reference instead
						\returns	The lefthand side of the assignment operator, but now referencing the same array
									data as the righthand side
					*/
					virtual Array& operator=(const Array& rhs_);
				
					/**
						\brief		Tests whether a given Array object is the same as another
						\author		GW
						\date		08/2013

						\param		rhs_	Array to compare against (righthand side of equality operator)
						\returns	true for the target and rhs_ being the same array, false otherwise
					*/
					virtual bool operator==(const Array& rhs_) const;
					
					/**
						\brief		Tests whether a given Array object is not the same as another
						\author		GW
						\date		08/2013

						\param		rhs_	Array to compare against (righthand side of inequality operator)
						\returns	true for the target and rhs_ being different arrays, false otherwise
					*/
					virtual bool operator!=(const Array& rhs_) const;

					/**
						\brief		Gets the array ref around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AIArrayRef for the target object

						\note		The caller should not release the returned AIArrayRef (its reference count is not
									increased before being returned).
						\note		Generally, you shouldn't need to call this method. Only use it if you know what
									you're doing. If a specific piece of functionality provided by Illustrator is not
									handled by this class (or related classes), then it should probably be added to the
									hdi_core library.
					*/
					virtual AIArrayRef aiArrayRef() const;

					/**
						\brief		Gets an entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the entry in question
						\param		e__		Return-by-reference for the Entry object representing the array entry at the
											provided index
						\returns	true if the entry can be acquired, false otherwise
					*/
					virtual bool getEntry(const uint32_t index_, Entry& e__) const;

					/**
						\brief		Sets an entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the entry in question
						\param		entry_	Entry to set on the array
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setEntry(const uint32_t index_, const Entry& entry_);

					/**
						\brief		Appends an entry to the end of the array
						\author		GW
						\date		08/2013

						\param		entry_	Entry to append to the array
						\returns	true if the entry was appended, false otherwise
					*/
					virtual bool appendEntry(const Entry& entry_);

					/**
						\brief		Removes an entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the entry to delete
						\returns	true if the entry was deleted, false otherwise
					*/
					virtual bool deleteEntry(const uint32_t index_);

					/**
						\brief		Gets the count of the array elements
						\author		GW
						\date		08/2013
						
						\returns	The count of the elements in the target array
					*/
					virtual uint32_t count() const;

					/**
						\brief		Gets an array entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the array entry
						\param		array__		Return-by-reference for the indicated array entry
						\returns	true if the array entry could be acquired, false otherwise
					*/
					virtual bool getArrayEntry(const uint32_t index_, Array& array__) const;

					/**
						\brief		Sets an array entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the array entry
						\param		value_	New value for the array entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setArrayEntry(const uint32_t index_, const Array& value_);

					/**
						\brief		Gets a dictionary entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the dictionary entry
						\param		dict__		Return-by-reference for the indicated dictionary entry
						\returns	true if the dictionary entry could be acquired, false otherwise
					*/
					virtual bool getDictEntry(const uint32_t index_, Dictionary& dict__) const;

					/**
						\brief		Sets a dictionary entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the dictionary entry
						\param		value_	New value for the dictionary entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setDictEntry(const uint32_t index_, const Dictionary& value_);

					/**
						\brief		Gets a bool entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the bool entry
						\param		value__		Return-by-reference for the indicated bool entry
						\returns	true if the bool entry could be acquired, false otherwise
					*/
					virtual bool getBoolEntry(const uint32_t index_, bool& value__) const;

					/**
						\brief		Sets a bool entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the bool entry
						\param		value_	New value for the bool entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setBoolEntry(const uint32_t index_, const bool value_);

					/**
						\brief		Gets an integer entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the integer entry
						\param		value__		Return-by-reference for the indicated integer entry
						\returns	true if the integer entry could be acquired, false otherwise
					*/
					virtual bool getIntEntry(const uint32_t index_, int64_t& value__) const;

					/**
						\brief		Sets an integer entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the integer entry
						\param		value_	New value for the integer entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setIntEntry(const uint32_t index_, const int64_t value_);

					/**
						\brief		Gets a floating point entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the floating point entry
						\param		value__		Return-by-reference for the indicated floating point entry
						\returns	true if the floating point entry could be acquired, false otherwise
					*/
					virtual bool getFloatEntry(const uint32_t index_, double& value__) const;

					/**
						\brief		Sets a floating point entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the floating-point entry
						\param		value_	New value for the floating-point entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.

						\note		This method compensates for values that are NaN by converting them to 0, and
									compensates for values that are -infinity or infinity by converting them to -FLT_MAX
									or FLT_MAX, respectively.
					*/
					virtual bool setFloatEntry(const uint32_t index_, const double value_);

					/**
						\brief		Gets a string entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the string entry
						\param		value__		Return-by-reference for the indicated UTF-8 string entry
						\returns	true if the string entry could be acquired, false otherwise
					*/
					virtual bool getStringEntry(const uint32_t index_, std::string& value__) const;

					/**
						\brief		Sets a string entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the string entry
						\param		value_	New value for the UTF-8 string entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setStringEntry(const uint32_t index_, const std::string& value_);

					/**
						\brief		Gets a UID entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the UID entry
						\param		uid__		Return-by-reference for the indicated UID entry
						\returns	true if the UID entry could be acquired, false otherwise
					*/
					virtual bool getUIDEntry(const uint32_t index_, UID& uid__) const;

					/**
						\brief		Sets a UID entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the UID entry
						\param		value_	New value for the UID entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setUIDEntry(const uint32_t index_, const UID& value_);

					/**
						\brief		Gets an Art entry from the array by index
						\author		GW
						\date		08/2013

						\param		index_		Index of the Art entry
						\param		art__		Return-by-reference for the indicated Art entry
						\returns	true if the Art entry could be acquired, false otherwise
					*/
					virtual bool getArtEntry(const uint32_t index_, Art& art__) const;

					/**
						\brief		Sets an Art entry in the array by index
						\author		GW
						\date		08/2013

						\param		index_	Index of the Art entry
						\param		value_	New value for the Art entry
						\returns	true if the entry was set, false otherwise

						\pre		An entry at the provided index must already exist in the array; to add a new entry,
									use the appendEntry() method.
					*/
					virtual bool setArtEntry(const uint32_t index_, const Art& value_);

					/**
						\brief		Duplicates the target Array object and the data it's wrapped around
						\author		GW
						\date		08/2013
						
						\returns	An Array object representing a duplicate of the original target array
					*/
					virtual Array duplicate() const;
				
					/**
						\brief		Swaps entries in the target Array object
						\author		GW
						\date		08/2013
						
						\param		index1_		Index of entry to swap with entry at index2_
						\param		index2_		Index of entry to swap with entry at index1_
						\returns	true if the swap was performed
					*/
					virtual bool swapEntries(const uint32_t index1_, const uint32_t index2_);
				
					/**
						\brief		Removes all entries from the array
						\author		GW
						\date		08/2013
						
						\returns	true if the array was cleared, false otherwise
					*/
					virtual bool clear();


				private:
					friend aip::IllustratorArray* __accessImpl(const Array&);
					friend Array __accessCtor(const aip::IllustratorArray&);

					/**
						\brief	Private implementation object
					*/
					aip::IllustratorArray* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Array(const aip::IllustratorArray&);
			};
			
			typedef std::unique_ptr<Array> ArrayUP;
			typedef std::shared_ptr<Array> ArraySP;
			typedef std::weak_ptr<Array> ArrayWP;
			
			extern aip::IllustratorArray* __accessImpl(const Array&);
			extern Array __accessCtor(const aip::IllustratorArray&);
		}
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_AI_ARRAY__
