/**
	\file
	\brief		Header file for Illustrator dictionary manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_AI_DICTIONARY__
#define __HDI_CORE_AI_DICTIONARY__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class IllustratorDictionary;
	}

	namespace core
	{
		class Art;
		class UID;

		namespace ai
		{
			class Array;
			class Entry;
		
			/**
				\brief	Embodies Illustrator's dictionary storage containers; can be placed inside an array, or another
						dictionary
			*/
			class Dictionary
			{
				public:
					/**
						\brief	Allows for iteration through the members of a dictionary, a la std::vector iteration

						\code
							// For example:
							for(hdi::core::ai::Dictionary::Iterator iter = someDict.begin(); !iter.atEnd(); ++iter)
							{
								// Do stuff with either iter.key() or iter.value()
							}
						\endcode
					*/
					class Iterator
					{
						public:
							/**
								\brief	Constructs an Iterator object from an existing Iterator object (copy constructor)
								\author	GW
								\date	10/2013

								\param	iter_	Existing Iterator object
							*/
							Iterator(const Iterator& iter_);

							/**
								\brief	Destructs an Iterator object
								\author	GW
								\date	08/2013
							*/
							~Iterator();
						
							/**
								\brief		Assigns one Iterator object to another
								\author		GW
								\date		10/2013

								\param		rhs_	Existing Iterator object to copy values from
								\returns	The target Iterator object, but with its values updated to match that of the
											rhs_ argument
							*/
							Iterator& operator=(const Iterator& rhs_);
							
							/**
								\brief		Tests whether a given Iterator object is the same as another
								\author		GW
								\date		10/2013

								\param		rhs_	Iterator to compare against (righthand side of equality operator)
								\returns	true for the target and rhs_ having the same state, false otherwise
							*/
							bool operator==(const Iterator& rhs_) const;
							
							/**
								\brief		Tests whether a given Iterator object is not the same as another
								\author		GW
								\date		10/2013

								\param		rhs_	Iterator to compare against (righthand side of inequality operator)
								\returns	true for the target and rhs_ having differing states, false otherwise
							*/
							bool operator!=(const Iterator& rhs_) const;

							/**
								\brief		Prefix increment operator for forward iteration
								\author		GW
								\date		08/2013
								
								\returns	A reference to the iterator after its position was changed
							*/
							Iterator& operator++();

							/**
								\brief		Postfix increment operator for forward iteration
								\author		GW
								\date		08/2013
								
								\returns	A copy of the original iterator before its position was changed
							*/
							Iterator operator++(int);

							/**
								\brief		Prefix decrement operator for reverse iteration
								\author		GW
								\date		08/2013
								
								\returns	A reference to the iterator after its position was changed
							*/
							Iterator& operator--();

							/**
								\brief		Postfix decrement operator for reverse iteration
								\author		GW
								\date		08/2013
								
								\returns	A copy of the original iterator before its position was changed
							*/
							Iterator operator--(int);

							/**
								\brief		Determines whether iteration should stop; works well as the conditional in a
											for() loop
								\author		GW
								\date		08/2013
								
								\returns	true if at the end of the list of members, false otherwise
							*/
							bool atEnd() const;

							/**
								\brief		Gets the key for the current entry pointed to by the iterator (or "" if at
											the end)
								\author		GW
								\date		08/2013
								
								\returns	The current key as UTF-8, or "" if at the end
							*/
							std::string key() const;

							/**
								\brief		Gets the current entry pointed to by the iterator
								\author		GW
								\date		08/2013

								\returns	The current entry, or NULL if at the end
							*/
							std::unique_ptr<Entry> value() const;


						private:
							// Only Dictionary can construct a new Iterator object
							friend class Dictionary;

							/**
								\brief	Private implementation data
							*/
							void* __data;
				
							/**
								\brief	Empty Iterator objects cannot be constructed
								\author	GW
								\date	08/2013
							*/
							Iterator();
						
							/**
								\brief	Creates a new iterator for some target dictionary
								\author	GW
								\date	08/2013

								\param	dict_	Dictionary through which we'll iterate, starting at the beginning
							*/
							explicit Iterator(const Dictionary& dict_);
					};
		
					typedef std::unique_ptr<Iterator> IteratorUP;
					typedef std::shared_ptr<Iterator> IteratorSP;
					typedef std::weak_ptr<Iterator> IteratorWP;

					/**
						\brief	Constructs a Dictionary object as a new data container
						\author	GW
						\date	08/2013
					*/
					Dictionary();

					/**
						\brief	Dictionary copy constructor
						\author	GW
						\date	08/2013
						
						\param	d_	Dictionary object to clone
					*/
					Dictionary(const Dictionary& d_);
				
					/**
						\brief	Constructs a new Dictionary object from an AIDictionaryRef
						\author	GW
						\date	08/2015
						
						\param	aiDict_		Raw Illustrator AIDictionaryRef
					*/
					Dictionary(const AIDictionaryRef aiDict_);

					/**
						\brief	Destructs a Dictionary object
						\author	GW
						\date	08/2013
					*/
					virtual ~Dictionary();
			
					/**
						\brief		Overloaded assignment operator for Dictionary objects
						\author		GW
						\date		08/2013

						\param		rhs_	Righthand side of the assignment operator; existing Dictionary object that
											the target should reference instead
						\returns	The lefthand side of the assignment operator, but now referencing the same dictionary
									data as the righthand side
					*/
					virtual Dictionary& operator=(const Dictionary& rhs_);
				
					/**
						\brief		Tests whether a given Dictionary object is the same as another
						\author		GW
						\date		08/2013

						\param		rhs_	Dictionary to compare against (righthand side of equality operator)
						\returns	true for the target and rhs_ being the same dict, false otherwise
					*/
					virtual bool operator==(const Dictionary& rhs_) const;
					
					/**
						\brief		Tests whether a given Dictionary object is not the same as another
						\author		GW
						\date		08/2013

						\param		rhs_	Dictionary to compare against (righthand side of inequality operator)
						\returns	true for the target and rhs_ being different dicts, false otherwise
					*/
					virtual bool operator!=(const Dictionary& rhs_) const;

					/**
						\brief		Gets the dictionary ref around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AIDictionaryRef for the target object

						\note		The caller should not release the returned AIDictionaryRef (its reference count is
									not increased before being returned).
						\note		Generally, you shouldn't need to call this method. Only use it if you know what
									you're doing. If a specific piece of functionality provided by Illustrator is not
									handled by this class (or related classes), then it should probably be added to the
									hdi_core library.
					*/
					virtual AIDictionaryRef aiDictionaryRef() const;

					/**
						\brief		Gets an entry from the array by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the entry in question, as UTF-8
						\param		entry__		Return-by-reference for an Entry object representing the dictionary
												entry with the provided key name
						\returns	true if the entry could be acquired, false otherwise
					*/
					virtual bool getEntry(const std::string& keyName_, Entry& entry__) const;

					/**
						\brief		Sets an entry in the dictionary by key name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the entry in question, as UTF-8
						\param		entry_		Entry to set on the dictionary
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setEntry(const std::string& keyName_, const Entry& entry_);

					/**
						\brief		Removes an entry in the dictionary by key name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the entry to delete, as UTF-8
						\returns	true if the entry was deleted, false otherwise
					*/
					virtual bool deleteEntry(const std::string& keyName_);

					/**
						\brief		Gets an array entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the array entry, as UTF-8
						\param		array__		Return-by-reference for the indicated array entry
						\returns	true if the array entry could be acquired, false otherwise
					*/
					virtual bool getArrayEntry(const std::string& keyName_, Array& array__) const;

					/**
						\brief		Sets an array entry in the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the array entry, as UTF-8
						\param		array_		New value for the array entry
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setArrayEntry(const std::string& keyName_, const Array& array_);

					/**
						\brief		Gets a bool entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the bool entry, as UTF-8
						\param		value__		Return-by-reference for the indicated bool entry
						\returns	true if the bool entry could be acquired, false otherwise
					*/
					virtual bool getBoolEntry(const std::string& keyName_, bool& value__) const;

					/**
						\brief		Sets a bool entry in the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the bool entry, as UTF-8
						\param		value_		New value for the bool entry
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setBoolEntry(const std::string& keyName_, const bool value_);

					/**
						\brief		Gets a dictionary entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the dictionary entry, as UTF-8
						\param		dict__		Return-by-reference for the indicated dictionary entry
						\returns	true if the dictionary entry could be acquired, false otherwise
					*/
					virtual bool getDictEntry(const std::string& keyName_, Dictionary& dict__) const;

					/**
						\brief		Sets a dictionary entry in the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the dictionary entry, as UTF-8
						\param		dict_		New value for the dictionary entry
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setDictEntry(const std::string& keyName_, const Dictionary& dict_);

					/**
						\brief		Gets an integer entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the integer entry, as UTF-8
						\param		value__		Return-by-reference for the indicated integer entry
						\returns	true if the integer entry could be acquired, false otherwise
					*/
					virtual bool getIntEntry(const std::string& keyName_, int64_t& value__) const;

					/**
						\brief		Sets an integer entry in the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the integer entry, as UTF-8
						\param		value_		New value for the integer entry
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setIntEntry(const std::string& keyName_, const int64_t value_);

					/**
						\brief		Gets a floating point entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the floating point entry, as UTF-8
						\param		value__		Return-by-reference for the indicated floating point entry
						\returns	true if the floating point entry could be acquired, false otherwise
					*/
					virtual bool getFloatEntry(const std::string& keyName_, double& value__) const;

					/**
						\brief		Sets a floating point entry in the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the floating point entry, as UTF-8
						\param		value_		New value for the floating point entry
						\returns	true if the entry was set, false otherwise

						\note		This method compensates for values that are NaN by converting them to 0, and
									compensates for values that are -infinity or infinity by converting them to -FLT_MAX
									or FLT_MAX, respectively.
					*/
					virtual bool setFloatEntry(const std::string& keyName_, const double value_);

					/**
						\brief		Gets a string entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the string entry, as UTF-8
						\param		value__		Return-by-reference for the indicated string entry, as UTF-8
						\returns	true if the string entry could be acquired, false otherwise
					*/
					virtual bool getStringEntry(const std::string& keyName_, std::string& value__) const;

					/**
						\brief		Sets a string entry in the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the string entry, as UTF-8
						\param		value_		New value for the string entry, as UTF-8
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setStringEntry(const std::string& keyName_, const std::string& value_);

					/**
						\brief		Gets a UID entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the UID entry, as UTF-8
						\param		uid__		Return-by-reference for the indicated UID entry
						\returns	true if the UID entry could be acquired, false otherwise
					*/
					virtual bool getUIDEntry(const std::string& keyName_, UID& uid__) const;

					/**
						\brief		Sets a UID entry in the dictionary by key
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the UID entry, as UTF-8
						\param		value_		New value for the UID entry
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setUIDEntry(const std::string& keyName_, const UID& value_);

					/**
						\brief		Gets an Art entry from the dictionary by name
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the Art entry, as UTF-8
						\param		art__		Return-by-reference for the indicated Art entry
						\returns	true if the Art entry could be acquired, false otherwise
					*/
					virtual bool getArtEntry(const std::string& keyName_, Art& art__) const;

					/**
						\brief		Sets an Art entry in the dictionary by key
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the art entry, as UTF-8
						\param		value_		New value for the art entry
						\returns	true if the entry was set, false otherwise
					*/
					virtual bool setArtEntry(const std::string& keyName_, const Art& value_);

					/**
						\brief		Tests whether the dictionary has an entry with a particular key
						\author		GW
						\date		08/2013

						\param		keyName_	Name of the entry in question, as UTF-8
						\returns	true if the dictionary contains the key, false otherwise
					*/
					virtual bool hasKey(const std::string& keyName_) const;
					
					/**
						\brief		Removes all entries from the dictionary
						\author		GW
						\date		08/2013
						
						\returns	true if the dictionary was cleared, false otherwise
					*/
					virtual bool clear();

					/**
						\brief		Gets an iterator for the entries in the dictionary, starting with the first
						\author		GW
						\date		08/2013

						\returns	A new iterator for the target dictionary

						\code
							// For example:
							for(hdi::core::Dictionary::Iterator iter = someDict.begin(); !iter.atEnd(); ++iter)
							{
								// Do stuff with either iter.key() or iter.value()
							}
						\endcode
					*/
					Iterator begin() const;


				private:
					friend aip::IllustratorDictionary* __accessImpl(const Dictionary&);
					friend Dictionary __accessCtor(const aip::IllustratorDictionary&);

					/**
						\brief	Private implementation object
					*/
					aip::IllustratorDictionary* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Dictionary(const aip::IllustratorDictionary&);
			};
			
			typedef std::unique_ptr<Dictionary> DictionaryUP;
			typedef std::shared_ptr<Dictionary> DictionarySP;
			typedef std::weak_ptr<Dictionary> DictionaryWP;
			
			extern aip::IllustratorDictionary* __accessImpl(const Dictionary&);
			extern Dictionary __accessCtor(const aip::IllustratorDictionary&);
		}
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_AI_DICTIONARY__
