/**
	\file
	\brief		Header file for Illustrator entry manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_AI_ENTRY__
#define __HDI_CORE_AI_ENTRY__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class IllustratorEntry;
	}

	namespace core
	{
		class Art;
		class UID;

		namespace ai
		{
			class Array;
			class Dictionary;

			/**
				\brief	Illustrator entry wrapper class to allow for entry creation, manipulation, storage, etc.
			*/
			class Entry
			{
				public:
					/**
						\brief	Describes the type of data an Entry object represents
					*/
					enum Type
					{
						UnknownType				= 0,
						BoolType				= 10,
						IntType					= 20,
						FloatType				= 30,
						StringType				= 40,
						BinaryType				= 50,	// Reserved for future use
						DictionaryType			= 60,
						ArrayType				= 70,
						AngleType				= 80,	// Reserved for future use
						ArtboardPointType		= 90,	// Reserved for future use
						ArtboardRectType		= 100,	// Reserved for future use
						ArtboardSegPointType	= 110,	// Reserved for future use
						ArtColorType			= 120,	// Reserved for future use
						ArtStyleType			= 130,	// Reserved for future use
						BezierType				= 140,	// Reserved for future use
						ColorType				= 150,	// Reserved for future use
						DashStyleType			= 160,	// Reserved for future use
						DashStyleMapType		= 170,	// Reserved for future use
						FillStyleType			= 180,	// Reserved for future use
						FillStyleMapType		= 190,	// Reserved for future use
						FontType				= 200,	// Reserved for future use
						HitDataType				= 210,	// Reserved for future use
						PathStyleType			= 220,	// Reserved for future use
						PathStyleMapType		= 230,	// Reserved for future use
						PointType				= 240,	// Reserved for future use
						RectType				= 250,	// Reserved for future use
						SegPointType			= 260,	// Reserved for future use
						SizeType				= 270,	// Reserved for future use
						StrokeStyleType			= 280,	// Reserved for future use
						StrokeStyleMapType		= 290,	// Reserved for future use
						TransformMatrixType		= 300,	// Reserved for future use
						UIDType					= 310,
						ArtType					= UIDType
					};
			
					/**
						\brief	Constructs an empty Entry object
						\author	GW
						\date	08/2013
						
						\note	To test if an Entry object is empty, call isEmpty() on it
						\note	Empty Entry objects do not relate to any actual data; they are designed to be "receivers"
								of some other Entry object via the overloaded assignment operator. Empty Entry objects
								are useless until such time (though it is safe to call any of their methods)
					*/
					Entry();

					/**
						\brief	Entry copy constructor
						\author	GW
						\date	08/2013
						
						\param	e_	Entry object to clone
					*/
					Entry(const Entry& e_);
				
					/**
						\brief	Constructs a new Entry object from an AIEntryRef
						\author	GW
						\date	08/2015
						
						\param	aiEntry_	Raw Illustrator AIEntryRef
					*/
					Entry(const AIEntryRef aiEntry_);

					/**
						\brief	Destructs an Entry object
						\author	GW
						\date	08/2013
					*/
					virtual ~Entry();
			
					/**
						\brief		Overloaded assignment operator for Entry objects
						\author		GW
						\date		08/2013

						\param		rhs_	Righthand side of the assignment operator; existing Entry object that the
											target should reference instead
						\returns	The lefthand side of the assignment operator, but now referencing the same entry
									data as the righthand side
					*/
					virtual Entry& operator=(const Entry& rhs_);
				
					/**
						\brief		Tests whether a given Entry object is the same as another
						\author		GW
						\date		08/2013

						\param		rhs_	Entry to compare against (righthand side of equality operator)
						\returns	true for the target and rhs_ being the same entry, false otherwise
					*/
					virtual bool operator==(const Entry& rhs_) const;
					
					/**
						\brief		Tests whether a given Entry object is not the same as another
						\author		GW
						\date		08/2013

						\param		rhs_	Entry to compare against (righthand side of inequality operator)
						\returns	true for the target and rhs_ being different entries, false otherwise
					*/
					virtual bool operator!=(const Entry& rhs_) const;

					/**
						\brief		Gets whether the target Entry object is empty (constructed with the default ctor)
						\author		GW
						\date		08/2013
						
						\returns	true if the target Entry object is empty, false otherwise
					*/
					virtual bool isEmpty() const;

					/**
						\brief		Gets the entry ref around which the target object is wrapped
						\author		GW
						\date		11/2013
						
						\returns	AIEntryRef for the target object

						\note		The caller should not release the returned AIEntryRef (its reference count is not
									increased before being returned).
						\note		Generally, you shouldn't need to call this method. Only use it if you know what
									you're doing. If a specific piece of functionality provided by Illustrator is not
									handled by this class (or related classes), then it should probably be added to the
									hdi_core library.
					*/
					virtual AIEntryRef aiEntryRef() const;

					/**
						\brief		Gets the type of the entry
						\author		GW
						\date		08/2013

						\returns	An enum value describing the type of the entry
					*/
					virtual Type entryType() const;

					/**
						\brief		Gets an array value from the entry
						\author		GW
						\date		08/2013
						
						\param		array__		Return-by-reference for the array value the entry is wrapped around
						\returns	true if the entry is of type ArrayType and array__ can be acquired, false otherwise
					*/
					virtual bool asArray(Array& array__) const;

					/**
						\brief		Gets a bool value from the entry
						\author		GW
						\date		08/2013
						
						\param		value__		Return-by-reference for the bool value the entry is wrapped around
						\returns	true if the entry is of type BoolType and value__ can be acquired, false otherwise
					*/
					virtual bool asBool(bool& value__) const;

					/**
						\brief		Gets a dictionary value from the entry
						\author		GW
						\date		08/2013
						
						\param		dict__	Return-by-reference for the dictionary value the entry is wrapped around
						\returns	true if the entry is of type DictionaryType and dict__ can be acquired, false
									otherwise
					*/
					virtual bool asDict(Dictionary& dict__) const;

					/**
						\brief		Gets an integer value from the entry
						\author		GW
						\date		08/2013
						
						\param		value__		Return-by-reference for the integer value the entry is wrapped around
						\returns	true if the entry is of type IntType and value__ can be acquired, false otherwise
					*/
					virtual bool asInt(int64_t& value__) const;

					/**
						\brief		Gets a floating-point value from the entry
						\author		GW
						\date		08/2013
						
						\param		value__		Return-by-reference for the floating point value the entry is wrapped around
						\returns	true if the entry is of type FloatType and value__ can be acquired, false otherwise
					*/
					virtual bool asFloat(double& value__) const;

					/**
						\brief		Gets a string value from the entry
						\author		GW
						\date		08/2013
						
						\param		value__		Return-by-reference for the UTF-8 string value the entry is wrapped around
						\returns	true if the entry is of type StringType and value__ can be acquired, false otherwise
					*/
					virtual bool asString(std::string& value__) const;

					/**
						\brief		Gets a UID value from the entry
						\author		GW
						\date		08/2013
						
						\param		uid__	Return-by-reference for the UID value the entry is wrapped around
						\returns	true if the entry is of type UIDType and uid__ can be acquired, false otherwise
					*/
					virtual bool asUID(UID& uid__) const;

					/**
						\brief		Gets an Art value from the entry
						\author		GW
						\date		08/2013
						
						\param		art__	Return-by-reference for the Art value the entry is wrapped around
						\returns	true if the entry is of type ArtType and art__ can be acquired, false otherwise
					*/
					virtual bool asArt(Art& art__) const;

					/**
						\brief		Statically constructs an Entry object from a given array
						\author		GW
						\date		08/2013

						\param		value_	Array value for the new entry
						\returns	An Entry object that contains the provided Array object
					*/
					static Entry FromArray(const Array& value_);

					/**
						\brief		Statically constructs an Entry object from a given bool
						\author		GW
						\date		08/2013

						\param		value_	bool value for the new entry
						\returns	An Entry object that contains the provided bool value
					*/
					static Entry FromBool(const bool value_);

					/**
						\brief		Statically constructs an Entry object from a given dictionary
						\author		GW
						\date		08/2013

						\param		value_	Dictionary value for the new entry
						\returns	An Entry object that contains the provided Dictionary object
					*/
					static Entry FromDict(const Dictionary& value_);

					/**
						\brief		Statically constructs an Entry object from a given integer
						\author		GW
						\date		08/2013

						\param		value_	Integer value for the new entry
						\returns	An Entry object that contains the provided integer value
					*/
					static Entry FromInt(const int64_t value_);

					/**
						\brief		Statically constructs an Entry object from a given floating point
						\author		GW
						\date		08/2013

						\param		value_	Floating point value for the new entry
						\returns	An Entry object that contains the provided double value
						
						\note		This method compensates for values that are NaN by converting them to 0, and
									compensates for values that are -infinity or infinity by converting them to -FLT_MAX
									or FLT_MAX, respectively.
					*/
					static Entry FromFloat(const double value_);

					/**
						\brief		Statically constructs an Entry object from a given string
						\author		GW
						\date		08/2013

						\param		value_	UTF-8 string value for the new entry
						\returns	An Entry object that contains the provided string value
					*/
					static Entry FromString(const std::string& value_);

					/**
						\brief		Statically constructs an Entry object from a given UID object
						\author		GW
						\date		08/2013

						\param		value_	UID value for the new entry
						\returns	An Entry object that contains the provided UID object
					*/
					static Entry FromUID(const UID& value_);

					/**
						\brief		Statically constructs an Entry object from a given Art object
						\author		GW
						\date		08/2013

						\param		value_	Art value for the new entry
						\returns	An Entry object that contains the provided Art object
					*/
					static Entry FromArt(const Art& value_);


				private:
					friend aip::IllustratorEntry* __accessImpl(const Entry&);
					friend Entry __accessCtor(const aip::IllustratorEntry&);

					/**
						\brief	Private implementation object
					*/
					aip::IllustratorEntry* __impl;

					/**
						\brief	Internal use only
						\author	GW
						\date	08/2013
					*/
					Entry(const aip::IllustratorEntry&);
			};
			
			typedef std::unique_ptr<Entry> EntryUP;
			typedef std::shared_ptr<Entry> EntrySP;
			typedef std::weak_ptr<Entry> EntryWP;
			
			extern aip::IllustratorEntry* __accessImpl(const Entry&);
			extern Entry __accessCtor(const aip::IllustratorEntry&);
		}
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_AI_ENTRY__
