/**
	\file
	\brief		Header file for Illustrator preferences manipulation
	\copyright	Hot Door, Inc. 2010-2025
*/

#ifndef __HDI_CORE_AI_PREFERENCES__
#define __HDI_CORE_AI_PREFERENCES__

#if defined(HDI_CORE_AIP_MODE)

#include <string>

namespace hdi
{
	namespace core
	{
		class Illustrator;
		class Point;
		class Rect;
	
		namespace ai
		{
			/**
				\brief	Allows access to and manipulation of Illustrator app preferences; can be used to store plugin
						preferences as well, but the core::Preferences class is recommended over this one.
			*/
			class Preferences
			{
				public:
					/**
						\brief		Gets a bool pref from the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		b__			Return-by-reference for the bool value
						\returns	true if the bool could be acquired, false otherwise
					*/
					bool getBoolPref(const std::string& prefix_, const std::string& suffix_, bool& b__) const;

					/**
						\brief		Sets a bool pref in the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		b_			bool value to set
						\returns	true if the bool value was set, false otherwise
					*/
					bool setBoolPref(const std::string& prefix_, const std::string& suffix_, const bool b_);

					/**
						\brief		Gets an integer pref from the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		n__			Return-by-reference for the integer value
						\returns	true if the number could be acquired, false otherwise
					*/
					bool getIntPref(const std::string& prefix_, const std::string& suffix_, int32_t& n__) const;

					/**
						\brief		Sets an integer pref in the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		n_			Integer value to set
						\returns	true if the number was set, false otherwise
					*/
					bool setIntPref(const std::string& prefix_, const std::string& suffix_, const int32_t n_);

					/**
						\brief		Gets a floating-point pref from the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		n__			Return-by-reference for the float value
						\returns	true if the number could be acquired, false otherwise
					*/
					bool getFloatPref(const std::string& prefix_, const std::string& suffix_, double& n__) const;

					/**
						\brief		Sets a floating-point pref in the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		n_			Float value to set
						\returns	true if the number was set, false otherwise
					*/
					bool setFloatPref(const std::string& prefix_, const std::string& suffix_, const double n_);

					/**
						\brief		Gets a point pref from the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		p__			Return-by-reference for the point pref
						\returns	true if the point could be acquired, false otherwise
					*/
					bool getPointPref(const std::string& prefix_, const std::string& suffix_, Point& p__) const;

					/**
						\brief		Sets a point pref in the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		p_			Point pref to set
						\returns	true if the point was set, false otherwise
					*/
					bool setPointPref(const std::string& prefix_, const std::string& suffix_, const Point& p_);

					/**
						\brief		Gets a rectangle pref from the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		r__			Return-by-reference for the rectangle pref
						\returns	true if the rectangle could be acquired, false otherwise
					*/
					bool getRectPref(const std::string& prefix_, const std::string& suffix_, Rect& r__) const;

					/**
						\brief		Sets a rectangle pref in the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		r_			Rectangle pref to set
						\returns	true if the rectangle was set, false otherwise
					*/
					bool setRectPref(const std::string& prefix_, const std::string& suffix_, const Rect& r_);

					/**
						\brief		Gets a string pref from the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		str__		Return-by-reference for the string pref, as UTF-8
						\returns	true if the string could be acquired, false otherwise
					*/
					bool getStringPref(const std::string& prefix_, const std::string& suffix_, std::string& str__) const;

					/**
						\brief		Sets a string pref in the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, or "" for an application preference,
												as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\param		str_		String pref to set, as UTF-8
						\returns	true if the string was set, false otherwise
					*/
					bool setStringPref(const std::string& prefix_, const std::string& suffix_, const std::string& str_);

					/**
						\brief		Check if a preference from the Illustrator prefs data exists
						\author		GW
						\date		03/2023

						\param		prefix_		Typically the name of the plug-in, as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\returns	true if the pref could be removed, false otherwise
					*/
					bool prefExists(const std::string& prefix_, const std::string& suffix_) const;

					/**
						\brief		Removes a preference from the Illustrator prefs data tree
						\author		GW
						\date		07/2014

						\param		prefix_		Typically the name of the plug-in, as UTF-8
						\param		suffix_		The name of the preference, as UTF-8
						\returns	true if the pref could be removed, false otherwise

						\warning	Do not attempt to remove a preference that your plugin did not create (or perhaps
									a pref created by another plugin under your control).
					*/
					bool removePref(const std::string& prefix_, const std::string& suffix_);


				private:
					// Only Illustrator can construct a Preferences object
					friend class core::Illustrator;

					/**
						\brief	Private constructor for the Preferences class
						\author	GW
						\date	07/2014
					*/
					Preferences();

					/**
						\brief	Destructs a Preferences object
						\author	GW
						\date	07/2014
					*/
					~Preferences();
			};
		}
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_AI_PREFERENCES__
