#include "panelPlugin.h"

#include <sstream>

#include "hdicoreArt.h"
#include "hdicoreCallback.h"
#include "hdicoreCurrentDocument.h"
#include "hdicoreDispatcher.h"
#include "hdicoreDivider.h"
#include "hdicoreIllustrator.h"
#include "hdicoreLayer.h"
#include "hdicoreNotifier.h"
#include "hdicorePlugin.h"
#include "hdicoreTimer.h"



// MARK: Panel Sample plugin lifecycle

panel::Plugin* panel::Plugin::__instance = NULL;

// static
panel::Plugin* const panel::Plugin::create(hdi::core::Plugin* const corePlug_)
{
	return (panel::Plugin::__instance = new panel::Plugin(corePlug_));
}

// static
panel::Plugin* const panel::Plugin::instance()
{
	return panel::Plugin::__instance;
}

// static
void panel::Plugin::destroy()
{
	delete panel::Plugin::__instance;
	panel::Plugin::__instance = NULL;
}

panel::Plugin::Plugin(hdi::core::Plugin* const corePlug_) : __corePlug(corePlug_), __docSecsSinceSave(0)
{
	this->__corePlug->setName("Hot Door Panel Sample Plugin");
	this->__corePlug->setMenuItemsUpdateCallback(HDI_CORE_CALLBACK(panel::Plugin, this, __updateMenuItemsCB));
}

panel::Plugin::~Plugin()
{
	// This doesn't belong to us, so don't delete it
	this->__corePlug = NULL;
}

void panel::Plugin::startup()
{
	this->__menuGroup = hdi::core::MenuGroup(
		hdi::core::OtherPalettesMenuGroup,	// Containing group
		"PanelSample"						// Menu item text
	);
	
	this->__docPanelMenuItem = hdi::core::MenuItem(
		this->__menuGroup,
		"Show the Current Doc Data panel",
		HDI_CORE_CALLBACK(panel::Plugin, this, __docPanelMenuItemCB)
	);
	
	// We'll never need to change this notifier, so just create it locally (i.e. not as a class member)
	hdi::core::Notifier docChangedNotifier(
		hdi::core::DocumentChangedNotifierType,
		HDI_CORE_CALLBACK(panel::Plugin, this, __docChangedCB)
	);
	this->__corePlug->dispatcher()->registerNotifier(docChangedNotifier);
	
	// We'll never need to change this notifier, so just create it locally (i.e. not as a class member)
	hdi::core::Notifier docSavedNotifier(
		hdi::core::DocumentWritePostprocessNotifierType,
		HDI_CORE_CALLBACK(panel::Plugin, this, __docWasSavedCB)
	);
	this->__corePlug->dispatcher()->registerNotifier(docSavedNotifier);
	
	// We'll never need to change this notifier, so just create it locally (i.e. not as a class member)
	hdi::core::Notifier docClosedNotifier(
		hdi::core::DocumentClosedNotifierType,
		HDI_CORE_CALLBACK(panel::Plugin, this, __docChangedCB)
	);
	this->__corePlug->dispatcher()->registerNotifier(docClosedNotifier);
	
	// We'll never need to change this notifier, so just create it locally (i.e. not as a class member)
	hdi::core::Notifier artboardCreatedNotifier(
		hdi::core::ArtboardCreatedNotifierType,
		HDI_CORE_CALLBACK(panel::Plugin, this, __artboardChangedCB)
	);
	this->__corePlug->dispatcher()->registerNotifier(artboardCreatedNotifier);
	
	// We'll never need to change this notifier, so just create it locally (i.e. not as a class member)
	hdi::core::Notifier artboardDeletedNotifier(
		hdi::core::ArtboardDeletedNotifierType,
		HDI_CORE_CALLBACK(panel::Plugin, this, __artboardChangedCB)
	);
	this->__corePlug->dispatcher()->registerNotifier(artboardDeletedNotifier);
	
	// We'll never need to change this notifier, so just create it locally (i.e. not as a class member)
	hdi::core::Notifier layerListNotifier(
		hdi::core::LayerSetNotifierType,
		HDI_CORE_CALLBACK(panel::Plugin, this, __layerChangedCB)
	);
	this->__corePlug->dispatcher()->registerNotifier(layerListNotifier);
	
	// We'll never need to change this notifier, so just create it locally (i.e. not as a class member)
	hdi::core::Notifier artChangedNotifier(
		hdi::core::ArtSelectionChangedNotifierType,
		HDI_CORE_CALLBACK(panel::Plugin, this, __artChangedCB)
	);
	this->__corePlug->dispatcher()->registerNotifier(artChangedNotifier);

	// We'll never need to change this timer, so just create it locally (i.e. not as a class member)
	hdi::core::Timer docSavedTimer(
		"HDIPanelSampleDocSavedTimer",
		1.0,	// Delay of 1 second
		HDI_CORE_CALLBACK(panel::Plugin, this, __updateSaveTimeCB)
	);
	this->__corePlug->dispatcher()->registerTimer(docSavedTimer);
}

void panel::Plugin::postStartup()
{
	const double marginWidth = 6.0;
	const double marginHeight = 6.0;

	const double fullLabelWidth = hdi::core::Panel::typicalWidth - (2.0 * marginWidth);

	this->__docPanel = hdi::core::Panel(
		"Current Doc Data",										// Panel title
		hdi::core::Size(hdi::core::Panel::typicalWidth, 140.0),	// Panel size
		false,													// Width is not resizeable
		false,													// Height is not resizeable
		hdi::core::ImageIDs()									// Panel has no image IDs
	);

	this->__docPanelNameLabel = hdi::core::Label(
		hdi::core::Point(marginWidth, marginHeight),
		"No current document",
		fullLabelWidth
	);
	this->__docPanelNameLabel.setBold(true);
	this->__docPanel.addWidget(this->__docPanelNameLabel);

	this->__docPanelSaveButton = hdi::core::Button(
		hdi::core::Point(marginWidth, this->__docPanelNameLabel.frame().bottomLeft().y + 2.0),
		"Save",
		hdi::core::PanelWindowType
	);
	this->__docPanelSaveButton.setEnabled(false);
	this->__docPanelSaveButton.setClickCallback(HDI_CORE_CALLBACK(panel::Plugin, this, __saveDocCB));
	this->__docPanel.addWidget(this->__docPanelSaveButton);

	this->__docPanelSaveLabel = hdi::core::Label(
		hdi::core::Point(this->__docPanelSaveButton.frame().topRight().x + 4.0, this->__docPanelSaveButton.frame().topRight().y),
		"(No document to save)",
		fullLabelWidth - this->__docPanelSaveButton.frame().size.width
	);
	this->__docPanel.addWidget(this->__docPanelSaveLabel);

	// We'll never need to change the divider, so just create it locally (i.e. not as a class member)
	hdi::core::Divider div1(
		hdi::core::Point(marginWidth, this->__docPanelSaveButton.frame().bottomLeft().y + marginHeight + 2.0),
		hdi::core::Panel::typicalWidth - (2.0 * marginWidth)
	);
	this->__docPanel.addWidget(div1);

	this->__docPanelArtboardsLabel = hdi::core::Label(
		hdi::core::Point(marginWidth, div1.frame().bottomLeft().y + marginHeight),
		"0 artboards",
		fullLabelWidth
	);
	this->__docPanel.addWidget(this->__docPanelArtboardsLabel);

	this->__docPanelLayersLabel = hdi::core::Label(
		hdi::core::Point(marginWidth, this->__docPanelArtboardsLabel.frame().bottomLeft().y),
		"0 layers",
		fullLabelWidth
	);
	this->__docPanel.addWidget(this->__docPanelLayersLabel);

	this->__docPanelGroupsLabel = hdi::core::Label(
		hdi::core::Point(marginWidth, this->__docPanelLayersLabel.frame().bottomLeft().y),
		"0 groups",
		fullLabelWidth
	);
	this->__docPanel.addWidget(this->__docPanelGroupsLabel);

	this->__docPanelArtsLabel = hdi::core::Label(
		hdi::core::Point(marginWidth, this->__docPanelGroupsLabel.frame().bottomLeft().y),
		"0 art objects",
		fullLabelWidth
	);
	this->__docPanel.addWidget(this->__docPanelArtsLabel);
}

void panel::Plugin::preShutdown()
{

}

void panel::Plugin::shutdown()
{
	this->__docPanel.destroy();
	
	this->__docPanelMenuItem.destroy();
	this->__menuGroup.destroy();
}

void panel::Plugin::__saveDocCB()
{
	hdi::core::CurrentDocument* currDoc = HDI_CORE_ILLUSTRATOR->currentDocument();
	if(!currDoc)
		return;

	currDoc->save();
}

void panel::Plugin::__docWasSavedCB()
{
	this->__docSecsSinceSave = 0;

	hdi::core::CurrentDocument* currDoc = HDI_CORE_ILLUSTRATOR->currentDocument();
	if(currDoc)
	{
		this->__docPanelNameLabel.setText(currDoc->fileName());
	}
}

void panel::Plugin::__updateSaveTimeCB()
{
	this->__docPanelSaveLabel.setText("");
	
	hdi::core::CurrentDocument* currDoc = HDI_CORE_ILLUSTRATOR->currentDocument();
	if(!currDoc)
	{
		this->__docSecsSinceSave = 0;
		this->__docPanelSaveLabel.setText("(No document to save)");
		return;
	}

	++this->__docSecsSinceSave;
	
	if(this->__docSecsSinceSave < 5)
	{
		this->__docPanelSaveLabel.setText("Saved just now");
	}
	else if(this->__docSecsSinceSave < 60)
	{
		std::ostringstream tmp;
		tmp << this->__docSecsSinceSave << " seconds since last save";
		this->__docPanelSaveLabel.setText(tmp.str());
	}
	else
	{
		std::ostringstream tmp;
		tmp << uint32_t(this->__docSecsSinceSave / 60.0) << " minutes since last save";
		this->__docPanelSaveLabel.setText(tmp.str());
	}
}

void panel::Plugin::__docChangedCB()
{
	this->__docPanelSaveButton.setEnabled(false);
	this->__docPanelNameLabel.setText("No current document");

	hdi::core::CurrentDocument* currDoc = HDI_CORE_ILLUSTRATOR->currentDocument();
	if(currDoc)
	{
		this->__docPanelNameLabel.setText(currDoc->fileName());
		this->__docPanelSaveButton.setEnabled(true);
	}
	
	this->__artboardChangedCB();
	this->__layerChangedCB();
	this->__artChangedCB();
}

void panel::Plugin::__artboardChangedCB()
{
	this->__docPanelArtboardsLabel.setText("0 artboards");

	hdi::core::CurrentDocument* currDoc = HDI_CORE_ILLUSTRATOR->currentDocument();
	if(!currDoc)
		return;

	const uint32_t count = currDoc->artboardCount();

	std::ostringstream tmp;
	tmp << count << " artboard" << (count == 1 ? "" : "s");
	this->__docPanelArtboardsLabel.setText(tmp.str());
}

void panel::Plugin::__layerChangedCB()
{
	this->__docPanelLayersLabel.setText("0 layers");

	hdi::core::CurrentDocument* currDoc = HDI_CORE_ILLUSTRATOR->currentDocument();
	if(!currDoc)
		return;

	const uint32_t count = currDoc->layerCount();

	std::ostringstream tmp;
	tmp << count << " layer" << (count == 1 ? "" : "s");
	this->__docPanelLayersLabel.setText(tmp.str());
}

void panel::Plugin::__artChangedCB()
{
	this->__docPanelGroupsLabel.setText("0 groups");
	this->__docPanelArtsLabel.setText("0 art objects");

	hdi::core::CurrentDocument* currDoc = HDI_CORE_ILLUSTRATOR->currentDocument();
	if(!currDoc)
		return;
	
	uint32_t groupCount = 0, artCount = 0;
	std::unique_ptr<hdi::core::Layer> layer0 = currDoc->firstLayer();
	if( !layer0.get() )
		return;
	
	for(hdi::core::Layer::Iterator iter = layer0->begin(); !iter.atEnd(); ++iter)
	{
		std::unique_ptr<hdi::core::Art> tmpLayerGroup = iter.layer()->group();
		if(!tmpLayerGroup.get() || tmpLayerGroup->isEmpty())
			continue;

		this->__countGroupsAndArts(*tmpLayerGroup, groupCount, artCount);
	}
	
	std::ostringstream tmp;
	tmp << groupCount << " group" << (groupCount == 1 ? "" : "s");
	this->__docPanelGroupsLabel.setText(tmp.str());
	
	tmp.str("");
	tmp << artCount << " art object" << (artCount == 1 ? "" : "s");
	this->__docPanelArtsLabel.setText(tmp.str());
}

void panel::Plugin::__updateMenuItemsCB()
{
	if(this->__docPanel.visible())
	{
		this->__docPanelMenuItem.setText("Hide Current Doc Data panel");
	}
	else
	{
		this->__docPanelMenuItem.setText("Show Current Doc Data panel");
	}
}

void panel::Plugin::__docPanelMenuItemCB()
{
	this->__docPanel.setVisible(!this->__docPanel.visible());
}

void panel::Plugin::__countGroupsAndArts(const hdi::core::Art& art_, uint32_t& numGroups__, uint32_t& numArts__)
{
	std::unique_ptr<hdi::core::Art> child0 = art_.firstChild();
	if( !child0.get() )
		return;

	for(hdi::core::Art::Iterator iter = child0->begin(); !iter.atEnd(); ++iter)
	{
		std::unique_ptr<hdi::core::Art> tmpArt = iter.art();
		if(tmpArt.get() && tmpArt->artType() == hdi::core::ArtTypeGroup)
		{
			++numGroups__;
			
			this->__countGroupsAndArts(*tmpArt, numGroups__, numArts__);
			continue;
		}
		
		++numArts__;
	}
}
