/**
	\file
	\brief		Header file for drawing Illustrator annotations during Annotator object callbacks
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_ANNOTATE__
#define __HDI_CORE_ANNOTATE__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreAngle.h"
#include "hdicoreColor.h"
#include "hdicoreRect.h"
#include "hdicoreSegPoint.h"
#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class Font;
		class SegPoint;

		/**
			\brief	Organizes all annotation drawing functions into one namespace, each of which will only perform their
					drawing task(s) if called from within an Annotator callback.
		*/
		namespace annotate
		{
			typedef std::vector<Point> PointVector;
			typedef std::vector<SegPoint> SegPointVector;
		
			/**
				\brief	Describes the type of font to use when annotating text; see also setFontType()
			*/
			enum FontType
			{
				DefaultFont,			// Mac: Lucida Grande Regular, 13pt;	Win: Tahoma Regular, 12pt

				DialogFont,				// Mac: Lucida Grande Regular, 13pt;	Win: Tahoma Regular, 12pt
				BoldDialogFont,			// Mac: Lucida Grande Bold, 13pt;		Win: Tahoma Bold, 12pt
				ItalicDialogFont,		// Mac: Verdana Italic, 12.5pt;			Win: Verdana Italic, 11.5pt
				BoldItalicDialogFont,	// Mac: Verdana Bold Italic, 12.5pt;	Win: Verdana Bold Italic, 11.5pt

				PanelFont,				// Mac: Lucida Grande Regular, 9pt;		Win: Tahoma Regular, 10.5pt
				BoldPanelFont,			// Mac: Lucida Grande Bold, 9pt;		Win: Tahoma Bold, 10.5pt
				ItalicPanelFont,		// Mac: Verdana Italic, 8.5pt;			Win: Verdana Italic, 9.5pt
				BoldItalicPanelFont		// Mac: Verdana Bold Italic, 8.5pt;		Win: Verdana Bold Italic, 9.5pt
			};

			/**
				\brief	A sort of parameter block struct to assist in passing options to text drawing and text 
						calculation functions
			*/
			struct TextConfig
			{
				/**
					\brief	When drawing text, used to specify the justification (can only be JustifyLeft, JustifyCenter,
							or JustifyRight, and defaults to JustifyLeft)
				*/
				ParagraphJustification justification;
				
				/**
					\brief	When drawing text, used to specify the rotation angle (defaults to zero degrees)
				*/
				Angle rotationAngle;
				
				/**
					\brief	When drawing text, used to specify the x/y point of rotation (if rotationAngle is non-zero),
							in a coordinate system relative to the provided top-left drawing point for said text (defaults
							to 0,0) - if this is specified and rotationPin is specified then this will take precedence
				*/
				Point rotationPoint;
				
				/**
					\brief	When drawing text, used to specify the pin point of rotation (if rotationAngle is non-zero,
							defaults to MidCenterPinPoint) - if this is specified and rotationPoint is specified then
							rotationPoint will take precedence
				*/
				BoundsPinPointType rotationPin;
				
				/**
					\brief	Whether the text has any Chinese, Japanese, or Korean characters (which defaults to false
							and will affect the bounds calculation algorithm)
				*/
				bool hasCJK;
				
				TextConfig() :
				justification(JustifyLeft),
				rotationAngle(),
				rotationPoint(),
				rotationPin(MidCenterPinPoint),
				hasCJK(false)
				{}
			};

			/**
				\brief		Draws a line with the given end points
				\author		GW
				\date		08/2013

				\param		start_	Start point for the line
				\param		end_	End point for the line
				\returns	true if the line was drawn, false otherwise
			*/
			bool line(const Point& start_, const Point& end_);

			/**
				\brief		Draws a dashed line with the given end points and dash specs
				\author		GW
				\date		03/2017

				\param		start_	Start point for the line
				\param		end_	End point for the line
				\param		dash_	Length of the dash components
				\param		gap_	Length of the gap components (empty space between dashes)
				\returns	true if the dashed line was drawn, false otherwise
			*/
			bool dashedLine(const Point& start_, const Point& end_, const double dash_, const double gap_);

			/**
				\brief		Draws a rectangle with the given coordinates and fill
				\author		GW
				\date		08/2013
	
				\param		rect_	Rectangle to draw
				\param		fill_	Whether to fill it with the current color
				\returns	true if the rectangle was drawn, false otherwise
			*/
			bool rect(const Rect& rect_, const bool fill_ = false);

			/**
				\brief		Draws a rounded rectangle with the given coordinates, radius, and fill
				\author		GW
				\date		12/2025
	
				\param		rect_	Rectangle to draw
				\param		radius_	Corner radius, in document view points
				\param		fill_	Whether to fill it with the current color
				\returns	true if the rounded rectangle was drawn, false otherwise
			*/
			bool roundedRect(const Rect& rect_, const double radius_, const bool fill_ = false);

			/**
				\brief		Draws an ellipse within the given rectangular bounds
				\author		GW
				\date		08/2013

				\param		bounds_		Rectangle in which the ellipse fits
				\param		fill_		Whether to fill it with the current color
				\returns	true if the line was drawn, false otherwise
			*/
			bool ellipse(const Rect& bounds_, const bool fill_ = false);

			/**
				\brief		Draws a polygon constructed of the given points
				\author		GW
				\date		08/2013

				\param		points_		Vertex points of the polygon
				\param		fill_		Whether to fill it with the current color
				\returns	true if the polygon was drawn, false otherwise
			*/
			bool polygon(const PointVector& points_, const bool fill_ = false);
			
			/**
				\brief		Draws a bezier from the given segment points
				\author		GW
				\date		07/2014
				
				\param		segPts_		Segment points of the bezier path
				\param		fill_		Whether to fill it with the current color
				\returns	true if the bezier was drawn, false otherwise
			*/
			bool bezier(const SegPointVector& segPts_, const bool fill_ = false);

			/**
				\brief		Draws an image at the given point
				\author		GW
				\date		08/2013

				\param		imgID_	Resource ID of the image to draw
				\param		pt_		Point at which to draw the image
				\returns	true if the image was drawn, false otherwise

				\note		At the moment, only PNGI resources are supported
			*/
			bool image(const int16_t imgID_, const Point& pt_);

			/**
				\brief		Draws an image from its raw PNG data at the given point, scaled to the document DPI
				\author		GW
				\date		12/2020
			 
				\param		png_		Raw bytes of PNG data to be drawn
				\param		pngSize_	Length of the buffer passed in png_ argument
				\param		pt_			Point at which to draw the image
				\returns	true if the image was drawn, false otherwise
			*/
			bool image(const byte* const png_, const uint32_t pngSize_, const Point& pt_);

			#if HDI_CORE_COMPILE_AI29_PLUS
				/**
					\brief		Draws an image from its raw ARGB data at the given point, scaled to the document DPI
					\author		GW
					\date		05/2025
				 
					\param		bytes_		Raw bytes of ARGB data to be drawn
					\param		size_		Dimensions of the image in pixels
					\param		pt_			Point at which to draw the image
					\returns	true if the image was drawn, false otherwise
					
					\warning	The length of bytes_ must be (floor(size_.width) * floor(size_.height) * 32)
				*/
				bool image(const byte* const bytes_, const Size& size_, const Point& pt_);
			#endif

			/**
				\brief		Draws an image from its raw PNG data at the given point, unscaled
				\author		GW
				\date		12/2020
			 
				\param		png_		Raw bytes of PNG data to be drawn
				\param		pngSize_	Length of the buffer passed in png_ argument
				\param		pt_			Point at which to draw the image
				\returns	true if the image was drawn, false otherwise
			*/
			bool imageUnscaled(const byte* const png_, const uint32_t pngSize_, const Point& pt_);

			#if HDI_CORE_COMPILE_AI29_PLUS
				/**
					\brief		Draws an image from its raw RGB+alpha data at the given point, unscaled
					\author		GW
					\date		05/2025
				 
					\param		bytes_		Raw bytes of ARGB data to be drawn
					\param		size_		Dimensions of the image in pixels
					\param		pt_			Point at which to draw the image
					\returns	true if the image was drawn, false otherwise
					
					\warning	The length of bytes_ must be (size_.width * size_.height * 32)
				*/
				bool imageUnscaled(const byte* const bytes_, const Size& size_, const Point& pt_);
			#endif

			/**
				\brief		Draws text with the current font and given config
				\author		GW
				\date		08/2013

				\param		text_		UTF-8 text to draw
				\param		pt_			Top-left point at which to draw the text (before any of config_ arg is applied)
				\param		config_		The configuration for other/advanced drawing options; pass in a default-
										constructed object to draw without any options applied
				\returns	true if the text was drawn, false otherwise
			*/
			bool text(const std::string& text_, const Point& pt_, const TextConfig& config_ = TextConfig());

			/**
				\brief		Draws horizontal text with the current font and left justification
				\author		GW
				\date		05/2025

				\param		text_		UTF-8 text to draw
				\param		pt_			Point at which to draw the baseline of the first line of text
				\returns	true if the text was drawn, false otherwise
			*/
			bool textAtBaseline(const std::string& text_, const Point& pt_);

			/**
				\brief		Gets the state of antialiased drawing for the current annotation context
				\author		GW
				\date		09/2025
				
				\returns	true if antialiased drawing is enabled, false otherwise
			*/
			bool antialias();

			/**
				\brief	Sets the state of antialiased drawing for the current annotation context
				\author	GW
				\date	09/2025
				
				\param	aa_		true to enable antialiased drawing, false otherwise
			*/
			void setAntialias(const bool aa_);

			/**
				\brief		Gets the current annotation color
				\author		GW
				\date		08/2013

				\returns	Color for the current annotation context
			*/
			Color color();

			/**
				\brief	Sets the current annotation color
				\author	GW
				\date	08/2013

				\param	color_	New color for annotation art
			*/
			void setColor(const Color& color_);

			/**
				\brief		Gets the line width used when drawing paths or unfilled objects
				\author		GW
				\date		07/2014
				
				\returns	The line width in document view points
			*/
			double lineWidth();
			
			/**
				\brief	Sets the line width used when drawing paths or unfilled objects
				\author	GW
				\date	07/2014
				
				\param	width_	The line width in document view points
			*/
			void setLineWidth(const double width_);
			
			/**
				\brief	Sets the current font for annotation text art
				\author	GW
				\date	08/2013

				\param	font_	New font for annotating
			*/
			void setFont(const Font& font_);

			/**
				\brief	Sets the current font size for annotation text art
				\author	GW
				\date	02/2014

				\param	size_	New font size for the current annotation context
			*/
			void setFontSize(const double size_);

			/**
				\brief		Gets the current font for annotation text art
				\author		GW
				\date		08/2013

				\returns	Font for the current annotation context
			*/
			std::unique_ptr<Font> font();

			/**
				\brief		Gets the current font size for annotation text art
				\author		GW
				\date		02/2014

				\returns	The font size for the current annotation context
			*/
			double fontSize();

			/**
				\brief	Sets the current font for annotation text art
				\author	GW
				\date	08/2013

				\param	font_	New font for annotating
			*/
			void setFontType(const FontType font_);

			/**
				\brief		Gets the bounds for the given text at the origin
				\author		GW
				\date		08/2013

				\param		text_		UTF-8 text for which bounds are needed
				\param		config_		The configuration for other/advanced drawing options; pass in a default-
										constructed object to calculate without any options applied
				\returns	The calculated bounds
			*/
			Size textSize(const std::string& text_, const TextConfig& config_ = TextConfig());

			/**
				\brief	Loops through all views of the current document and invalidates their annotations
				\author	GW
				\date	08/2013
			*/
			void invalidateAll();
		}
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ANNOTATE__
