/**
	\file
	\brief		Header file for Illustrator annotator manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_ANNOTATOR__
#define __HDI_CORE_ANNOTATOR__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace aip
	{
		class Annotator;
	}

	namespace core
	{
		class Callback;
	
		/**
			\brief	Deals with annotator registration and drawing callbacks
		*/
		class Annotator
		{
			public:
				/**
					\brief	Options for how an annotator should behave
				*/
				enum Options
				{
					DrawBeforeAllOthers		= 1 << 0,
					DrawOnInactiveDocuments	= 1 << 1,
					DontDrawOnInactiveViews	= 1 << 2
				};
				
				/**
					\brief	Constructs an empty Annotator object
					\author	GW
					\date	08/2013
					
					\note	To test if an Annotator object is empty, call isEmpty() on it
					\note	Empty Annotator objects do not relate to any Illustrator annotator; they are designed to be
							"receivers" of some other Annotator object via the overloaded assignment operator. Empty
							Annotator objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				Annotator();
			
				/**
					\brief	Constructs a new Annotator object from an existing Annotator object (copy constructor)
					\author	GW
					\date	08/2013

					\param	a_	Existing Annotator object
				*/
				Annotator(const Annotator& a_);
			
				/**
					\brief	Constructs an Annotator object, with a unique name and callback
					\author	GW
					\date	08/2013

					\param	name_		Unique string to identify the annotator, as UTF-8
					\param	responder_	Callback to perform when notified by Illustrator that it's time to annotate
				*/
				Annotator(const std::string& name_, const Callback& responder_);

				/**
					\brief	Annotator destructor; deactivates the annotator and deletes the registered callback
					\author	GW
					\date	08/2013
				*/
				virtual ~Annotator();
			
				/**
					\brief		Assigns one Annotator object to another
					\author		GW
					\date		08/2013

					\param		rhs_	Existing Annotator object to copy values from
					\returns	The target Annotator object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual Annotator& operator=(const Annotator& rhs_);
				
				/**
					\brief		Tests whether a given Annotator object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Annotator to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same annotator, false otherwise
				*/
				virtual bool operator==(const Annotator& rhs_) const;
				
				/**
					\brief		Tests whether a given Annotator object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Annotator to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different annotators, false otherwise
				*/
				virtual bool operator!=(const Annotator& rhs_) const;

				/**
					\brief		Gets whether the target Annotator object is empty (constructed with the default ctor)
					\author		GW
					\date		08/2013
					
					\returns	true if the target Annotator object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief	Destroys the annotator, converting the target object to an empty Annotator object (see
							the default constructor for more info on empty Annotator objects)
					\author	GW
					\date	03/2014

					\pre	If you have registered the annotator with Dispatcher::registerAnnotator(), then be sure to
							unregister the annotator before calling the destroy() method.

					\note	This method must be called for any existing annotators by the time the plugin is shutting
							down. If you no longer retain any of your own references to an annotator then the lib will
							call this method for you as it is shutting down.
				*/
				virtual void destroy();

				/**
					\brief		Gets the annotator handle around which the target object is wrapped
					\author		GW
					\date		11/2013
					
					\returns	AIAnnotatorHandle for the target object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIAnnotatorHandle aiAnnotatorHandle() const;

				/**
					\brief		Gets the annotator's unique name
					\author		GW
					\date		08/2013
					
					\returns	Name of the annotator, as UTF-8
				*/
				virtual std::string name() const;

				/**
					\brief		Gets whether the annotator is active
					\author		GW
					\date		08/2013
					
					\returns	true if the annotator is enabled, false otherwise
				*/
				virtual bool active() const;

				/**
					\brief	Sets whether the annotator is active
					\author	GW
					\date	08/2013

					\param	active_		true to enable the annotator, false to disable

					\note	This method simply calls activate() or deactivate() as appropriate
				*/
				virtual void setActive(const bool active_);

				/**
					\brief	Activates (enables) the annotator
					\author	GW
					\date	08/2013
				*/
				virtual void activate();

				/**
					\brief	Deactivates (disables) the annotator
					\author	GW
					\date	08/2013
				*/
				virtual void deactivate();

				/**
					\brief		Gets the options that have been applied to the target annotator
					\author		GW
					\date		08/2013
					
					\returns	Bitwise OR of options for this annotator
				*/
				virtual Options options() const;

				/**
					\brief	Sets the options for the target annotator
					\author	GW
					\date	08/2013

					\param	flags_	New bitwise OR of annotator options
				*/
				virtual void setOptions(const Options flags_);

				/**
					\brief		Gets the registered drawing callback for the annotator
					\author		GW
					\date		08/2013

					\returns	Currently registered annotator callback
				*/
				virtual Callback* const callback() const;
				
				/**
					\brief	Sets the callback for the annotator
					\author	GW
					\date	08/2013

					\param	cb_		New callback to be performed when the annotator needs to draw
				*/
				virtual void setCallback(const Callback& cb_);


			private:
				friend aip::Annotator* __accessImpl(const Annotator&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<Annotator> AnnotatorUP;
		typedef std::shared_ptr<Annotator> AnnotatorSP;
		typedef std::weak_ptr<Annotator> AnnotatorWP;
		
		extern aip::Annotator* __accessImpl(const Annotator&);
	}
}

hdi::core::Annotator::Options operator|(
	const hdi::core::Annotator::Options lhs_,
	const hdi::core::Annotator::Options rhs_
);

hdi::core::Annotator::Options& operator|=(
	hdi::core::Annotator::Options& lhs__,
	const hdi::core::Annotator::Options rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ANNOTATOR__
