/**
	\file
	\brief		Header file for array preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_ARRAY_PREF__
#define __HDI_CORE_ARRAY_PREF__

#include <vector>

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		class Color;
		class Font;
		class Point;
		class Rect;
		class Size;

		class BoolPref;
		class ColorPref;
		class DictionaryPref;
		class FontPref;
		class NumberPref;
		class PointPref;
		class RectPref;
		class SizePref;
		class StringPref;
		
		#if defined(HDI_CORE_AIP_MODE)
			class ArtboardPoint;
			class ArtboardRect;
			class ArtColor;
			class TransformMatrix;
		
			class ArtboardPointPref;
			class ArtboardRectPref;
			class ArtColorPref;
			class DashStylePref;
			class DashStyleMapPref;
			class FillStylePref;
			class FillStyleMapPref;
			class PathStylePref;
			class PathStyleMapPref;
			class StrokeStylePref;
			class StrokeStyleMapPref;
			class TransformMatrixPref;
		#endif
	
		/**
			\brief	Acts as an array-style container to store any other persistent data types, including arrays and
					dictionaries, and can itself be stored inside another container type.
		*/
		class ArrayPref : public PrefData
		{
			public:
				typedef std::vector< std::shared_ptr<PrefData> > PrefDataVector;
			
				/**
					\brief	Constructs a new ArrayPref object from an existing ArrayPref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	array_	Existing ArrayPref object
				*/
				ArrayPref(const ArrayPref& array_);

				/**
					\brief		Constructs an ArrayPref object, with new data, to be added to a container at a later time
					\author		GW
					\date		09/2013

					\param		elements_	Initial array elements, if any
					\param		name_		Name for the data array once it has been added to its container, as UTF-8

					\note		The name_ argument can only be "" (omitted) if the object will be added directly to
								another ArrayPref object.
				*/
				ArrayPref(const PrefDataVector* elements_ = NULL, const std::string& name_ = "");

				/**
					\brief	Destructs an ArrayPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~ArrayPref();

				/**
					\brief		Allows one ArrayPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ArrayPref object, but with its value updated to match that of rhs_
				*/
				virtual ArrayPref& operator=(const ArrayPref& rhs_);
				
				/**
					\brief		Convenience method to clone an ArrayPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ArrayPref object
					
					\note		If you subclass ArrayPref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ArrayPref subclass, you will
								experience "object slicing" when adding the pref to a container.
								
					\warning	The caller must manage the memory for the returned ArrayPref object.
				*/
				virtual ArrayPref* clone() const;

				/**
					\brief		Gets the count of elements in the top-level of the array
					\author		GW
					\date		09/2013

					\returns	Number of elements in the array
				*/
				virtual uint32_t count() const;

				/**
					\brief		Gets a single element object from the array data tree
					\author		GW
					\date		09/2013

					\param		index_	Index of the element to get from the top-level of the array data tree
					\param		data__	Return-by-reference for the element found at the given index
					\returns	true if the element could be acquired, false otherwise

					\note		The type of the returned object can be determined from its dataType() method, such that
								correct casting can be accomplished.
				*/
				virtual bool element(const uint32_t index_, std::unique_ptr<PrefData>& data__) const;

				/**
					\brief		Adds a persistent data object to the end of the array
					\author		GW
					\date		09/2013

					\param		data_	New data element to add to the end of the array
					\returns	true if the element was added, or false otherwise
				*/
				virtual bool pushElement(const PrefData& data_);

				/**
					\brief		Removes a persistent data object from the end of the array
					\author		GW
					\date		09/2013

					\param		data__	Return-by-reference for the PrefData object for the element that was removed
					\returns	true if the element could be popped and acquired, false otherwise

					\note		The type of the returned object can be determined from its dataType() method, such that
								correct casting can be accomplished.
				*/
				virtual bool popElement(std::unique_ptr<PrefData>& data__);

				/**
					\brief		Inserts a persistent data object at an arbitrary position in the array
					\author		GW
					\date		09/2013

					\param		index_	Desired index for the new data element
					\param		data_	New data element to insert in the array
					\returns	true if the element was inserted successfully, or false otherwise

					\note		Index_ must be in the range [0,n], where n is the count of objects already in the array.
				*/
				virtual bool insertElement(const uint32_t index_, const PrefData& data_);

				/**
					\brief		Removes a persistent data object from an arbitrary position in the array
					\author		GW
					\date		09/2013

					\param		index_	Index of the desired data element to be removed
					\param		data__	Return-by-reference for the PrefData object for the element that was removed
					\returns	true if the element could be removed and acquired, false otherwise

					\note		The type of the returned object can be determined from its dataType() method, such that
								correct casting can be accomplished.
					\note		Index_ must be in the range [0,n], where n is the count of objects already in the array.
					\note		All elements located after index_ will have their index decremented
				*/
				virtual bool removeElement(const uint32_t index_, std::unique_ptr<PrefData>& data__);
				
				/**
					\brief	Removes all elements from an array
					\author	GW
					\date	09/2013
				*/
				virtual void clear();
				
				/**
					\brief		Gets a persistent array object from the containing array
					\author		GW
					\date		09/2013

					\param		index_		Index of the array JSON node to get from the top-level and return as a
											neatly-packed persistent data type
					\param		array__		Return-by-reference for the array element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getArrayElement(const uint32_t index_, ArrayPref& array__) const;
				
				/**
					\brief		Similar to getArrayElement(), except that, if the element does not already exist, it
								will be created anew with a series of default elements
					\author		GW
					\date		09/2013

					\param		index_		Index of the array JSON node to get from the top-level and return
					\param		elements_	Default elements for the array, in case it didn't already exist (NULL, or
											an empty vector, for none)
					\param		array__		Return-by-reference for the array element located at index_ (or created anew)
					\returns	true if the element could be acquired or created, false otherwise

					\note		The order of the default elements in the array will match that of the passed elements_
								argument.
				*/
				virtual bool getOrCreateArrayElement(
					const uint32_t index_,
					const PrefDataVector* elements_,
					ArrayPref& array__
				);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent point object from the array
						\author		GW
						\date		09/2013
						
						\param		index_	Index of the artboard point JSON node to get from the top-level and return as a
											neatly-packed persistent data type
						\param		abp__	Return-by-reference for the artboard point element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getArtboardPointElement(const uint32_t index_, ArtboardPointPref& abp__) const;

					/**
						\brief		Similar to getArtboardPointElement(), except that, if the element does not already exist,
									it will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		index_		Index of the artboard point JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		abp__		Return-by-reference for the artboard point element located at index_ (or
												created anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtboardPointElement(
						const uint32_t index_,
						const ArtboardPoint& default_,
						ArtboardPointPref& abp__
					);

					/**
						\brief		Gets a persistent artboard rect object from the array
						\author		GW
						\date		09/2013
						
						\param		index_	Index of the artboard rect JSON node to get from the top-level and return as a
											neatly-packed persistent data type
						\param		abr__	Return-by-reference for the artboard rect element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getArtboardRectElement(const uint32_t index_, ArtboardRectPref& abr__) const;

					/**
						\brief		Similar to getArtboardRectElement(), except that, if the element does not already exist,
									it will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		index_		Index of the artboard rect JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		abr__		Return-by-reference for the artboard rect element located at index_ (or
												created anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtboardRectElement(
						const uint32_t index_,
						const ArtboardRect& default_,
						ArtboardRectPref& abr__
					);

					/**
						\brief		Gets a persistent art color object from the array
						\author		GW
						\date		09/2013
						
						\param		index_	Index of the art color JSON node to get from the top-level and return as a neatly-
											packed persistent data type
						\param		ac__	Return-by-reference for the art color element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getArtColorElement(const uint32_t index_, ArtColorPref& ac__) const;

					/**
						\brief		Similar to getArtColorElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		index_		Index of the art color JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		ac__		Return-by-reference for the art color element located at index_ (or created
												anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtColorElement(
						const uint32_t index_,
						const ArtColor& default_,
						ArtColorPref& ac__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent bool object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the bool JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		b__		Return-by-reference for the bool element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getBoolElement(const uint32_t index_, BoolPref& b__) const;

				/**
					\brief		Similar to getBoolElement(), except that, if the element does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the bool JSON node to get from the top-level and return
					\param		default_	Default value for the element, in case it didn't already exist
					\param		b__			Return-by-reference for the bool element located at index_ (or created anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateBoolElement(const uint32_t index_, const bool default_, BoolPref& b__);

				/**
					\brief		Gets a persistent color object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the color JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		c__		Return-by-reference for the color element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getColorElement(const uint32_t index_, ColorPref& c__) const;

				/**
					\brief		Similar to getColorElement(), except that, if the element does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the color JSON node to get from the top-level and return
					\param		default_	Default value for the element, in case it didn't already exist
					\param		c__			Return-by-reference for the color element located at index_ (or created anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateColorElement(const uint32_t index_, const Color& default_, ColorPref& c__);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent dash style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the dash style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the dash style element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getDashStyleElement(const uint32_t index_, DashStylePref& ds__);
				
					/**
						\brief		Similar to getDashStyleElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the dash style JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		ds__		Return-by-reference for the dash style element located at index_ (or created
												anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateDashStyleElement(
						const uint32_t index_,
						const DashStyle& default_,
						DashStylePref& ds__
					);
				
					/**
						\brief		Gets a persistent dash style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the dash style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the dash style map member located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getDashStyleMapElement(const uint32_t index_, DashStyleMapPref& ds__);
				
					/**
						\brief		Similar to getDashStyleMapElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the dash style map JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		ds__		Return-by-reference for the dash style map member located at index_ (or
												created anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateDashStyleMapElement(
						const uint32_t index_,
						const DashStyle::Map& default_,
						DashStyleMapPref& ds__
					);
				#endif
				// HDI_CORE_AIP_MODE
				
				/**
					\brief		Gets a persistent dictionary object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the dictionary JSON node to get from the top-level and return as a
										neatly-packed persistent data type
					\param		dict__	Return-by-reference for the dictionary element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getDictElement(const uint32_t index_, DictionaryPref& dict__) const;
				
				/**
					\brief		Similar to getDictElement(), except that, if the element does not already exist, it will
								be created anew with a series of default members
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the dictionary JSON node to get from the top-level and return
					\param		members_	Default members for the dictionary, in case it didn't already exist (NULL,
											or an empty vector, for none)
					\param		dict__		Return-by-reference for the dictionary element located at index_ (or created
											anew)
					\returns	true if the element could be acquired or created, false otherwise

					\note		The names of the elements in the dictionary will match that of the individual members in
								the passed members_ argument.
				*/
				virtual bool getOrCreateDictElement(
					const uint32_t index_,
					const PrefDataVector* members_,
					DictionaryPref& dict__
				);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent fill style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the fill style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		fs__	Return-by-reference for the fill style element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getFillStyleElement(const uint32_t index_, FillStylePref& fs__);
				
					/**
						\brief		Similar to getFillStyleElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the fill style JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		fs__		Return-by-reference for the fill style element located at index_ (or created
												anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateFillStyleElement(
						const uint32_t index_,
						const FillStyle& default_,
						FillStylePref& fs__
					);
				
					/**
						\brief		Gets a persistent fill style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the fill style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		fs__	Return-by-reference for the fill style map member located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getFillStyleMapElement(const uint32_t index_, FillStyleMapPref& fs__);
				
					/**
						\brief		Similar to getFillStyleMapElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the fill style map JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		fs__		Return-by-reference for the fill style map member located at index_ (or
												created anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateFillStyleMapElement(
						const uint32_t index_,
						const FillStyle::Map& default_,
						FillStyleMapPref& fs__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent font object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the font JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		font__	Return-by-reference for the font element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getFontElement(const uint32_t index_, FontPref& font__) const;

				/**
					\brief		Similar to getFontElement(), except that, if the element does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_			Index of the font JSON node to get from the top-level and return
					\param		defaultFont_	Default font for the element, in case it didn't already exist
					\param		defaultSize_	Default font size for the element, in case it didn't already exist
					\param		font__			Return-by-reference for the font element located at index_ (or created anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateFontElement(
					const uint32_t index_,
					const Font& defaultFont_,
					const double defaultSize_,
					FontPref& font__
				);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent matrix object from the array
						\author		GW
						\date		09/2013
						
						\param		index_	Index of the matrix JSON node to get from the top-level and return as a neatly-
											packed persistent data type
						\param		tm__	Return-by-reference for the transformation matrix element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getTransformMatrixElement(const uint32_t index_, TransformMatrixPref& tm__) const;

					/**
						\brief		Similar to getTransformMatrixElement(), except that, if the element does not already
									exist, it will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		index_		Index of the matrix JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		tm__		Return-by-reference for the transformation matrix element located at index_
												(or created anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateTransformMatrixElement(
						const uint32_t index_,
						const TransformMatrix& default_,
						TransformMatrixPref& tm__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent number object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the number JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		n__		Return-by-reference for the number element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getNumberElement(const uint32_t index_, NumberPref& n__) const;

				/**
					\brief		Similar to getNumberElement(), except that, if the element does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the number JSON node to get from the top-level and return
					\param		default_	Default integer value for the element, in case it didn't already exist
					\param		n__			Return-by-reference for the number element located at index_ (or created
											anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateIntElement(const uint32_t index_, const int64_t default_, NumberPref& n__);

				/**
					\brief		Similar to getNumberElement(), except that, if the element does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the number JSON node to get from the top-level and return
					\param		default_	Default double value for the element, in case it didn't already exist
					\param		n__			Return-by-reference for the number element located at index_ (or created
											anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateFloatElement(const uint32_t index_, const double default_, NumberPref& n__);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent path style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the path style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		ps__	Return-by-reference for the path style element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getPathStyleElement(const uint32_t index_, PathStylePref& ps__);
				
					/**
						\brief		Similar to getPathStyleElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the path style JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		ps__		Return-by-reference for the path style element located at index_ (or created
												anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreatePathStyleElement(
						const uint32_t index_,
						const PathStyle& default_,
						PathStylePref& ps__
					);
				
					/**
						\brief		Gets a persistent path style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the path style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		ps__	Return-by-reference for the path style map member located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getPathStyleMapElement(const uint32_t index_, PathStyleMapPref& ps__);
				
					/**
						\brief		Similar to getPathStyleMapElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the path style map JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		ps__		Return-by-reference for the path style map member located at index_ (or
												created anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreatePathStyleMapElement(
						const uint32_t index_,
						const PathStyle::Map& default_,
						PathStyleMapPref& ps__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent point object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the point JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		p__		Return-by-reference for the point element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getPointElement(const uint32_t index_, PointPref& p__) const;

				/**
					\brief		Similar to getPointElement(), except that, if the element does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the point JSON node to get from the top-level and return
					\param		default_	Default value for the element, in case it didn't already exist
					\param		p__			Return-by-reference for the point element located at index_ (or created
											anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreatePointElement(const uint32_t index_, const Point& default_, PointPref& p__);

				/**
					\brief		Gets a persistent rect object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the rect JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		r__		Return-by-reference for the rect element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getRectElement(const uint32_t index_, RectPref& r__) const;

				/**
					\brief		Similar to getRectElement(), except that, if the element does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the rect JSON node to get from the top-level and return
					\param		default_	Default value for the element, in case it didn't already exist
					\param		r__			Return-by-reference for the rect element located at index_ (or created anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateRectElement(const uint32_t index_, const Rect& default_, RectPref& r__);

				/**
					\brief		Gets a persistent size object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the size JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		s__		Return-by-reference for the size element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getSizeElement(const uint32_t index_, SizePref& s__) const;

				/**
					\brief		Similar to getSizeElement(), except that, if the element does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the size JSON node to get from the top-level and return
					\param		default_	Default value for the element, in case it didn't already exist
					\param		s__			Return-by-reference for the size element located at index_ (or created anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateSizeElement(const uint32_t index_, const Size& default_, SizePref& s__);

				/**
					\brief		Gets a persistent string object from the array
					\author		GW
					\date		09/2013
					
					\param		index_	Index of the string JSON node to get from the top-level and return as a neatly-
										packed persistent data type
					\param		str__	Return-by-reference for the string element located at index_
					\returns	true if the element could be acquired, false otherwise
				*/
				virtual bool getStringElement(const uint32_t index_, StringPref& str__) const;

				/**
					\brief		Similar to getStringElement(), except that, if the element does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		index_		Index of the string JSON node to get from the top-level and return
					\param		default_	Default value for the element, in case it didn't already exist, as UTF-8
					\param		str__		Return-by-reference for the string element located at index_ (or created
											anew)
					\returns	true if the element could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateStringElement(
					const uint32_t index_,
					const std::string& default_,
					StringPref& str__
				);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent stroke style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the stroke style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		ss__	Return-by-reference for the stroke style element located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getStrokeStyleElement(const uint32_t index_, StrokeStylePref& ss__);
				
					/**
						\brief		Similar to getStrokeStyleElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the stroke style JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		ss__		Return-by-reference for the stroke style element located at index_ (or created
												anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateStrokeStyleElement(
						const uint32_t index_,
						const StrokeStyle& default_,
						StrokeStylePref& ss__
					);
				
					/**
						\brief		Gets a persistent stroke style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		index_	Index of the stroke style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		ss__	Return-by-reference for the stroke style map member located at index_
						\returns	true if the element could be acquired, false otherwise
					*/
					virtual bool getStrokeStyleMapElement(const uint32_t index_, StrokeStyleMapPref& ss__);
				
					/**
						\brief		Similar to getStrokeStyleMapElement(), except that, if the element does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		index_		Index of the stroke style map JSON node to get from the top-level and return
						\param		default_	Default value for the element, in case it didn't already exist
						\param		ss__		Return-by-reference for the stroke style map member located at index_ (or
												created anew)
						\returns	true if the element could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateStrokeStyleMapElement(
						const uint32_t index_,
						const StrokeStyle::Map& default_,
						StrokeStyleMapPref& ss__
					);
				#endif
				// HDI_CORE_AIP_MODE
		};
		
		typedef std::unique_ptr<ArrayPref> ArrayPrefUP;
		typedef std::shared_ptr<ArrayPref> ArrayPrefSP;
		typedef std::weak_ptr<ArrayPref> ArrayPrefWP;
	}
}

#endif
// __HDI_CORE_ARRAY_PREF__
