/**
	\file
	\brief		Header file for geometric point manipulation on the Illustrator artboard
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_ARTBOARD_POINT__
#define __HDI_CORE_ARTBOARD_POINT__

#if defined(HDI_CORE_AIP_MODE)

#include <vector>

#include "hdicoreAngle.h"

namespace hdi
{
	namespace aip
	{
		class ArtboardPoint;
	}

	namespace core
	{
		class TransformMatrix;
	
		/**
			\brief	Describes a point on the Illustrator artboard
		*/
		class ArtboardPoint
		{
			public:
				typedef std::vector<ArtboardPoint> ArtboardPointVector;
			
				/**
					\brief	Position on the x-axis
				*/
				double x;

				/**
					\brief	Position on the y-axis
				*/
				double y;

				/**
					\brief	Default ArtboardPoint constructor, setting the point to (0,0)
					\author	GW
					\date	08/2013
				*/
				ArtboardPoint();

				/**
					\brief	ArtboardPoint copy constructor
					\author	GW
					\date	08/2013

					\param	pt_		Existing ArtboardPoint object to copy
				*/
				ArtboardPoint(const ArtboardPoint& pt_);

				/**
					\brief	ArtboardPoint constructor, taking x and y arguments
					\author	GW
					\date	08/2013

					\param	x_	Location of the item on the x-axis
					\param	y_	Location of the item on the y-axis
				*/
				ArtboardPoint(const double x_, const double y_);

				/**
					\brief	ArtboardPoint destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~ArtboardPoint();

				/**
					\brief		"Named constructor" for a point that is a given length and angle away from the origin
					\author		GW
					\date		08/2013

					\param		length_		Length away from the origin
					\param		angle_		Angle that the point is rotated about the origin and away from the x-axis
					\returns	An ArtboardPoint at the given distance and angle from the origin (0,0)
				*/
				static ArtboardPoint LengthAngle(const double length_, const Angle& angle_);

				/**
					\brief		Gets whether a given point is within a certain tolerance ("close to") the target point
					\author		GW
					\date		08/2013

					\param		compare_	Point to compare with
					\param		tol_		Tolerance between the points; a definition of what constitutes "close"
					\returns	true if the points are within tol_ distance of one another, false otherwise
				*/
				virtual bool closeTo(const ArtboardPoint& compare_, const double tol_) const;

				/**
					\brief		Gets the dot product of two ArtboardPoint objects
					\author		GW
					\date		08/2013

					\param		p2_		Second ArtboardPoint object with which the dot product is calculated
					\returns	The dot product of the target point and the given point (p2_)
				*/
				virtual double dotProduct(const ArtboardPoint& p2_) const;

				/**
					\brief		Offsets a point along the x- and y-axes by given amounts
					\author		GW
					\date		08/2013

					\param		tx_		Amount to offset along the x-axis
					\param		ty_		Amount to offset along the y-axis
					\returns	A copy of the target point, offset by the given amounts

					\note		Negative values go toward 0.0 if the object to be transformed has a positive x and/or y
								value; positive values go away from 0.0
				*/
				virtual ArtboardPoint offset(const double tx_, const double ty_) const;

				/**
					\brief		Rotates the point about a given "origin" point by a certain angle
					\author		GW
					\date		08/2013

					\param		origin_		New "origin" to rotate around
					\param		angle_		Angle to rotate by
					\returns	A copy of the target point, rotated as specified
				*/
				virtual ArtboardPoint rotate(const ArtboardPoint& origin_, const Angle& angle_) const;

				/**
					\brief		"Moves" a point by a given length and angle away from its current position
					\author		GW
					\date		08/2013

					\param		length_		Length away from the current position
					\param		angle_		Angle that the point is moved from the current position
					\returns	A copy of the target point, moved as specified
				*/
				virtual ArtboardPoint move(const double length_, const Angle& angle_) const;

				/**
					\brief		Transforms a point via a 2-dimensional transformation matrix
					\author		GW
					\date		08/2013

					\param		m_	Transformation matrix to apply to the point
					\returns	A copy of the target point, transformed as specified by the matrix
				*/
				virtual ArtboardPoint transform(const TransformMatrix& m_) const;

				/**
					\brief		Checks if the point is within the provided polygon
					\author		GW
					\date		08/2013

					\param		points_[]	Points that describe the polygon
					\returns	true if the point lies within the provided polygon, false otherwise
				*/
				virtual bool insidePolygon(const ArtboardPointVector& points_) const;

				/**
					\brief		Allows one ArtboardPoint object to be assigned from another
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ArtboardPoint object, but with its value updated to match that of rhs_
				*/
				virtual ArtboardPoint& operator=(const ArtboardPoint& rhs_);

				/**
					\brief		Allows two ArtboardPoint objects to be compared with the == operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the == operator
					\returns	true if this.x == rhs_.x and this.y == rhs_.y

					\note		Due to the nature of floating point comparisons, it is better to use a method like
								closeTo() to determine if two ArtboardPoints are "equal"
				*/
				virtual bool operator==(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two ArtboardPoint objects to be compared with the != operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the != operator
					\returns	true if this.x != rhs_.x or this.y != rhs_.y

					\note		Due to the nature of floating point comparisons, it is better to use a method like
								closeTo() to determine if two ArtboardPoints are "equal"
				*/
				virtual bool operator!=(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two ArtboardPoint objects to be compared with the < operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the < operator
					\returns	true if this->x and this->y are less than rhs_.x and rhs_.y, respectively
				*/
				virtual bool operator<(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two ArtboardPoint objects to be compared with the <= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the <= operator
					\returns	true if this->x and this->y are less than/equal to rhs_.x and rhs_.y, respectively
				*/
				virtual bool operator<=(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two ArtboardPoint objects to be compared with the > operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the > operator
					\returns	true if this->x and this->y are greater than rhs_.x and rhs_.y, respectively
				*/
				virtual bool operator>(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two ArtboardPoint objects to be compared with the >= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the >= operator
					\returns	true if this->x and this->y are greater than/equal to rhs_.x and rhs_.y, respectively
				*/
				virtual bool operator>=(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows a value to be added to both this->x and this->y using the + operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the + operator
					\returns	A copy of the target ArtboardPoint, with a value added to both the x and y members
				*/
				virtual ArtboardPoint operator+(const double rhs_) const;

				/**
					\brief		Allows a value to be subtracted from both this->x and this->y using the - operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the - operator
					\returns	A copy of the target ArtboardPoint, with a value subtracted from both the x and y members
				*/
				virtual ArtboardPoint operator-(const double rhs_) const;

				/**
					\brief		Allows a value to be multiplied with both this->x and this->y using the * operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the * operator
					\returns	A copy of the target ArtboardPoint, with both the x and y members multiplied by a value
				*/
				virtual ArtboardPoint operator*(const double rhs_) const;

				/**
					\brief		Allows both this->x and this->y to be divided by a value using the / operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the / operator
					\returns	A copy of the target ArtboardPoint, with both the x and y members divided by a value
				*/
				virtual ArtboardPoint operator/(const double rhs_) const;

				/**
					\brief		Allows two points to have their x and y values added using the + operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the + operator
					\returns	A copy of the target ArtboardPoint, with all x and y values summed together
				*/
				virtual ArtboardPoint operator+(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two points to have their x and y values subtracted using the - operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the - operator
					\returns	A copy of the target ArtboardPoint, with rhs_ x and y values subtracted from this x and y
				*/
				virtual ArtboardPoint operator-(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two points to have their x and y values multiplied using the * operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the * operator
					\returns	A copy of the target ArtboardPoint, with this x and y values multipled by rhs_ x and y
				*/
				virtual ArtboardPoint operator*(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows two points to have their x and y values divided using the / operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the / operator
					\returns	A copy of the target ArtboardPoint, with this x and y values divided by rhs_ x and y
				*/
				virtual ArtboardPoint operator/(const ArtboardPoint& rhs_) const;

				/**
					\brief		Allows a value to be added and assigned to both this->x and this->y using the += operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the += operator
					\returns	The lefthand side of the += operator, with a value added to both the x and y members
				*/
				virtual ArtboardPoint& operator+=(const double rhs_);

				/**
					\brief		Allows a value to be subtracted from, and assigned to, both this->x and this->y using the
								-= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the -= operator
					\returns	The lefthand side of the -= operator, with a value subtracted from both the x and y members
				*/
				virtual ArtboardPoint& operator-=(const double rhs_);

				/**
					\brief		Allows a value to be multiplied with, and assigned to, both this->x and this->y using the
								*= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the *= operator
					\returns	The lefthand side of the *= operator, with both the x and y members multiplied by a value
				*/
				virtual ArtboardPoint& operator*=(const double rhs_);

				/**
					\brief		Allows both this->x and this->y to be divided by a value, and assigned, using the /=
								operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the /= operator
					\returns	The lefthand side of the /= operator, with both the x and y members divided by a value
				*/
				virtual ArtboardPoint& operator/=(const double rhs_);

				/**
					\brief		Allows two points to have their x and y values added and assigned using the += operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the += operator
					\returns	The lefthand side of the += operator, with all x and y values summed together
				*/
				virtual ArtboardPoint& operator+=(const ArtboardPoint& rhs_);

				/**
					\brief		Allows two points to have their x and y values subtracted and assigned using the -=
								operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the -= operator
					\returns	The lefthand side of the -= operator, with rhs_ x and y values subtracted from this x and y
				*/
				virtual ArtboardPoint& operator-=(const ArtboardPoint& rhs_);

				/**
					\brief		Allows two points to have their x and y values multiplied and assigned using the *=
								operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the *= operator
					\returns	The lefthand side of the *= operator, with this x and y values multipled by rhs_ x and y
				*/
				virtual ArtboardPoint& operator*=(const ArtboardPoint& rhs_);

				/**
					\brief		Allows two points to have their x and y values divided and assigned using the /= operator
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the /= operator
					\returns	The lefthand side of the /= operator, with this x and y values divided by rhs_ x and y
				*/
				virtual ArtboardPoint& operator/=(const ArtboardPoint& rhs_);

				/**
					\brief		Gets the angle of the vector (this, rhs_), if this were moved to the origin
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the ^ operator (called the angle operator in this case)
					\returns	The angle formed by vector (this, rhs_)
				*/
				virtual Angle operator^(const ArtboardPoint& rhs_) const;

				/**
					\brief		Gets the distance between this and rhs_
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the | operator (called the length operator in this case)
					\returns	The distance between this and rhs_
				*/
				virtual double operator|(const ArtboardPoint& rhs_) const;


			private:
				friend aip::ArtboardPoint* __accessImpl(const ArtboardPoint&);
				friend ArtboardPoint __accessCtor(const aip::ArtboardPoint&);

				/**
					\brief	Private implementation object
				*/
				aip::ArtboardPoint* __impl;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				ArtboardPoint(const aip::ArtboardPoint&);
		};
		
		typedef std::unique_ptr<ArtboardPoint> ArtboardPointUP;
		typedef std::shared_ptr<ArtboardPoint> ArtboardPointSP;
		typedef std::weak_ptr<ArtboardPoint> ArtboardPointWP;
		
		extern aip::ArtboardPoint* __accessImpl(const ArtboardPoint&);
		extern ArtboardPoint __accessCtor(const aip::ArtboardPoint&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ARTBOARD_POINT__
