/**
	\file
	\brief		Header file for artboard point preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_ARTBOARD_POINT_PREF__
#define __HDI_CORE_ARTBOARD_POINT_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardPoint.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of 2D artboard points
		*/
		class ArtboardPointPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty ArtboardPointPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a ArtboardPointPref object is empty, call isEmpty() on it
					\note	Empty ArtboardPointPref objects do not relate to any actual preference data; they are
							designed to be "receivers" of some other ArtboardPointPref object via the overloaded
							assignment operator. Empty ArtboardPointPref objects are useless until such time (though it
							is safe to call any of their methods).
				*/
				ArtboardPointPref();
			
				/**
					\brief	Constructs a new ArtboardPointPref object from an existing ArtboardPointPref object (copy
							constructor)
					\author	GW
					\date	09/2013

					\param	abp_	Existing ArtboardPointPref object
				*/
				ArtboardPointPref(const ArtboardPointPref& abp_);

				/**
					\brief	Constructs an ArtboardPointPref object, with new data, to be added to a container at a later
							time
					\author	GW
					\date	09/2013

					\param	value_	Initial point value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				ArtboardPointPref(const ArtboardPoint& value_, const std::string& name_ = "");

				/**
					\brief	Destructs an ArtboardPointPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~ArtboardPointPref();

				/**
					\brief		Allows one ArtboardPointPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ArtboardPointPref object, but with its value updated to match that of rhs_
				*/
				virtual ArtboardPointPref& operator=(const ArtboardPointPref& rhs_);
				
				/**
					\brief		Convenience method to clone an ArtboardPointPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ArtboardPointPref object
					
					\note		If you subclass ArtboardPointPref, you MUST overload this method yourself! If you don't
								and/or your clone() method does not return an instance of your ArtboardPointPref
								subclass, you will experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned ArtboardPointPref object.
				*/
				virtual ArtboardPointPref* clone() const;

				/**
					\brief		Gets the point for this object
					\author		GW
					\date		09/2013

					\returns	The ArtboardPoint object being described by this wrapper
				*/
				virtual ArtboardPoint artboardPoint() const;

				/**
					\brief		Sets the point for this object
					\author		GW
					\date		09/2013

					\param		pt_		New point value
					\returns	true if the point is successfully updated, or false otherwise
				*/
				virtual bool setArtboardPoint(const ArtboardPoint& pt_);
		};
		
		typedef std::unique_ptr<ArtboardPointPref> ArtboardPointPrefUP;
		typedef std::shared_ptr<ArtboardPointPref> ArtboardPointPrefSP;
		typedef std::weak_ptr<ArtboardPointPref> ArtboardPointPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ARTBOARD_POINT_PREF__
