/**
	\file
	\brief		Header file for geometric rectangle manipulation on the Illustrator artboard
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_ARTBOARD_RECT__
#define __HDI_CORE_ARTBOARD_RECT__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreArtboardPoint.h"
#include "hdicoreSize.h"

namespace hdi
{
	namespace aip
	{
		class ArtboardRect;
	}

	namespace core
	{
		/**
			\brief	Describes a rectangular area on the Illustrator artboard
		*/
		class ArtboardRect
		{
			public:
				/**
					\brief	Origin point for the rect (top-left, when size.width and size.height are positive)
				*/
				ArtboardPoint origin;

				/**
					\brief	Dimensions of the rect
				*/
				Size size;

				/**
					\brief	Default ArtboardRect constructor, sets the top-left to (0,0) and the width and height to 0
					\author	GW
					\date	08/2013
				*/
				ArtboardRect();

				/**
					\brief	ArtboardRect copy constructor
					\author	GW
					\date	08/2013

					\param	rect_	ArtboardRect object to copy values from
				*/
				ArtboardRect(const ArtboardRect& rect_);

				/**
					\brief	Constructor to create an ArtboardRect from an ArtboardPoint object and Size object
					\author	GW
					\date	08/2013

					\param	pt_		Top-left point of the rectangle
					\param	size_	Size (width and height of the rectangle)
				*/
				ArtboardRect(const ArtboardPoint& pt_, const Size& size_);

				/**
					\brief	Constructor to create an ArtboardRect object from top-left and bottom-right ArtboardPoint
							objects
					\author	GW
					\date	08/2013

					\param	tl_		Top-left point of the rectangle
					\param	br_		Bottom-right point of the rectangle
				*/
				ArtboardRect(const ArtboardPoint& tl_, const ArtboardPoint& br_);
				
				/**
					\brief	ArtboardRect destructor
					\author	GW
					\date	08/2013
				*/
				virtual ~ArtboardRect();

				/**
					\brief		"Named constructor" to create an ArtboardRect from an (x,y) top-left point and a width 
								and height
					\author		GW
					\date		08/2013

					\param		x_			Position of the left of the rect along the x-axis
					\param		y_			Position of the top of the rect along the y-axis
					\param		width_		Width of the rect
					\param		height_		Height of the rect

					\returns	New ArtboardRect object with the given geometry
				*/
				static ArtboardRect XYWidthHeight(
					const double x_,
					const double y_,
					const double width_,
					const double height_
				);

				/**
					\brief		"Named constructor" to create an ArtboardRect from the values of its bounding edges
					\author		GW
					\date		08/2013

					\param		left_		Position of the left of the rect along the x-axis
					\param		top_		Position of the top of the rect along the y-axis
					\param		right_		Position of the right of the rect along the x-axis
					\param		bottom_		Position of the bottom of the rect along the y-axis

					\returns	New ArtboardRect object with the given geometry
				*/
				static ArtboardRect LeftTopRightBottom(
					const double left_,
					const double top_,
					const double right_,
					const double bottom_
				);

				/**
					\brief		Allows one ArtboardRect object to be assigned from another
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ArtboardRect object, but with its value updated to match that of rhs_
				*/
				virtual ArtboardRect& operator=(const ArtboardRect& rhs_);
				
				/**
					\brief		Tests whether a given ArtboardRect object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	ArtboardRect to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same rect, false otherwise
				*/
				virtual bool operator==(const ArtboardRect& rhs_) const;
				
				/**
					\brief		Tests whether a given ArtboardRect object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	ArtboardRect to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different rects, false otherwise
				*/
				virtual bool operator!=(const ArtboardRect& rhs_) const;

				/**
					\brief		Gets the position of the left bounding edge of the rectangle along the x-axis
					\author		GW
					\date		08/2013

					\returns	The position of the left side of the rectangle
				*/
				virtual double left() const;

				/**
					\brief		Gets the position of the top bounding edge of the rectangle along the y-axis
					\author		GW
					\date		08/2013

					\returns	The position of the top side of the rectangle
				*/
				virtual double top() const;

				/**
					\brief		Gets the position of the right bounding edge of the rectangle along the x-axis
					\author		GW
					\date		08/2013

					\returns	The position of the right side of the rectangle
				*/
				virtual double right() const;

				/**
					\brief		Gets the position of the bottom bounding edge of the rectangle along the y-axis
					\author		GW
					\date		08/2013

					\returns	The position of the bottom side of the rectangle
				*/
				virtual double bottom() const;

				/**
					\brief	Sets the position of the left bounding edge of the rectangle along the x-axis
					\author	GW
					\date	08/2013

					\param	left_	New left edge of the rect

					\note	The right edge will be unaffected (e.g. the width will automatically be adjusted such that
							the right bounding edge stays at the same position)
				*/
				virtual void setLeft(const double left_);

				/**
					\brief	Sets the position of the top bounding edge of the rectangle along the y-axis
					\author	GW
					\date	08/2013

					\param	top_	New top edge of the rect

					\note	The bottom edge will be unaffected (e.g. the height will automatically be adjusted such that
							the bottom bounding edge stays at the same position)
				*/
				virtual void setTop(const double top_);

				/**
					\brief	Sets the position of the right bounding edge of the rectangle along the x-axis
					\author	GW
					\date	08/2013

					\param	right_	New right edge of the rect

					\note	The left edge will be unaffected
				*/
				virtual void setRight(const double right_);

				/**
					\brief	Sets the position of the bottom bounding edge of the rectangle along the y-axis
					\author	GW
					\date	08/2013

					\param	bottom_		New bottom edge of the rect

					\note	The top edge will be unaffected
				*/
				virtual void setBottom(const double bottom_);

				/**
					\brief		Insets a rect by given amounts on the x- and y-axes (or expands if negative)
					\author		GW
					\date		08/2013

					\param		x_	Amount to inset the rect on the left and right sides
					\param		y_	Amount to inset the rect on the top and bottom sides
					\returns	A copy of the target ArtboardRect, with its coordinates adjusted for the inset

					\note		A positive value causes an inset, while a negative value causes an "outset" (expansion)
				*/
				virtual ArtboardRect inset(const double x_, const double y_) const;

				/**
					\brief		Gets the top-left point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the top-left of the rect
					
					\note		The point returned may not be the geometric top-left of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint topLeft() const;

				/**
					\brief		Gets the top-center point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the top-center of the rect
					
					\note		The point returned may not be the geometric top-center of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint topCenter() const;

				/**
					\brief		Gets the top-right point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the top-right of the rect
					
					\note		The point returned may not be the geometric top-right of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint topRight() const;

				/**
					\brief		Gets the mid-left point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the mid-left of the rect
					
					\note		The point returned may not be the geometric mid-left of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint midLeft() const;

				/**
					\brief		Gets the mid-center point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the mid-center of the rect
				*/
				virtual ArtboardPoint midCenter() const;

				/**
					\brief		Gets the mid-right point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the mid-right of the rect
					
					\note		The point returned may not be the geometric mid-right of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint midRight() const;

				/**
					\brief		Gets the bottom-left point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the bottom-left of the rect
					
					\note		The point returned may not be the geometric bottom-left of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint bottomLeft() const;

				/**
					\brief		Gets the bottom-center point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the bottom-center of the rect
					
					\note		The point returned may not be the geometric bottom-center of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint bottomCenter() const;

				/**
					\brief		Gets the bottom-right point of the rect
					\author		GW
					\date		08/2013

					\returns	An ArtboardPoint object describing the bottom-right of the rect
					
					\note		The point returned may not be the geometric bottom-right of the rect in the case of a
								negative width and/or height
				*/
				virtual ArtboardPoint bottomRight() const;

				/**
					\brief		Gets a "pin point" from the rect bounds
					\author		GW
					\date		08/2013

					\param		pinPt_	Pin point which will be turned into an absolute point from the rect
					\returns	An ArtboardPoint object for the given pin point
				*/
				virtual ArtboardPoint pointFromPinPoint(const BoundsPinPointType pinPt_) const;

				/**
					\brief		Gets the area of the rect
					\author		GW
					\date		08/2013

					\returns	The area of the rect

					\note		The value returned will always be positive
				*/
				virtual double area() const;

				/**
					\brief		Gets the perimeter of the rect
					\author		GW
					\date		08/2013
					
					\returns	The perimeter of the rect
				*/
				virtual double perimeter() const;

				/**
					\brief		Gets whether a given rect is equal to the target rect, within a given tolerance
					\author		GW
					\date		08/2013

					\param		compare_	Rect to compare to the target
					\param		tol_		Tolerance about the sides of the rect
					\returns	true if the rects are equal within tol_, false otherwise
				*/
				virtual bool equalWithinTol(const ArtboardRect& compare_, const double tol_) const;

				/**
					\brief		Gets whether a given point lies inside the target rect
					\author		GW
					\date		08/2013

					\param		pt_		Point to test
					\returns	true if the point is within the bounds of the rect, false otherwise
				*/
				virtual bool containsPoint(const ArtboardPoint& pt_) const;

				/**
					\brief		Checks whether the target rect overlaps a given rect
					\author		GW
					\date		08/2013

					\param		rect2_	Rect to compare against for overlapping boundaries
					\returns	true if the target rect and the provided rect overlap
				*/
				virtual bool overlaps(const ArtboardRect& rect2_) const;

				/**
					\brief		Enlarges the target rect, if necessary, to enclose a given point
					\author		GW
					\date		08/2013

					\param		pt_		Point the rect should enclose
					\returns	true if the rect was enlarged, false otherwise
				*/
				virtual bool enclosePoint(const ArtboardPoint& pt_);

				/**
					\brief		Unions two rects, forming a rect that has bounds sufficient to enclose both initial rects
					\author		GW
					\date		08/2013

					\param		rect2_	Second rect to consider in the union
					\returns	A new ArtboardRect large enough to enclose both the target rect and rect2_
				*/
				virtual ArtboardRect unionRects(const ArtboardRect& rect2_) const;

				/**
					\brief		Scales the width and height of a rect by given amounts on the x- and y- axes
					\author		GW
					\date		08/2013

					\param		xScaler_	Multiplier for scaling horizontally (e.g. 0.75 is 75% of the current size)
					\param		yScaler_	Multiplier for scaling vertically (e.g. 0.75 is 75% of the current size)
					\returns	A copy of the target ArtboardRect that is scaled as specified
				*/
				virtual ArtboardRect scale(const double xScaler_, const double yScaler_) const;

				/**
					\brief		Gets the intersection between two rects
					\author		GW
					\date		08/2013

					\param		rect2_		Second rect to test for an intersection with the target rect
					\param		iRect__		Return-by-reference for the intersection rect
					\returns	true if an intersection exists, false otherwise

					\note		iRect__ will be the rect that describes the intersecting area between rect2_ and the
								target rect.
				*/
				virtual bool intersection(const ArtboardRect& rect2_, ArtboardRect& iRect__) const;

				/**
					\brief	Gets the width and height of an ellipse that would enclose the rect
					\author	GW
					\date	08/2013

					\param	width__		Return-by-reference for the ellipse width
					\param	height__	Return-by-reference for the ellipse height

					\note	The ellipse will have the same proportions as the rect
				*/
				virtual void enclosingEllipse(double& width__, double& height__) const;

				/**
					\brief		Finds the intersection points on a rectangle that a line has with it
					\author		GW
					\date		08/2013

					\param		a_		Start point for the line
					\param		b_		End point for the line
					\param		iPt1__	Return-by-reference for the first intersection point found
					\param		iPt2__	Return-by-reference for the second intersection point found
					\returns	An enum describing which intersection points were found, if any

					\note		For reference, this function always checks clockwise from the left of the rectangle.
				*/
				virtual RectLineIntersectionType lineIntersectionPoints(
					const ArtboardPoint& a_,
					const ArtboardPoint& b_,
					ArtboardPoint& iPt1__,
					ArtboardPoint& iPt2__
				) const;

				/**
					\brief		Finds the intersection points on a rectangle, were it rotated, that a line has with it
					\author		GW
					\date		08/2013

					\param		a_		Start point for the line
					\param		b_		End point for the line
					\param		angle_	Amount the rectangle is rotated
					\param		iPt1__	Return-by-reference for the first intersection point found
					\param		iPt2__	Return-by-reference for the second intersection point found
					\returns	An enum describing which intersection points were found, if any

					\note		For reference, this function always checks clockwise from the left of the rectangle.
				*/
				virtual RectLineIntersectionType lineIntersectionPoints(
					const ArtboardPoint& a_,
					const ArtboardPoint& b_,
					const Angle& angle_,
					ArtboardPoint& iPt1__,
					ArtboardPoint& iPt2__
				) const;


			private:
				friend aip::ArtboardRect* __accessImpl(const ArtboardRect&);
				friend ArtboardRect __accessCtor(const aip::ArtboardRect&);

				/**
					\brief	Private implementation object
				*/
				aip::ArtboardRect* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				ArtboardRect(const aip::ArtboardRect&);
		};
		
		typedef std::unique_ptr<ArtboardRect> ArtboardRectUP;
		typedef std::shared_ptr<ArtboardRect> ArtboardRectSP;
		typedef std::weak_ptr<ArtboardRect> ArtboardRectWP;
		
		extern aip::ArtboardRect* __accessImpl(const ArtboardRect&);
		extern ArtboardRect __accessCtor(const aip::ArtboardRect&);
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_ARTBOARD_RECT__
