/**
	\file
	\brief		Header file for Illustrator cubic bezier manipulation
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_BEZIER__
#define __HDI_CORE_BEZIER__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreAngle.h"
#include "hdicoreArtboardRect.h"

namespace hdi
{
	namespace aip
	{
		class Bezier;
	}

	namespace core
	{
		class ArtboardSegPoint;
	
		/**
			\brief	Allows for cubic bezier calculations and manipulation (e.g. division, tangents, normals, etc.)
		*/
		class Bezier
		{
			public:
				/**
					\brief	Indicates what happened when a bezier was adjusted to pass through a point
				*/
				enum AdjustmentResult
				{
					AdjustmentReflectedStartOut	= 1 << 0,
					AdjustmentReflectedEndIn	= 1 << 1,
					AdjustmentRotatedStartOut	= 1 << 2,
					AdjustmentRotatedEndIn		= 1 << 3
				};
				
				/**
					\brief	Constructs a new Bezier object with all points at (0,0)
					\author	GW
					\date	08/2013
				*/
				Bezier();

				/**
					\brief	Constructs a new Bezier object from an existing Bezier object (copy constructor)
					\author	GW
					\date	08/2013

					\param	bez_	Existing Bezier object
				*/
				Bezier(const Bezier& bez_);

				/**
					\brief	Constructs a new Bezier object from two segment endpoints
					\author	GW
					\date	08/2013

					\param	segPt1_		First segment point
					\param	segPt2_		Second segment point

					\note	This will ultimately call setPoints() internally with args segPt1_.p, segPt1_.out,
							segPt2_.in, and segPt2_.p, respectively.
				*/
				Bezier(const ArtboardSegPoint& segPt1_, const ArtboardSegPoint& segPt2_);

				/**
					\brief	Constructs a new Bezier object from individual component points
					\author	GW
					\date	08/2013

					\param	start_		First anchor point
					\param	startOut_	First control point
					\param	endIn_		Second control point
					\param	end_		Second anchor point
				*/
				Bezier(
					const ArtboardPoint& start_,
					const ArtboardPoint& startOut_,
					const ArtboardPoint& endIn_,
					const ArtboardPoint& end_
				);

				/**
					\brief	Destructs a Bezier object
					\author	GW
					\date	08/2013
				*/
				virtual ~Bezier();

				/**
					\brief		Allows one Bezier object to be assigned from another
					\author		GW
					\date		08/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Bezier object, but with its value updated to match that of rhs_
				*/
				virtual Bezier& operator=(const Bezier& rhs_);

				/**
					\brief		Gets the first anchor point of the bezier
					\author		GW
					\date		08/2013

					\returns	The first anchor point, AKA the start point
				*/
				virtual ArtboardPoint startPoint() const;

				/**
					\brief		Gets the first control point of the bezier
					\author		GW
					\date		08/2013

					\returns	The first control point, AKA the start-out point
				*/
				virtual ArtboardPoint startOutPoint() const;

				/**
					\brief		Gets the second control point of the bezier
					\author		GW
					\date		08/2013

					\returns	The second control point, AKA the end-in point
				*/
				virtual ArtboardPoint endInPoint() const;

				/**
					\brief		Gets the second anchor point of the bezier
					\author		GW
					\date		08/2013

					\returns	The second anchor point, AKA the end point
				*/
				virtual ArtboardPoint endPoint() const;

				/**
					\brief		Adjusts the bezier such that the given t-value will pass through the provided point
					\author		GW
					\date		08/2013

					\param		pt_		Point the bezier should pass through at t_
					\param		t_		T-value along the bezier to force through pt_
					\returns	An enum value describing how the bezier was adjusted
				*/
				virtual AdjustmentResult adjustThruPoint(const ArtboardPoint& pt_, const double t_);

				/**
					\brief	Splits the target bezier into two separate beziers at a given t-value
					\author	GW
					\date	08/2013

					\param	t_		T-value along the target bezier to perform the split
					\param	bez1__	Return-by-reference for the first bezier resulting from the split
					\param	bez2__	Return-by-reference for the second bezier resulting from the split
				*/
				virtual void divide(const double t_, Bezier& bez1__, Bezier& bez2__) const;

				/**
					\brief		Gets a section of the target bezier, from a starting t-value to an ending t-value
					\author		GW
					\date		08/2013
					
					\param		startT_		Starting position (t-value) of the bezier subsection
					\param		endT_		Ending position (t-value) of the bezier subsection
					\returns	A Bezier object describing the desired subsection of the original target bezier
				*/
				virtual Bezier section(const double startT_, const double endT_) const;

				/**
					\brief		Evaluates a given t-value along the target bezier as a point on the artboard
					\author		GW
					\date		08/2013

					\param		t_	T-value along the bezier for the point in question
					\returns	The point for the given t-value along the target bezier
				*/
				virtual ArtboardPoint evaluate(const double t_) const;

				/**
					\brief		Gets the length of the bezier
					\author		GW
					\date		08/2013
					
					\returns	The length of the target bezier in points
				*/
				virtual double length() const;

				/**
					\brief		Gives approximate length of a bezier from t-value startT_ to t-value endT_
					\author		GW
					\date		08/2013

					\param		startT_		Starting position (t-value) to begin the calculation
					\param		endT_		Ending position (t-value) to end the calculation
					\returns	The length of the subsection of the target bezier between startT_ and endT_
				*/
				virtual double sublength(const double startT_, const double endT_) const;

				/**
					\brief		Gives the t-value at a specific length along the target bezier
					\author		GW
					\date		08/2013
					
					\param		length_		The length along the target bezier, in points, for which a t-value is desired
					\returns	The t-value for the provided length, or 0.0 for negative lengths, or 1.0 for lengths
								greater than the total length of the target bezier
				*/
				virtual double tAtLength(const double length_) const;

				/**
					\brief	Sets the anchor and control points of the bezier
					\author	GW
					\date	08/2013

					\param	start_		First anchor point
					\param	startOut_	First control point
					\param	endIn_		Second control point
					\param	end_		Second anchor point
				*/
				virtual void setPoints(
					const ArtboardPoint& start_,
					const ArtboardPoint& startOut_,
					const ArtboardPoint& endIn_,
					const ArtboardPoint& end_
				);
				
				/**
					\brief		Gets the tangent angle at some t-value along the bezier
					\author		GW
					\date		08/2013

					\param		t_	T-value on the bezier whose tangent angle is needed
					\returns	The angle of the tangent at the given t-value, or the tangent angle at 0.0 for values
								less than 0.0, or the tangent angle at 1.0 for values greater than 1.0
				*/
				virtual Angle tangentAngle(const double t_) const;
				
				/**
					\brief		Gets the normal angle at some t-value along the bezier
					\author		GW
					\date		08/2013

					\param		t_	T-value on the bezier whose normal angle is needed
					\returns	The angle of the normal at the given t-value, or the normal angle at 0.0 for values
								less than 0.0, or the normal angle at 1.0 for values greater than 1.0
				*/
				virtual Angle normalAngle(const double t_) const;
				
				/**
					\brief		Determines whether the bezier has a curved path
					\author		GW
					\date		08/2013
					
					\returns	true for curved beziers, false for straight
				*/
				virtual bool curved() const;

				/**
					\brief		Gets the radius of the curve at some t-value along the bezier
					\author		GW
					\date		08/2013

					\param		t_			T-value on the bezier whose radius is needed
					\param		radius__	Return-by-reference for the radius of the curve of the bezier at t_
					\returns	true for success, false for failure (i.e. straight beziers do not have a radius)
				*/
				virtual bool radius(const double t_, double& radius__) const;

				/**
					\brief		Gets the center point of the curve at some t-value along the bezier
					\author		GW
					\date		08/2013

					\param		t_			T-value on the bezier whose radius is needed
					\param		center__	Return-by-reference for the center point of the curve of the bezier at t_
					\returns	true for success, false for failure (i.e. straight beziers do not have a center point)
				*/
				virtual bool center(const double t_, ArtboardPoint& center__) const;

				/**
					\brief		Gets both the radius and center point of the curve at some t-value along the bezier
					\author		GW
					\date		08/2013

					\param		t_			T-value on the bezier whose radius is needed
					\param		radius__	Return-by-reference for the radius of the curve of the bezier at t_
					\param		center__	Return-by-reference for the center point of the curve of the bezier at t_
					\returns	true for success, false for failure (i.e. straight beziers do not have a radius)
					
					\note		Calculations for the radius and center point at a t-value are related, so use this
								method for performance reasons if you need both values.
				*/
				virtual bool radiusAndCenter(const double t_, double& radius__, ArtboardPoint& center__) const;

				/**
					\brief		Gets whether the bezier describes an arc segment of a circle
					\author		GW
					\date		08/2013
					
					\returns	true if the bezier represents a circular arc segment, false otherwise
				*/
				virtual bool isCircularArc() const;

				/**
					\brief		Gets the bounds rect for the target bezier; that is, the rect that completely encloses
								all the control points, NOT that rect that encloses the extent of the bezier curvature
					\author		GW
					\date		08/2013
					
					\returns	The rectangle that completely contains the target bezier
				*/
				virtual ArtboardRect bounds() const;
				
				/**
					\brief		Tests whether a given Bezier object is the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Bezier to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same bezier (having equal anchor and control
								points), false otherwise
				*/
				virtual bool operator==(const Bezier& rhs_) const;
				
				/**
					\brief		Tests whether a given Bezier object is not the same as another
					\author		GW
					\date		08/2013

					\param		rhs_	Bezier to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different beziers (having unequal anchor and control
								points), false otherwise
				*/
				virtual bool operator!=(const Bezier& rhs_) const;


			private:
				friend aip::Bezier* __accessImpl(const Bezier&);
				friend Bezier __accessCtor(const aip::Bezier&);

				/**
					\brief	Private implementation object
				*/
				aip::Bezier* __impl;

				/**
					\brief	Internal use only
					\author	GW
					\date	08/2013
				*/
				Bezier(const aip::Bezier&);
		};
		
		typedef std::unique_ptr<Bezier> BezierUP;
		typedef std::shared_ptr<Bezier> BezierSP;
		typedef std::weak_ptr<Bezier> BezierWP;
		
		extern aip::Bezier* __accessImpl(const Bezier&);
		extern Bezier __accessCtor(const aip::Bezier&);
	}
}

hdi::core::Bezier::AdjustmentResult operator|(
	const hdi::core::Bezier::AdjustmentResult lhs_,
	const hdi::core::Bezier::AdjustmentResult rhs_
);

hdi::core::Bezier::AdjustmentResult& operator|=(
	hdi::core::Bezier::AdjustmentResult& lhs__,
	const hdi::core::Bezier::AdjustmentResult rhs_
);

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_BEZIER__
