/**
	\file
	\brief		Header file for plugin user interface box widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_BOX__
#define __HDI_CORE_BOX__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Box class for UI boxes, which form a thin labeled border around a group of other widgets
		*/
		class Box : public Widget
		{
			public:
				/**
					\brief	Constructs an empty Box object
					\author	GW
					\date	09/2013
					
					\note	To test if a Box object is empty, call isEmpty() on it
					\note	Empty Box objects do not relate to any actual UI widget; they are designed to be "receivers"
							of some other Box object via the overloaded assignment operator. Empty Box objects are
							useless until such time (though it is safe to call any of their methods).
				*/
				Box();

				/**
					\brief	Constructs a new Box object from an existing Box object (copy constructor)
					\author	GW
					\date	09/2013

					\param	box_	Existing Box object
				*/
				Box(const Box& box_);

				/**
					\brief	Constructs a box with an optional title
					\author	GW
					\date	09/2013

					\param	frame_	Position and size of the box, in 1x resolution coordinates
					\param	title_	Title of the box, as UTF-8
				*/
				Box(const Rect& frame_, const std::string& title_ = "");
				
				/**
					\brief	Box destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~Box();

				/**
					\brief		Allows one Box object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Box object, but with its value updated to match that of rhs_
				*/
				virtual Box& operator=(const Box& rhs_);
				
				/**
					\brief		Convenience method to clone a Box object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Box object
					
					\note		If you subclass Box, you MUST overload this method yourself! If you don't and/or your
								clone() method does not return an instance of your Box subclass, you will experience
								"object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned Box object.
				*/
				virtual Box* clone() const;
				
				/**
					\brief		Convenience method to duplicate a Box object, creating a new and identical UI element
								to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new Box object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Box object.
				*/
				virtual Box* duplicate() const;

				/**
					\brief		Returns BoxType, as boxes are always boxes!
					\author		GW
					\date		09/2013
					
					\returns	BoxType
				*/
				virtual Type type() const;

				/**
					\brief		Gets the current text value of the box
					\author		GW
					\date		09/2013
					
					\returns	The current text value, as UTF-8
				*/
				virtual std::string text() const;

				/**
					\brief	Sets the current text value of the box
					\author	GW
					\date	09/2013

					\param	text_	New text value, as UTF-8
				*/
				virtual void setText(const std::string& text_);

				/**
					\brief		Returns true, because boxes are always enabled
					\author		GW
					\date		09/2013

					\returns	true
				*/
				virtual bool enabled();

				/**
					\brief	Overrides Widget::setEnabled() because boxes are always enabled
					\author	GW
					\date	09/2013

					\param	enable_		true to enable, false to disable (ignored)
				*/
				virtual void setEnabled(const bool enable_);
		};
		
		typedef std::unique_ptr<Box> BoxUP;
		typedef std::shared_ptr<Box> BoxSP;
		typedef std::weak_ptr<Box> BoxWP;
	}
}

#endif
// __HDI_CORE_BOX__
