/**
	\file
	\brief		Header file for accessing Illustrator's built-in tools
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_BUILTIN_TOOL__
#define __HDI_CORE_BUILTIN_TOOL__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicoreTypes.h"

namespace hdi
{
	namespace core
	{
		class ThirdPartyPlugin;
	
		/**
			\brief	Class for accessing and querying built-in Illustrator tools
		*/
		class BuiltinTool
		{
			public:
				/**
					\brief	Describes the built-in Illustrator tools that can be acquired
				*/
				enum AdobeTool
				{
					HorizontalTypeTool			= 10,
					HorizontalAreaTypeTool		= 20,
					HorizontalPathTypeTool		= 30,
					VerticalTypeTool			= 40,
					VerticalAreaTypeTool		= 50,
					VerticalPathTypeTool		= 60,
					RTLHorizontalTypeTool		= 70,
					RTLHorizontalAreaTypeTool	= 80,
					RTLHorizontalPathTypeTool	= 90,
					TextPositioningTool			= 100,

					LineTool				= 110,
					ArcTool					= 120,
					RectangleTool			= 130,
					RoundedRectangleTool	= 140,
					EllipseTool				= 150,
					PolygonTool				= 160,
					StarTool				= 170,
					SpiralTool				= 180,
					PluginEllipseTool		= 190,
					PluginRectangleTool		= 200,

					SelectTool				= 210,
					DirectSelectTool		= 220,
					DirectObjectSelectTool	= 230,
					ToggleSelectTool		= 240,
					ToggleDirectSelectTool	= 250,

					ScaleTool			= 260,
					RotateTool			= 270,
					ReflectTool			= 280,
					ShearTool			= 290,
					FreeTransformTool	= 300,

					ColumnGraphTool			= 310,
					StackedColumnGraphTool	= 320,
					LineGraphTool			= 330,
					PieGraphTool			= 340,
					AreaGraphTool			= 350,
					ScatterGraphTool		= 360,
					BarGraphTool			= 370,
					StackedBarGraphTool		= 380,
					RadarGraphTool			= 390,

					ScrollTool			= 400,
					FreehandTool		= 410,
					FreehandSmoothTool	= 420,
					FreehandEraseTool	= 440,

					PenTool			= 450,
					ScissorsTool	= 460,
					KnifeTool		= 470,
					ReshapeTool		= 480,
					WidthTool		= 490,
					StencilTool		= 493,
					NaturalJoinTool	= 496,

					BlendTool	= 500,
					MeasureTool	= 510,
					PageTool	= 520,
					ZoomTool	= 530,

					AddAnchorPointTool		= 540,
					DeleteAnchorPointTool	= 550,
					ConvertAnchorPointTool	= 560,

					GradientVectorTool	= 570,
					BrushTool			= 580,
					EyedropperTool		= 590,
					MeshEditingTool		= 600,
					PatternTileTool		= 610,

					DirectLassoTool				= 620,
					MagicWandTool				= 630,
					RectangularGridTool			= 640,
					PolarGridTool				= 650,
					PerspectiveGridTool			= 660,
					PerspectiveSelectionTool	= 670,

					FlareTool		= 680,
					WarpTool		= 690,
					NewTwirlTool	= 700,
					PuckerTool		= 710,
					BloatTool		= 720,
					ScallopTool		= 730,
					CrystallizeTool	= 740,
					WrinkleTool		= 750,
					CurvatureTool	= 755,

					SliceTool			= 760,
					SliceSelectTool		= 770,

					SymbolSprayerTool  	= 780,
					SymbolShifterTool  	= 790,
					SymbolScruncherTool	= 800,
					SymbolSizerTool		= 810,
					SymbolSpinnerTool  	= 820,
					SymbolStainerTool  	= 830,
					SymbolScreenerTool 	= 840,
					SymbolStylerTool   	= 850,

					PlanarPaintbucketTool	= 860,
					PlanarFaceSelectTool	= 870,

					EraserTool	= 880,
					CropTool	= 890,

					BlobTool		= 900,
					LiveMergeTool	= 910,

					PlaceGunTool	= 920,
					ChartsTool		= 930,
					
					ShaperTool		= 940,
					SymmetryTool	= 950,
					
					PuppetWarpTool			= 960,
					DiffusionColoringTool	= 970,

					SmartEditTool	= 980
				};
			
				/**
					\brief	Constructs an empty BuiltinTool object
					\author	GW
					\date	07/2014
					
					\note	To test if a BuiltinTool object is empty, call isEmpty() on it
					\note	Empty BuiltinTool objects do not relate to any actual Illustrator tool; they are designed to
							be "receivers" of some other BuiltinTool object via the overloaded assignment operator.
							Empty BuiltinTool objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				BuiltinTool();
			
				/**
					\brief	Constructs a new BuiltinTool object from an existing BuiltinTool object (copy constructor)
					\author	GW
					\date	07/2014

					\param	tool_	Existing BuiltinTool object
				*/
				BuiltinTool(const BuiltinTool& tool_);
				
				/**
					\brief		Constructs a new BuiltinTool object by acquiring the underlying built-in Illustrator tool
					\author		GW
					\date		07/2014
					
					\param		tool_	Enum value representing the desired tool
					
					\warning	Some built-in tools are truly built into Illustrator, such that they do not have an
								associated plugin and also do not have an associated AIToolHandle. In this case, some
								class methods will not be functional. To determine this situation, call the isEmpty()
								method and the aiToolHandle() method, looking for false and NULL, respectively.
				*/
				BuiltinTool(const AdobeTool tool_);

				/**
					\brief	Destructs a BuiltinTool object
					\author	GW
					\date	07/2014
				*/
				virtual ~BuiltinTool();
				
				/**
					\brief		Assigns one BuiltinTool object to another
					\author		GW
					\date		07/2014

					\param		rhs_	Existing BuiltinTool object to copy values from
					\returns	The target BuiltinTool object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual BuiltinTool& operator=(const BuiltinTool& rhs_);
				
				/**
					\brief		Tests whether a given BuiltinTool object is the same as another
					\author		GW
					\date		07/2014

					\param		rhs_	BuiltinTool to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same tool, false otherwise
				*/
				virtual bool operator==(const BuiltinTool& rhs_) const;
				
				/**
					\brief		Tests whether a given BuiltinTool object is not the same as another
					\author		GW
					\date		07/2014

					\param		rhs_	BuiltinTool to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different tools, false otherwise
				*/
				virtual bool operator!=(const BuiltinTool& rhs_) const;

				/**
					\brief		Gets the tool handle around which the target object is wrapped
					\author		GW
					\date		07/2014
					
					\returns	AIToolHandle for this object

					\note		Generally, you shouldn't need to call this method. Only use it if you know what you're
								doing. If a specific piece of functionality provided by Illustrator is not handled by
								this class (or related classes), then it should probably be added to the hdi_core library.
				*/
				virtual AIToolHandle aiToolHandle() const;

				/**
					\brief		Gets whether the target BuiltinTool object is empty (constructed with the default ctor)
					\author		GW
					\date		07/2014
					
					\returns	true if the target BuiltinTool object is empty, false otherwise
				*/
				virtual bool isEmpty() const;
				
				/**
					\brief		Gets the plugin associated with the tool, if any
					\author		GW
					\date		09/2014
					
					\returns	A new ThirdPartyPlugin object, or NULL for none/error
				*/
				virtual std::unique_ptr<ThirdPartyPlugin> plugin() const;
				
				/**
					\brief		Gets whether the target BuiltinTool object is the currently selected tool
					\author		GW
					\date		07/2014
					
					\returns	true if the tool is selected, false otherwise
				*/
				virtual bool selected() const;
				
				/**
					\brief	Sets the target BuiltinTool object as the currently selected tool
					\author	GW
					\date	09/2014
				*/
				virtual void setSelected();

				/**
					\brief		Gets the tool's title
					\author		GW
					\date		07/2014

					\returns	The title for the tool, as UTF-8
				*/
				virtual std::string title() const;

				/**
					\brief		Gets the tool's tip string
					\author		GW
					\date		07/2014

					\returns	The rollover tip for the tool, as UTF-8
				*/
				virtual std::string tip() const;


			private:
				AIToolHandle __toolHandle;
				std::string __toolName;
		};
		
		typedef std::unique_ptr<BuiltinTool> BuiltinToolUP;
		typedef std::shared_ptr<BuiltinTool> BuiltinToolSP;
		typedef std::weak_ptr<BuiltinTool> BuiltinToolWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_BUILTIN_TOOL__
