/**
	\file
	\brief		Header file for plugin user interface checkbox widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_CHECKBOX__
#define __HDI_CORE_CHECKBOX__

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;

		/**
			\brief	Checkbox widget, which has both a checkbox "button" and a text label to its right
		*/
		class Checkbox : public Widget
		{
			public:
				/**
					\brief	Describes the state of a checkbox
				*/
				enum StateType
				{
					StateMixed	= -1,
					StateOff	= 0,
					StateOn		= 1
				};

				/**
					\brief	Constructs an empty Checkbox object
					\author	GW
					\date	09/2013
					
					\note	To test if a Checkbox object is empty, call isEmpty() on it
					\note	Empty Checkbox objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other Checkbox object via the overloaded assignment operator. Empty
							Checkbox objects are useless until such time (though it is safe to call any of their methods).
				*/
				Checkbox();

				/**
					\brief	Constructs a new Checkbox object from an existing Checkbox object (copy constructor)
					\author	GW
					\date	09/2013

					\param	checkbox_	Existing Checkbox object
				*/
				Checkbox(const Checkbox& checkbox_);

				/**
					\brief	Constructs a checkbox (using a StateType) and a window container type
					\author	GW
					\date	09/2013

					\param	loc_			Top-left location of the checkbox, in 1x resolution coordinates
					\param	label_			String for the checkbox, as UTF-8
					\param	containerType_	Type of window that will contain the widget
					\param	state_			Initial checkbox state
					
					\note	The containerType_ argument is necessary for this constructor because it computes the ideal
							width for the new checkbox from the initial checkbox label string. However, Adobe uses
							different font sizes for different types of windows, and we can't correctly compute the
							width without knowing what font size to use.
				*/
				explicit Checkbox(
					const Point& loc_,
					const std::string& label_,
					const WindowType containerType_,
					const StateType state_ = StateOff
				);

				/**
					\brief	Constructs a checkbox (using a StateType)
					\author	GW
					\date	09/2013

					\param	loc_	Top-left location of the checkbox, in 1x resolution coordinates
					\param	label_	String for the checkbox, as UTF-8
					\param	width_	Width of the checkbox and its label, at 1x resolution
					\param	state_	Initial checkbox state
				*/
				explicit Checkbox(
					const Point& loc_,
					const std::string& label_,
					const double width_,
					const StateType state_ = StateOff
				);
				
				/**
					\brief	Checkbox destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~Checkbox();

				/**
					\brief		Allows one Checkbox object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target Checkbox object, but with its value updated to match that of rhs_
				*/
				virtual Checkbox& operator=(const Checkbox& rhs_);
				
				/**
					\brief		Convenience method to clone a Checkbox object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new Checkbox object
					
					\note		If you subclass Checkbox, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your Checkbox subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned Checkbox object.
				*/
				virtual Checkbox* clone() const;
				
				/**
					\brief		Convenience method to duplicate a Checkbox object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new Checkbox object (and new UI element)
					
					\warning	The caller must manage the memory for the returned Checkbox object.
				*/
				virtual Checkbox* duplicate() const;

				/**
					\brief		Gets the checkbox state
					\author		GW
					\date		09/2013

					\returns	Current checkbox state
				*/
				virtual StateType state() const;
				
				/**
					\brief	Sets the checkbox state
					\author	GW
					\date	09/2013

					\param	state_	New checkbox state
				*/
				virtual void setState(const StateType state_);

				/**
					\brief	Alternate setter for the checkbox state
					\author	GW
					\date	09/2013

					\param	checked_	true if the checkbox is checked, false otherwise

					\note	If you need to set a mixed state, use the other setter
				*/
				virtual void setChecked(const bool checked_);

				/**
					\brief	Simulates a click on the checkbox
					\author	GW
					\date	09/2013
				*/
				virtual void click();
				
				/**
					\brief		Gets the current click callback for the checkbox
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered click callback
				*/
				virtual Callback* const clickCallback() const;
				
				/**
					\brief	Sets the click callback for the checkbox
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the checkbox is clicked
				*/
				virtual void setClickCallback(const Callback& callback_);

				/**
					\brief		Gets the ideal width for a checkbox, given its label and a known height
					\author		GW
					\date		09/2013

					\param		text_			Checkbox label, as UTF-8
					\param		height_			Known height into which the checkbox must fit
					\param		containerType_	Type of window for the widget (because font sizes differ)
					\returns	Ideal width for the given text to fit into the given height, at 1x resolution
				*/
				static double idealWidth(const std::string& text_, const double height_, const WindowType containerType_);
		};
		
		typedef std::unique_ptr<Checkbox> CheckboxUP;
		typedef std::shared_ptr<Checkbox> CheckboxSP;
		typedef std::weak_ptr<Checkbox> CheckboxWP;
	}
}

#endif
// __HDI_CORE_CHECKBOX__
