/**
	\file
	\brief		Header file for color preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_COLOR_PREF__
#define __HDI_CORE_COLOR_PREF__

#include "hdicoreColor.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		/**
			\brief	Allows for storage of colors
		*/
		class ColorPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty ColorPref object
					\author	GW
					\date	09/2013
					
					\note	To test if a ColorPref object is empty, call isEmpty() on it
					\note	Empty ColorPref objects do not relate to any actual preference data; they are designed to be
							"receivers" of some other ColorPref object via the overloaded assignment operator. Empty
							ColorPref objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				ColorPref();
			
				/**
					\brief	Constructs a new ColorPref object from an existing ColorPref object (copy constructor)
					\author	GW
					\date	09/2013

					\param	c_	Existing ColorPref object
				*/
				ColorPref(const ColorPref& c_);

				/**
					\brief	Constructs a ColorPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	09/2013

					\param	value_	Initial color value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				ColorPref(const Color& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a ColorPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~ColorPref();

				/**
					\brief		Allows one ColorPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ColorPref object, but with its value updated to match that of rhs_
				*/
				virtual ColorPref& operator=(const ColorPref& rhs_);
				
				/**
					\brief		Convenience method to clone an ColorPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ColorPref object
					
					\note		If you subclass ColorPref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ColorPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned ColorPref object.
				*/
				virtual ColorPref* clone() const;

				/**
					\brief		Gets the color for this object
					\author		GW
					\date		09/2013

					\returns	The Color object being described by this wrapper
				*/
				virtual Color color() const;

				/**
					\brief		Sets the color for this object
					\author		GW
					\date		01/2012

					\param		color_	New color value
					\returns	true if the color is successfully updated, or false otherwise
				*/
				virtual bool setColor(const Color& color_);
		};
		
		typedef std::unique_ptr<ColorPref> ColorPrefUP;
		typedef std::shared_ptr<ColorPref> ColorPrefSP;
		typedef std::weak_ptr<ColorPref> ColorPrefWP;
	}
}

#endif
// __HDI_CORE_COLOR_PREF__
