/**
	\file
	\brief		Header file for plugin user interface color well widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_COLOR_WELL__
#define __HDI_CORE_COLOR_WELL__

#if defined(HDI_CORE_AIP_MODE)
	#include "hdicoreArtColor.h"
#elif defined(HDI_CORE_PSP_MODE)
	#include "hdicoreColor.h"
#endif

#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		#if defined(HDI_CORE_AIP_MODE)
			typedef ArtColor PluginColor;
		#elif defined(HDI_CORE_PSP_MODE)
			typedef Color PluginColor;
		#endif

		class Callback;

		/**
			\brief	Class for color well widgets, which form a clickable square area that represents a color
		*/
		class ColorWell : public Widget
		{
			public:
				/**
					\brief	Constructs an empty ColorWell object
					\author	GW
					\date	09/2013
					
					\note	To test if a ColorWell object is empty, call isEmpty() on it
					\note	Empty ColorWell objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other ColorWell object via the overloaded assignment operator. Empty
							ColorWell objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				ColorWell();

				/**
					\brief	Constructs a new ColorWell object from an existing ColorWell object (copy constructor)
					\author	GW
					\date	09/2013

					\param	colorwell_	Existing ColorWell object
				*/
				ColorWell(const ColorWell& colorwell_);

				/**
					\brief	Constructs a color well
					\author	GW
					\date	09/2013

					\param	frame_	Dimensions and top-left location of the color well, in 1x resolution coordinates
					\param	color_	Initial value of the color well

					\note	The type of color with which a ColorWell is "seeded" (the color it has before the color
							picker is shown) will affect the type of color that is reported by the currentColor() method
							(e.g. in a value-changed callback). It is recommended that you set the current color of a
							color well with a color whose type matches that of the current document's color model.
				*/
				ColorWell(const Rect& frame_, const PluginColor& color_);

				/**
					\brief	ColorWell destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~ColorWell();

				/**
					\brief		Allows one ColorWell object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target ColorWell object, but with its value updated to match that of rhs_
				*/
				virtual ColorWell& operator=(const ColorWell& rhs_);
				
				/**
					\brief		Convenience method to clone a ColorWell object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new ColorWell object
					
					\note		If you subclass ColorWell, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your ColorWell subclass, you will
								experience "object slicing" when adding the widget to a window.
								
					\warning	The caller must manage the memory for the returned ColorWell object.
				*/
				virtual ColorWell* clone() const;
				
				/**
					\brief		Convenience method to duplicate a ColorWell object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new ColorWell object (and new UI element)
					
					\warning	The caller must manage the memory for the returned ColorWell object.
				*/
				virtual ColorWell* duplicate() const;
				
				/**
					\brief		Shows a color picker and provides the chosen color, unless the user canceled
					\author		GW
					\date		01/2025
					
					\param		inColor_	The initial color to display in the picker
					\param		outColor__	Return-by-refence for the color chosen by the user
					\returns	true if a color was chosen, false otherwise; ignore the color__ param value if false
								is returned
				*/
				static bool showColorPicker(const PluginColor& inColor_, PluginColor& outColor__);

				/**
					\brief		Gets whether any color picker dialog is currently showing due to a color well being
								clicked or the showColorPicker() method being called
					\author		GW
					\date		12/2015
					
					\returns	true for color picker dialog showing, false otherwise
				*/
				static bool colorPickerShowing();
			
				/**
					\brief		Gets the current value of the color well
					\author		GW
					\date		09/2013

					\returns	An ArtColor object representing the current color of the color well
				*/
				virtual PluginColor currentColor() const;

				/**
					\brief	Sets the current value of the color well
					\author	GW
					\date	09/2013

					\param	color_	New color values for the color well

					\note	The type of color with which a ColorWell is "seeded" (the color it has before the color
							picker is shown) will affect the type of color that is reported by the currentColor() method
							(e.g. in a value-changed callback). It is recommended that you set the current color of a
							color well with a color whose type matches that of the current document's color model.
				*/
				virtual void setCurrentColor(const PluginColor& color_);

				/**
					\brief	Simulates the color well having its value changed
					\author	GW
					\date	09/2013
				*/
				virtual void valueChanged();
				
				/**
					\brief		Gets the value changed callback for the color well
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered value changed callback
				*/
				virtual Callback* const valueChangedCallback() const;
				
				/**
					\brief	Sets the value changed callback
					\author	GW
					\date	09/2013

					\param	callback_	New callback for when the color well's value has changed
				*/
				virtual void setValueChangedCallback(const Callback& callback_);
		};
		
		typedef std::unique_ptr<ColorWell> ColorWellUP;
		typedef std::shared_ptr<ColorWell> ColorWellSP;
		typedef std::weak_ptr<ColorWell> ColorWellWP;
	}
}

#endif
// __HDI_CORE_COLORWELL__
