/**
	\file
	\brief		Header file for creating entry objects with multiple columns, to be added to Widget subclasses (that
				support them)
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_COLUMN_ENTRY__
#define __HDI_CORE_COLUMN_ENTRY__

#include <vector>

#include "hdicoreMacros.h"

#include "hdicoreEntryValue.h"

namespace hdi
{
	namespace pui
	{
		class ColumnEntryData;
	}
	
	namespace core
	{
		/**
			\brief	Allows column views to store more data than simply strings and indices
		*/
		class ColumnEntry
		{
			public:
				typedef std::vector<EntryValue> ValueVector;
				typedef std::vector<bool> EditableVector;
				typedef std::vector< void* > DataVector;
			
				/**
					\brief	When you want a separator in a container widget, create a ColumnEntry object with this
							as its string value
				*/
				static const char* separatorValue;
			
				/**
					\brief	Constructs an empty ColumnEntry object
					\author	GW
					\date	01/2015
					
					\note	To test if a ColumnEntry object is empty, call isEmpty() on it
					\note	Empty ColumnEntry objects do not relate to any entry in any widget; they are designed to
							be "receivers" of some other ColumnEntry object via the overloaded assignment operator.
							Empty ColumnEntry objects are useless until such time (though it is safe to call any of
							their methods).
				*/
				ColumnEntry();
			
				/**
					\brief	Constructs a new ColumnEntry object from an existing ColumnEntry object (copy
							constructor)
					\author	GW
					\date	01/2015

					\param	e_	Existing ColumnEntry object
				*/
				ColumnEntry(const ColumnEntry& e_);

				/**
					\brief	Constructs a ColumnEntry object with some custom data
					\author	GW
					\date	01/2015

					\param	id_			Entry ID string, as UTF-8 (must form a unique ID path in the target container)
					\param	values_		Entry value objects, one for each column
					\param	userData_	Data to store in each column of the entry
					\param	enabled_	Whether the entry should be enabled
					\param	editable_	Whether each cell in the entry should be inline editable
				*/
				ColumnEntry(
					const std::string& id_,
					const ValueVector& values_,
					const DataVector& userData_,
					const bool enabled_,
					const EditableVector& editable_
				);

				/**
					\brief	Destructs a ColumnEntry object
					\author	GW
					\date	01/2015
				*/
				virtual ~ColumnEntry();
				
				/**
					\brief		Assigns one ColumnEntry object to another
					\author		GW
					\date		01/2015

					\param		rhs_	Existing ColumnEntry object to copy values from
					\returns	The target ColumnEntry object, but with its values updated to match that of the rhs_
								argument
				*/
				virtual ColumnEntry& operator=(const ColumnEntry& rhs_);

				/**
					\brief		Gets whether the target ColumnEntry object is empty (constructed with the default
								ctor)
					\author		GW
					\date		01/2015
					
					\returns	true if the target ColumnEntry object is empty, false otherwise
				*/
				virtual bool isEmpty() const;

				/**
					\brief		Gets the ID string of an entry
					\author		GW
					\date		01/2015

					\returns	The unique ID of the entry, as UTF-8
				*/
				virtual std::string entryID() const;
				
				/**
					\brief	Sets the ID string of an entry
					\author	GW
					\date	01/2015

					\param	id_		New entry ID string, as UTF-8

					\note	It is up to the caller to ensure that the ID path is not in use with other entries within the
							same container widget.
				*/
				virtual void setEntryID(const std::string& id_);
				
				/**
					\brief		Gets the value objects of an entry
					\author		GW
					\date		01/2015

					\returns	The entry's value objects
				*/
				virtual ValueVector values() const;
				
				/**
					\brief	Sets the value objects of an entry
					\author	GW
					\date	01/2015

					\param	values_		New entry value objects
				*/
				virtual void setValues(const ValueVector& values_);

				/**
					\brief		Gets whether the entry should be enabled (selectable by the user in the list)
					\author		GW
					\date		01/2015
					
					\returns	true for enabled, false otherwise
				*/
				virtual bool enabled() const;

				/**
					\brief	Sets whether the entry should be enabled (selectable by the user in the list)
					\author	GW
					\date	01/2015
					
					\param	enabled_	true for enabled, false otherwise
				*/
				virtual void setEnabled(const bool enabled_);
				
				/**
					\brief		Gets the editability of each cell in the entry (whether the user can edit it inline)
					\author		GW
					\date		07/2022
					
					\returns	true for enabled, false otherwise
				*/
				virtual EditableVector editable() const;

				/**
					\brief	Sets the editability of each cell in the entry (whether the user can edit it inline)
					\author	GW
					\date	07/2022
					
					\param	editable_	true for enabled, false otherwise
				*/
				virtual void setEditable(const EditableVector& editable_);

				/**
					\brief		Gets the custom data of an entry
					\author		GW
					\date		01/2015

					\returns	The data stored inside the target entry object
				*/
				virtual DataVector userData() const;
				
				/**
					\brief	Sets the custom data of an entry
					\author	GW
					\date	01/2015

					\param	data_	New custom data to store
				*/
				virtual void setUserData(const DataVector& data_);
				
				/**
					\brief		Tests whether a given ColumnEntry object is the same as another
					\author		GW
					\date		01/2015

					\param		rhs_	ColumnEntry to compare against (righthand side of equality operator)
					\returns	true for the target and rhs_ being the same entry, false otherwise
				*/
				virtual bool operator==(const ColumnEntry& rhs_) const;
				
				/**
					\brief		Tests whether a given ColumnEntry object is not the same as another
					\author		GW
					\date		01/2015

					\param		rhs_	ColumnEntry to compare against (righthand side of inequality operator)
					\returns	true for the target and rhs_ being different entries, false otherwise
				*/
				virtual bool operator!=(const ColumnEntry& rhs_) const;


			private:
				friend pui::ColumnEntryData* __accessImpl(const ColumnEntry&);

				/**
					\brief	Private implementation data
				*/
				void* __data;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	09/2013
				*/
				void* __impl() const;
		};
		
		typedef std::unique_ptr<ColumnEntry> ColumnEntryUP;
		typedef std::shared_ptr<ColumnEntry> ColumnEntrySP;
		typedef std::weak_ptr<ColumnEntry> ColumnEntryWP;
		
		extern pui::ColumnEntryData* __accessImpl(const ColumnEntry&);
	}
}

#endif
// __HDI_CORE_COLUMN_ENTRY__
