/**
	\file
	\brief		Header file for plugin user interface custom widgets
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_CUSTOM_WIDGET__
#define __HDI_CORE_CUSTOM_WIDGET__

#include <vector>

#include "hdicoreColor.h"
#include "hdicoreWidget.h"

namespace hdi
{
	namespace core
	{
		class Callback;
		class Font;
		
		#if defined(HDI_CORE_AIP_MODE)
			class Panel;
		#endif
		
		#if defined(HDI_CORE_PSP_MODE)
			class PixelMap;
		#endif
		
		/**
			\brief	Class for custom plugin widgets; receives notification of a variety of events, can be custom drawn,
					etc.
		*/
		class CustomWidget : public Widget
		{
			public:
				typedef std::vector<Point> PointVector;
			
				/**
					\brief	Constructs an empty CustomWidget object
					\author	GW
					\date	09/2013
					
					\note	To test if a CustomWidget object is empty, call isEmpty() on it
					\note	Empty CustomWidget objects do not relate to any actual UI widget; they are designed to be
							"receivers" of some other CustomWidget object via the overloaded assignment operator. Empty
							CustomWidget objects are useless until such time (though it is safe to call any of their
							methods).
				*/
				CustomWidget();

				/**
					\brief	Constructs a new CustomWidget object from an existing CustomWidget object (copy constructor)
					\author	GW
					\date	09/2013

					\param	widget_		Existing CustomWidget object

					\note	This ctor does not perform a "deep copy" as it knows nothing about any potential subclass it
							might have. If you've subclassed CustomWidget, create your own copy ctor and use it instead.
				*/
				CustomWidget(const CustomWidget& widget_);

				/**
					\brief	CustomWidget constructor, taking an argument for the frame which will receive events
					\author	GW
					\date	09/2013

					\param	frame_	Top-left location and dimensions of the custom widget, in 1x resolution coordinates
				*/
				CustomWidget(const Rect& frame_);
				
				/**
					\brief	CustomWidget destructor
					\author	GW
					\date	09/2013
				*/
				virtual ~CustomWidget();

				/**
					\brief		Allows one CustomWidget object to be assigned from another
					\author		GW
					\date		10/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target CustomWidget object, but with its value updated to match that of rhs_
				*/
				virtual CustomWidget& operator=(const CustomWidget& rhs_);
				
				/**
					\brief		Convenience method to clone a CustomWidget object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new CustomWidget object
					
					\note		If you subclass CustomWidget, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your CustomWidget subclass, you will
								experience "object slicing" when adding the widget to a window.

					\warning	The caller must manage the memory for the returned CustomWidget object.
				*/
				virtual CustomWidget* clone() const;
				
				/**
					\brief		Convenience method to duplicate a CustomWidget object, creating a new and identical UI
								element to the target (but not belonging to the same containing window)
					\author		GW
					\date		11/2013
					
					\returns	A pointer to the new CustomWidget object (and new UI element)
					
					\warning	The caller must manage the memory for the returned CustomWidget object.
				*/
				virtual CustomWidget* duplicate() const;

				/**
					\brief	Forces the widget to redraw, calling the appropriate callbacks to do so
					\author	GW
					\date	09/2013
				*/
				virtual void redraw();

				/**
					\brief		Gets the modifier keys used during a click/drag event
					\author		GW
					\date		09/2013

					\returns	A bitwise OR of the ModifierKey enum values for the current modifier key state
					
					\note		The return value of this method is only valid during a click or drag event.
				*/
				virtual ModifierKey getClickDragModifiers() const;
				
				/**
					\brief		Gets the mouse locations during a click/drag event
					\author		GW
					\date		06/2015

					\param		start__		Return-by-reference for the position of the mouse when drag started,
											in 1x resolution coordinates
					\param		end__		Return-by-reference for the position of the mouse where drag is currently,
											in 1x resolution coordinates
					\returns	true if the values could be acquired, false otherwise
					
					\note		The return values of this method are only valid during a click or drag event.
				*/
				virtual bool getClickDragLocations(Point& start__, Point& end__) const;
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the delta positions during a scrolling event
						\author		GW
						\date		03/2018
					 
						\param		dx__	Return-by-reference for the x-axis delta value
						\param		dy__	Return-by-reference for the y-axis delta value
						\returns	true if the values could be acquired, false otherwise
					 
						\note		The return values of this method are only valid during a scroll event.
						\note		There is an inherent difference in how Mac and Windows report scrolling deltas. On Mac,
									the value indicates the number of pixels the content should move. On Windows, the value
									indicates the number of "lines" that the content should be adjusted by (usually +/-1).
									On both platforms, the sign of the value indicates the direction on the relevant axis.
					*/
					virtual bool getScrollDeltas(double& dx__, double& dy__) const;
				#endif
				// HDI_CORE_PSP_MODE

				/**
					\brief		Indicates whether the mouse is currently over the widget (useful for implementing
								rollover effects, if desired)
					\author		GW
					\date		06/2015

					\returns	true if the cursor if over the widget bounds, false otherwise
				*/
				virtual bool isMouseOver() const;

				/**
					\brief		Gets the current mouse over callback
					\author		GW
					\date		11/2015

					\returns	A pointer to the currently registered mouse over callback
				*/
				virtual Callback* const mouseOverCallback() const;

				/**
					\brief	Sets the mouse over callback
					\author	GW
					\date	11/2015

					\param	cb_		New callback (target and action) for when the widget receives a mouse over event
				*/
				virtual void setMouseOverCallback(const Callback& cb_);

				/**
					\brief		Gets the current mouse move callback
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered mouse move callback
				*/
				virtual Callback* const mouseMoveCallback() const;

				/**
					\brief	Sets the mouse move callback
					\author	GW
					\date	09/2013

					\param	cb_		New callback (target and action) for when the widget receives a mouse move
				*/
				virtual void setMouseMoveCallback(const Callback& cb_);

				/**
					\brief		Gets the current mouse out callback
					\author		GW
					\date		11/2015

					\returns	A pointer to the currently registered mouse out callback
				*/
				virtual Callback* const mouseOutCallback() const;

				/**
					\brief	Sets the mouse out callback
					\author	GW
					\date	11/2015

					\param	cb_		New callback (target and action) for when the widget receives a mouse out event
				*/
				virtual void setMouseOutCallback(const Callback& cb_);

				/**
					\brief		Gets whether the widget automatically takes focus on click (mouse down, specifically)
					\author		GW
					\date		07/2018

					\returns	true if the focus is taken when clicked, false otherwise
				*/
				virtual bool focusesOnClick() const;

				/**
					\brief	Sets whether the widget automatically takes focus on click (mouse down, specifically)
					\author	GW
					\date	07/2018

					\param	focus_	true to take focus when clicked, false otherwise

					\note	The default upon construction of a CustomWidget object is to take focus when clicked. If this
							behavior is undesired, call this method and pass false.
				*/
				virtual void setFocusesOnClick(const bool focus_);

				/**
					\brief		Gets the current mouse down callback
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered mouse down callback
				*/
				virtual Callback* const mouseDownCallback() const;

				/**
					\brief	Sets the mouse down callback
					\author	GW
					\date	09/2013

					\param	cb_		New callback (target and action) for when the widget receives a mouse down
				*/
				virtual void setMouseDownCallback(const Callback& cb_);

				/**
					\brief		Gets the current mouse drag callback
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered mouse drag callback
				*/
				virtual Callback* const mouseDragCallback() const;

				/**
					\brief	Sets the mouse drag callback
					\author	GW
					\date	09/2013

					\param	cb_		New callback (target and action) for when the widget receives a mouse drag
				*/
				virtual void setMouseDragCallback(const Callback& cb_);

				/**
					\brief		Gets the current mouse up callback
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered mouse up callback
				*/
				virtual Callback* const mouseUpCallback() const;

				/**
					\brief	Sets the mouse up callback
					\author	GW
					\date	09/2013

					\param	cb_		New callback (target and action) for when the widget receives a mouse up
				*/
				virtual void setMouseUpCallback(const Callback& cb_);
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief		Gets the current scrolling callback
						\author		GW
						\date		03/2018
					 
						\returns	A pointer to the currently registered scroll callback
					*/
					virtual Callback* const scrollCallback() const;
			
					/**
						\brief	Sets the scrolling callback
						\author	GW
						\date	03/2018

						\param	cb_		New callback (target and action) for when the widget receives a scroll event
					*/
					virtual void setScrollCallback(const Callback& cb_);
				#endif
				// HDI_CORE_PSP_MODE
			
				/**
					\brief		Gets whether dragging support for this particular widget is enabled
					\author		GW
					\date		09/2018
				 
					\returns	true if drag-and-drop source support is enabled, false otherwise
				*/
				virtual bool draggable() const;
			
				/**
					\brief	Sets whether dragging support for this particular widget is enabled
					\author	GW
					\date	09/2018
				 
					\param	enable_		true to enable drag-and-drop source support, false otherwise
				*/
				virtual void setDraggable(const bool enable_);
			
				/**
					\brief		Gets the data that will be passed when the widget is dropped on a receiver
					\author		GW
					\date		09/2018
				 
					\returns	The droppable data
				 
					\note	A value of "" (empty string; the default) will result in the text value of the widget itself
							being the drag data.
				*/
				virtual std::string dragData() const;
			
				/**
					\brief	Sets the data that will be passed when the widget is dropped on a receiver
					\author	GW
					\date	09/2018
				 
					\param	data_	Data to be passed to a receiving drag-and-drop widget (when dropped)
				 
					\note	Passing a value of "" (empty string; the default) will result in the text value of the widget
							itself being the drag data.
				*/
				virtual void setDragData(const std::string& data_);
			
				/**
					\brief		Gets whether dropping support for this particular widget is enabled
					\author		GW
					\date		09/2018
				 
					\returns	true if drag-and-drop destination support is enabled, false otherwise
				*/
				virtual bool droppable() const;
			
				/**
					\brief	Sets whether dropping support for this particular widget is enabled
					\author	GW
					\date	09/2018
				 
					\param	enable_		true to enable drag-and-drop destination support, false otherwise
				*/
				virtual void setDroppable(const bool enable_);
			
				/**
					\brief		Gets the current potential-drop callback for the widget (for when the mouse is moving
								over the widget, in the middle of a drag-and-drop operation)
					\author		GW
					\date		06/2019

					\returns	A pointer to the currently registered potential-drop callback
				*/
				virtual Callback* const potentialDropCallback() const;
			
				/**
					\brief	Sets the potential-drop callback for the widget
					\author	GW
					\date	06/2019

					\param	callback_	New callback for when the widget receives a potential-drop event (when the mouse
								is moving over the widget, in the middle of a drag-and-drop operation)
				*/
				virtual void setPotentialDropCallback(const Callback& callback_);
			
				/**
					\brief		Gets the point at which the drop might occur, in the widget's coordinate system
					\author		GW
					\date		06/2019
				 
					\returns	The point at which the drop might occur
				 
					\warning	The result of this method is only valid when it is called from within a potential-drop
								callback.
				*/
				virtual Point potentialDropPoint() const;
			
				/**
					\brief		Gets the current drop callback for the widget
					\author		GW
					\date		09/2018

					\returns	A pointer to the currently registered drop callback
				*/
				virtual Callback* const dropCallback() const;
			
				/**
					\brief	Sets the drop callback for the widget
					\author	GW
					\date	09/2018

					\param	callback_	New callback for when the widget receives a drop
				*/
				virtual void setDropCallback(const Callback& callback_);
			
				/**
					\brief		Gets the data that was passed when the widget received a drop
					\author		GW
					\date		09/2018
				 
					\returns	The dropped data
				 
					\warning	The result of this method is only valid when it is called from within a drop callback.
				*/
				virtual std::string droppedData() const;
			
				/**
					\brief		Gets the point at which the drop occurred, in the widget's coordinate system
					\author		GW
					\date		09/2018
				 
					\returns	The point at which the drop occurred
				 
					\warning	The result of this method is only valid when it is called from within a drop callback.
				*/
				virtual Point droppedPoint() const;
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets the panel that the target custom widget has been associated with
						\author		GW
						\date		09/2018
					 
						\returns	Panel that will be shown in a popdown borderless fashion whenever the custom widget
									implementer wants
					*/
					virtual std::unique_ptr<Panel> linkedPanel() const;
			
					/**
						\brief	Sets the panel that can be shown when the whenever the custom widget implementer wants
						\author	GW
						\date	09/2018
					 
						\param	p_	New panel to show in a popdown borderless fashion
					*/
					virtual void setLinkedPanel(const Panel& p_);
			
					/**
						\brief	Removes the panel that should be shown when the label is clicked
						\author	GW
						\date	09/2018
					*/
					virtual void removeLinkedPanel();
			
					/**
						\brief	Shows the linked panel in a popdown borderless fashion at the given location
						\author	GW
						\date	09/2018
					 
						\param	pt_		Location of the panel, in local coordinates
						\param	focus_	Widget within the linked panel to focus upon showing
					*/
					virtual void showLinkedPanel(const Point& pt_, const Widget* const focus_);
			
					/**
						\brief	Hides the linked panel if it is currently showing
						\author	GW
						\date	01/2025
					*/
					virtual void hideLinkedPanel();
				#endif

				/**
					\brief		Gets the platform-specific graphics drawing context for the current draw call
					\author		GW
					\date		11/2016

					\returns	The current context if in the middle of the drawing callback execution, NULL otherwise
				*/
				virtual PlatformDrawingContextPtr platformDC() const;

				/**
					\brief		Gets the current draw callback
					\author		GW
					\date		09/2013

					\returns	A pointer to the currently registered draw callback
				*/
				virtual Callback* const drawCallback() const;

				/**
					\brief	Sets the draw callback
					\author	GW
					\date	09/2013

					\param	cb_		New callback (target and action) for when the widget show draw itself
				*/
				virtual void setDrawCallback(const Callback& cb_);

				/**
					\brief		Gets whether the widget automatically clears the frame (i.e. paints the background color
								over the frame) before calling the draw callback
					\author		GW
					\date		07/2014

					\returns	true if the frame is cleared, false otherwise
				*/
				virtual bool clearsOnRedraw() const;

				/**
					\brief	Sets whether the widget automatically clears the frame (i.e. paints the background color
							over the frame) before calling the draw callback
					\author	GW
					\date	07/2014

					\param	clear_	true to clear the frame, false otherwise

					\note	The default upon construction of a CustomWidget object is to clear the frame. If this
							behavior is undesired, call this function and pass false.
				*/
				virtual void setClearsOnRedraw(const bool clear_);
				
				/**
					\brief	Sets the color for the current widget drawing context
					\author	GW
					\date	09/2013

					\param	color_	New drawing context color

					\note	This method is only functional during a widget's draw callback (set via the setDrawCallback()
							method)
				*/
				virtual void setDrawColor(const Color& color_);

				/**
					\brief	Draws a line from a given point to an endpoint; will be the color set by setDrawColor()
					\author	GW
					\date	09/2013

					\param	from_	Start point for the line, in local coordinates and at 1x resolution
					\param	to_		End point for the line, in local coordinates

					\note	This method is only functional during a widget's draw callback (set via the setDrawCallback()
							method)
				*/
				virtual void drawLine(const Point& from_, const Point& to_);

				/**
					\brief	Draws a rect, optionally filled; will be the color set by setDrawColor()
					\author	GW
					\date	09/2013

					\param	rect_	Rectangle to draw, in local coordinates and at 1x resolution
					\param	fill_	Whether to fill the rectangle

					\note	This method is only functional during a widget's draw callback (set via the setDrawCallback()
							method)
				*/
				virtual void drawRect(const Rect& rect_, const bool fill_ = false);

				/**
					\brief	Draws a rounded rect, optionally filled; will be the color set by setDrawColor()
					\author	GW
					\date	10/2018

					\param	rect_		Rectangle to draw, in local coordinates and at 1x resolution
					\param	radius_		Radius for the corners at 1x resolution
					\param	fill_		Whether to fill the rectangle

					\note	This method is only functional during a widget's draw callback (set via the setDrawCallback()
							method)
					\note	If radius_ is larger than half of either the rectangle width or height, it will be reduced
							accordingly.
				*/
				virtual void drawRoundedRect(const Rect& rect_, const double radius_, const bool fill_ = false);

				/**
					\brief	Draws an ellipse, optionally filled; will be the color set by setDrawColor()
					\author	GW
					\date	01/2017

					\param	rect_	Rectangle in which to draw the ellipse, in local coordinates and at 1x resolution
					\param	fill_	Whether to fill the ellipse

					\note	This method is only functional during a widget's draw callback (set via the setDrawCallback()
							method)
				*/
				virtual void drawEllipse(const Rect& rect_, const bool fill_ = false);

				/**
					\brief	Draws a polygon composed of a given set of points; can optionally be filled with the current
							draw color
					\author	GW
					\date	09/2013

					\param	points_		Points that compose the polygon, in local coordinates and at 1x resolution
					\param	fill_		Whether to fill the polygon

					\note	This method is only functional during a widget's draw callback (set via the setDrawCallback()
							method)
				*/
				virtual void drawPolygon(const PointVector& points_, const bool fill_ = false);
			
				/**
					\brief	Draws a PNGI image resource at the given point
					\author	GW
					\date	11/2015
					
					\param	tl_			Top-left location of the image, in local coordinates and at 1x resolution
					\param	pngID_		PNGI image resource ID
					\param	scale_		On Mac, this indicates how much the image should be scaled to adjust for a HiDPI
										display. At this time, only values of 1.0 and 2.0 are accepted.

					\note	This method is only functional during a widget's draw callback (set via the setDrawCallback()
							method).
					\note	PNGI resource IDs from 0-5000 are reserved for hdi_core.
				*/
				virtual void drawImage(const Point& tl_, const int16_t pngID_, const double scale_ = 1.0);
			
				/**
					\brief	Draws text at the given point
					\author	GW
					\date	06/2018
				 
					\param	tl_			Top-left location of the text, in local coordinates and at 1x resolution
					\param	text_		Text value to be drawn, in UTF-8
					\param	font_		Font to use when drawing the string
					\param	fontSize_	Font size to use

					\note	This method is only functional during a widget's draw callback (set via the
							setDrawCallback() method)
				*/
				virtual void drawText(
					const Point& tl_,
					const std::string& text_,
					const Font& font_,
					const double fontSize_
				);
			
				#if defined(HDI_CORE_PSP_MODE)
					/**
						\brief	Draws the provided pixel data, typically provided by Photoshop itself originally (but
								might be further modified by your plugin), in the custom widget
						\author	GW
						\date	01/2018
						
						\param	srcPM_		Describes a block of pixels from Photoshop to display
						\param	srcArea_	Indicates the rectangle of the source pixel map to display
						\param	dstTL_		Provides the top-left destination pixel in the custom widget
						\param	dstArea_	Provides the destination rectangle for drawing the pixels (often matches
											srcArea_ - only Mac necessitates this argument; Windows can acquire this
											from the current drawing context)

						\note	This method is only functional during a widget's draw callback (set via the
								setDrawCallback() method), and only applies to drawing that is occuring during acquire,
								export, filter, format, and selection messages.
						\note	If you would like to draw into your own buffer, use the drawPixelData() method within the
								appropriate *Message class.
					*/
					virtual void drawPixelData(
						const PixelMap& srcPM_,
						const Rect& srcArea_,
						const Point& dstTL_,
						const Rect& dstArea_
					);
				#endif
				// HDI_CORE_PSP_MODE
		};
		
		typedef std::unique_ptr<CustomWidget> CustomWidgetUP;
		typedef std::shared_ptr<CustomWidget> CustomWidgetSP;
		typedef std::weak_ptr<CustomWidget> CustomWidgetWP;
	}
}

#endif
// __HDI_CORE_CUSTOM_WIDGET__
