/**
	\file
	\brief		Header file for dash style map preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_DASH_STYLE_MAP_PREF__
#define __HDI_CORE_DASH_STYLE_MAP_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace aip
	{
		class PersistentDictionary;
	}

	namespace core
	{
		/**
			\brief	Allows for storage of dash style maps
		*/
		class DashStyleMapPref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty DashStyleMapPref object
					\author	GW
					\date	11/2015
					
					\note	To test if a DashStyleMapPref object is empty, call isEmpty() on it
					\note	Empty DashStyleMapPref objects do not relate to any actual preference data; they are designed
							to be "receivers" of some other DashStyleMapPref object via the overloaded assignment operator.
							Empty DashStyleMapPref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				DashStyleMapPref();
			
				/**
					\brief	Constructs a new DashStyleMapPref object from an existing DashStyleMapPref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	ds_		Existing DashStyleMapPref object
				*/
				DashStyleMapPref(const DashStyleMapPref& ds_);

				/**
					\brief	Constructs a DashStyleMapPref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial dash style map value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				DashStyleMapPref(const DashStyle::Map& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a DashStyleMapPref object
					\author	GW
					\date	11/2015
				*/
				virtual ~DashStyleMapPref();

				/**
					\brief		Allows one DashStyleMapPref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target DashStyleMapPref object, but with its value updated to match that of rhs_
				*/
				virtual DashStyleMapPref& operator=(const DashStyleMapPref& rhs_);
				
				/**
					\brief		Convenience method to clone an DashStyleMapPref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new DashStyleMapPref object
					
					\note		If you subclass DashStyleMapPref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your DashStyleMapPref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned DashStyleMapPref object.
				*/
				virtual DashStyleMapPref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	DashStyleMapType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the dash style map value for this object
					\author		GW
					\date		11/2015

					\returns	The dash style map value being described by this wrapper
				*/
				virtual DashStyle::Map dashStyleMap() const;

				/**
					\brief		Sets the dash style map value for this object
					\author		GW
					\date		11/2015

					\param		value_	New dash style map value
					\returns	true if the dash style map is successfully updated, or false otherwise
				*/
				virtual bool setDashStyleMap(const DashStyle::Map& value_);


			private:
				friend class StrokeStyleMapPref;

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2015
				*/
				DashStyleMapPref(aip::PersistentDictionary*&);
		};
		
		typedef std::unique_ptr<DashStyleMapPref> DashStyleMapPrefUP;
		typedef std::shared_ptr<DashStyleMapPref> DashStyleMapPrefSP;
		typedef std::weak_ptr<DashStyleMapPref> DashStyleMapPrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_DASH_STYLE_MAP_PREF__
