/**
	\file
	\brief		Header file for dash style preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_DASH_STYLE_PREF__
#define __HDI_CORE_DASH_STYLE_PREF__

#if defined(HDI_CORE_AIP_MODE)

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace aip
	{
		class PersistentDictionary;
	}

	namespace core
	{
		/**
			\brief	Allows for storage of dash styles
		*/
		class DashStylePref : public PrefData
		{
			public:
				/**
					\brief	Constructs an empty DashStylePref object
					\author	GW
					\date	11/2015
					
					\note	To test if a DashStylePref object is empty, call isEmpty() on it
					\note	Empty DashStylePref objects do not relate to any actual preference data; they are designed to
							be "receivers" of some other DashStylePref object via the overloaded assignment operator.
							Empty DashStylePref objects are useless until such time (though it is safe to call any of
							their methods)
				*/
				DashStylePref();
			
				/**
					\brief	Constructs a new DashStylePref object from an existing DashStylePref object (copy constructor)
					\author	GW
					\date	11/2015

					\param	ds_		Existing DashStylePref object
				*/
				DashStylePref(const DashStylePref& ds_);

				/**
					\brief	Constructs a DashStylePref object, with new data, to be added to a container at a later time
					\author	GW
					\date	11/2015

					\param	value_	Initial dash style value
					\param	name_	Name for the data once it has been added to its container, as UTF-8

					\note	The name_ argument can only be "" (omitted) if the object will be added directly to an
							ArrayPref object.
				*/
				DashStylePref(const DashStyle& value_, const std::string& name_ = "");

				/**
					\brief	Destructs a DashStylePref object
					\author	GW
					\date	11/2015
				*/
				virtual ~DashStylePref();

				/**
					\brief		Allows one DashStylePref object to be assigned from another
					\author		GW
					\date		11/2015

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target DashStylePref object, but with its value updated to match that of rhs_
				*/
				virtual DashStylePref& operator=(const DashStylePref& rhs_);
				
				/**
					\brief		Convenience method to clone an DashStylePref object on the heap
					\author		GW
					\date		11/2015

					\returns	A pointer to the new DashStylePref object
					
					\note		If you subclass DashStylePref, you MUST overload this method yourself! If you don't and/or
								your clone() method does not return an instance of your DashStylePref subclass, you will
								experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned DashStylePref object.
				*/
				virtual DashStylePref* clone() const;

				/**
					\brief		Gets the type of persistent data
					\author		GW
					\date		11/2015

					\returns	DashStyleType

					\note		This is useful for knowing how to cast a PrefData object to its proper subclass
				*/
				virtual Type dataType() const;

				/**
					\brief		Gets the dash style value for this object
					\author		GW
					\date		11/2015

					\returns	The dash style value being described by this wrapper
				*/
				virtual DashStyle dashStyle() const;

				/**
					\brief		Sets the dash style value for this object
					\author		GW
					\date		11/2015

					\param		value_	New dash style value
					\returns	true if the dash style is successfully updated, or false otherwise
				*/
				virtual bool setDashStyle(const DashStyle& value_);


			private:
				friend class StrokeStylePref;

				/**
					\brief	Internal use only
					\author	GW
					\date	11/2015
				*/
				DashStylePref(aip::PersistentDictionary*&);
		};
		
		typedef std::unique_ptr<DashStylePref> DashStylePrefUP;
		typedef std::shared_ptr<DashStylePref> DashStylePrefSP;
		typedef std::weak_ptr<DashStylePref> DashStylePrefWP;
	}
}

#endif
// HDI_CORE_AIP_MODE

#endif
// __HDI_CORE_DASH_STYLE_PREF__
