/**
	\file
	\brief		Header file for dictionary preference data storage
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_DICTIONARY_PREF__
#define __HDI_CORE_DICTIONARY_PREF__

#include <vector>

#include "hdicorePathStyle.h"
#include "hdicorePrefData.h"

namespace hdi
{
	namespace core
	{
		class Color;
		class Font;
		class Point;
		class Rect;
		class Size;

		class ArrayPref;
		class BoolPref;
		class ColorPref;
		class FontPref;
		class NumberPref;
		class PointPref;
		class RectPref;
		class SizePref;
		class StringPref;
		
		#if defined(HDI_CORE_AIP_MODE)
			class ArtboardPoint;
			class ArtboardRect;
			class ArtColor;
			class TransformMatrix;
		
			class ArtboardPointPref;
			class ArtboardRectPref;
			class ArtColorPref;
			class DashStylePref;
			class DashStyleMapPref;
			class FillStylePref;
			class FillStyleMapPref;
			class PathStylePref;
			class PathStyleMapPref;
			class StrokeStylePref;
			class StrokeStyleMapPref;
			class TransformMatrixPref;
		#endif

		/**
			\brief	Acts as a dictionary-style container to store any other persistent data types, including arrays and
					dictionaries, and can itself be stored inside another container type.
		*/
		class DictionaryPref : public PrefData
		{
			public:
				typedef std::vector< std::shared_ptr<PrefData> > PrefDataVector;
				typedef std::vector< std::string > KeyVector;
			
				/**
					\brief	Allows for iteration through the members of a dictionary pref, a la std::vector iteration

					\code
						// Here's an example, iterating through the children of dict:
						for(hdi::core::DictionaryPref::Iterator iter = dict.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff with iter.key() or iter.value()
						}
					\endcode
				*/
				class Iterator
				{
					public:
						/**
							\brief	Constructs an Iterator object from an existing Iterator object (copy constructor)
							\author	GW
							\date	09/2014

							\param	iter_	Existing Iterator object
						*/
						Iterator(const Iterator& iter_);
					
						/**
							\brief	Destructs an Iterator object
							\author	GW
							\date	09/2014
						*/
						~Iterator();
						
						/**
							\brief		Assigns one Iterator object to another
							\author		GW
							\date		09/2014

							\param		rhs_	Existing Iterator object to copy values from
							\returns	The target Iterator object, but with its values updated to match that of the
										rhs_ argument
						*/
						Iterator& operator=(const Iterator& rhs_);
						
						/**
							\brief		Tests whether a given Iterator object is the same as another
							\author		GW
							\date		09/2014

							\param		rhs_	Iterator to compare against (righthand side of equality operator)
							\returns	true for the target and rhs_ having the same state, false otherwise
						*/
						bool operator==(const Iterator& rhs_) const;
						
						/**
							\brief		Tests whether a given Iterator object is not the same as another
							\author		GW
							\date		09/2014

							\param		rhs_	Iterator to compare against (righthand side of inequality operator)
							\returns	true for the target and rhs_ having differing states, false otherwise
						*/
						bool operator!=(const Iterator& rhs_) const;

						/**
							\brief		Prefix increment operator for forward iteration
							\author		GW
							\date		09/2014
							
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator++();

						/**
							\brief		Postfix increment operator for forward iteration
							\author		GW
							\date		09/2014
							
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator++(int);

						/**
							\brief		Prefix decrement operator for reverse iteration
							\author		GW
							\date		09/2014
							
							\returns	A reference to the iterator after its position was changed
						*/
						Iterator& operator--();

						/**
							\brief		Postfix decrement operator for reverse iteration
							\author		GW
							\date		09/2014
							
							\returns	A copy of the original iterator before its position was changed
						*/
						Iterator operator--(int);

						/**
							\brief		Determines whether iteration should stop; works well as the conditional in a
										for() loop
							\author		GW
							\date		09/2014
							
							\returns	true if at the end of the list of members, false otherwise
						*/
						bool atEnd() const;

						/**
							\brief		Gets the current member's key as pointed to by the iterator
							\author		GW
							\date		09/2014

							\returns	The current member's key, or "" if at the end
						*/
						std::string key() const;

						/**
							\brief		Gets the current member's value as pointed to by the iterator
							\author		GW
							\date		09/2014

							\returns	The current member's value, or NULL if at the end
						*/
						std::unique_ptr<PrefData> value() const;


					private:
						// Only DictionaryPref can construct a new Iterator object
						friend class DictionaryPref;

						/**
							\brief	Private implementation data
						*/
						void* __data;
				
						/**
							\brief	Empty Iterator objects cannot be constructed
							\author	GW
							\date	09/2014
						*/
						Iterator();
					
						/**
							\brief	Creates a new iterator for a DictionaryPref
							\author	GW
							\date	09/2014

							\param	dict_	Dictionary to be iterated through
						*/
						explicit Iterator(const DictionaryPref& dict_);
				};
		
				typedef std::unique_ptr<Iterator> IteratorUP;
				typedef std::shared_ptr<Iterator> IteratorSP;
				typedef std::weak_ptr<Iterator> IteratorWP;
				
				/**
					\brief	Constructs a new DictionaryPref object from an existing DictionaryPref object (copy
							constructor)
					\author	GW
					\date	09/2013

					\param	dict_	Existing DictionaryPref object
				*/
				DictionaryPref(const DictionaryPref& dict_);

				/**
					\brief		Constructs a DictionaryPref object, with new data, to be added to a container at a later
								time
					\author		GW
					\date		09/2013

					\param		members_	Initial dictionary members, if any
					\param		name_		Name for the data dictionary once it has been added to its container, as UTF-8

					\note		The name_ argument can only be "" (omitted) if the object will be added directly to an
								array.
				*/
				DictionaryPref(const PrefDataVector* members_ = NULL, const std::string& name_ = "");

				/**
					\brief	Destructs a DictionaryPref object
					\author	GW
					\date	09/2013
				*/
				virtual ~DictionaryPref();

				/**
					\brief		Allows one DictionaryPref object to be assigned from another
					\author		GW
					\date		09/2013

					\param		rhs_	Righthand side of the = operator; the object to copy values from
					\returns	The target DictionaryPref object, but with its value updated to match that of rhs_
				*/
				virtual DictionaryPref& operator=(const DictionaryPref& rhs_);
				
				/**
					\brief		Convenience method to clone an DictionaryPref object on the heap
					\author		GW
					\date		10/2013

					\returns	A pointer to the new DictionaryPref object
					
					\note		If you subclass DictionaryPref, you MUST overload this method yourself! If you don't
								and/or your clone() method does not return an instance of your DictionaryPref subclass,
								you will experience "object slicing" when adding the pref to a container.

					\warning	The caller must manage the memory for the returned DictionaryPref object.
				*/
				virtual DictionaryPref* clone() const;

				/**
					\brief		Gets a vector of all keys contained by the target dictionary
					\author		GW
					\date		02/2025
					
					\returns	A vector of all keys in the dictionary
				*/
				virtual KeyVector allKeys() const;

				/**
					\brief		Adds a PrefData object into the dictionary member tree at the top-level
					\author		GW
					\date		09/2013

					\param		data_	New data node to add into the top-level of the dictionary data tree
					\returns	true if the member was added, or false otherwise

					\note		The data is added to the top-most level of the dictionary data tree, with any children
								of its own appearing underneath the data_ member.
				*/
				virtual bool addMember(const PrefData& data_);

				/**
					\brief		Gets a single member object from the dictionary data tree
					\author		GW
					\date		09/2013

					\param		name_	Name of the member to get from the top-level of the dictionary data tree, as
										UTF-8
					\param		data__	Return-by-reference for the member found with the given name
					\returns	true if the member could be acquired, false otherwise

					\note		The type of the returned object can be determined from its dataType() method, such that
								correct casting can be accomplished.
				*/
				virtual bool member(const std::string& name_, std::unique_ptr<PrefData>& data__) const;

				/**
					\brief		Removes a PrefData member from the dictionary
					\author		GW
					\date		09/2013

					\param		name_	Name/key of the member to remove from the top-level of the dictionary data tree,
										as UTF-8
					\param		data__	Return-by-reference for the PrefData object for the member that was removed
					\returns	true if the member could be removed and acquired, false otherwise

					\note		The type of the returned object can be determined from its dataType() method, such that
								correct casting can be accomplished.
				*/
				virtual bool removeMember(const std::string& name_, std::unique_ptr<PrefData>& data__);

				/**
					\brief	Removes all members from the target dictionary
					\author	GW
					\date	02/2025
				*/
				virtual void clear();
				
				/**
					\brief		Gets an iterator for the dictionary
					\author		GW
					\date		09/2014
					
					\returns	An Iterator object, for iterating through the target's members
					
					\code
						// Here's an example, iterating through the children of dict:
						for(hdi::core::DictionaryPref::Iterator iter = dict.begin(); !iter.atEnd(); ++iter)
						{
							// do stuff with iter.key() or iter.value()
						}
					\endcode
				*/
				virtual Iterator begin() const;
				
				/**
					\brief		Gets a persistent array object from the containing dictionary
					\author		GW
					\date		09/2013

					\param		key_		UTF-8 name/key of the array JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
					\param		array__		Return-by-reference for the array member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getArrayMember(const std::string& key_, ArrayPref& array__) const;
				
				/**
					\brief		Similar to getArrayMember(), except that, if the member does not already exist, it will
								be created anew with a series of default elements
					\author		GW
					\date		09/2013

					\param		name_		UTF-8 name/key of the array JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
					\param		elements_	Default elements for the array, in case it didn't already exist (NULL, or
											an empty vector, for none)
					\param		array__		Return-by-reference for the array member identified by key_ (or created anew)
					\returns	true if the member could be acquired or created, false otherwise

					\note		The order of the default elements in the array will match that of the passed elements_
								argument.
				*/
				virtual bool getOrCreateArrayMember(
					const std::string& name_,
					const PrefDataVector* elements_,
					ArrayPref& array__
				);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent artboard point object from the dictionary
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the artboard point JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		abp__	Return-by-reference for the artboard point member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getArtboardPointMember(const std::string& key_, ArtboardPointPref& abp__) const;

					/**
						\brief		Similar to getArtboardPointMember(), except that, if the member does not already exist,
									it will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the artboard point JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		abp__		Return-by-reference for the artboard point member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtboardPointMember(
						const std::string& key_,
						const ArtboardPoint& default_,
						ArtboardPointPref& abp__
					);

					/**
						\brief		Gets a persistent artboard rect object from the dictionary
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the artboard rect JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		abr__	Return-by-reference for the artboard rect member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getArtboardRectMember(const std::string& key_, ArtboardRectPref& abr__) const;

					/**
						\brief		Similar to getArtboardRectMember(), except that, if the member does not already exist,
									it will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the artboard rect JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		abr__		Return-by-reference for the artboard rect member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtboardRectMember(
						const std::string& key_,
						const ArtboardRect& default_,
						ArtboardRectPref& abr__
					);

					/**
						\brief		Gets a persistent art color object from the dictionary
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the art color JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		ac__	Return-by-reference for the art color member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getArtColorMember(const std::string& key_, ArtColorPref& ac__) const;

					/**
						\brief		Similar to getArtColorMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the art color JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		ac__		Return-by-reference for the art color member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateArtColorMember(
						const std::string& key_, 
						const ArtColor& default_,
						ArtColorPref& ac__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent bool object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the bool JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		b__		Return-by-reference for the bool member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getBoolMember(const std::string& key_, BoolPref& b__) const;

				/**
					\brief		Similar to getBoolMember(), except that, if the member does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the bool JSON node to get from the top-level and return as
											a neatly-packed PrefData subclass
					\param		default_	Default value for the member, in case it didn't already exist
					\param		b__			Return-by-reference for the bool member identified by key_ (or created anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateBoolMember(
					const std::string& key_, 
					const bool default_,
					BoolPref& b__
				);

				/**
					\brief		Gets a persistent color object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the color JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		c__		Return-by-reference for the color member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getColorMember(const std::string& key_, ColorPref& c__) const;

				/**
					\brief		Similar to getColorMember(), except that, if the member does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF8 name/key of the color JSON node to get from the top-level and return as
											a neatly-packed PrefData subclass
					\param		default_	Default value for the member, in case it didn't already exist
					\param		c__			Return-by-reference for the color member identified by key_ (or created anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateColorMember(
					const std::string& key_, 
					const Color& default_,
					ColorPref& c__
				);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent dash style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the dash style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the dash style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getDashStyleMember(const std::string& key_, DashStylePref& ds__);
				
					/**
						\brief		Similar to getDashStyleMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the dash style JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		ds__		Return-by-reference for the dash style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateDashStyleMember(
						const std::string& key_,
						const DashStyle& default_,
						DashStylePref& ds__
					);
				
					/**
						\brief		Gets a persistent dash style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the dash style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		ds__	Return-by-reference for the dash style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getDashStyleMapMember(const std::string& key_, DashStyleMapPref& ds__);
				
					/**
						\brief		Similar to getDashStyleMapMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the dash style map JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		ds__		Return-by-reference for the dash style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateDashStyleMapMember(
						const std::string& key_,
						const DashStyle::Map& default_,
						DashStyleMapPref& ds__
					);
				#endif
				// HDI_CORE_AIP_MODE
			
				/**
					\brief		Gets a persistent dictionary object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the dictionary JSON node to get from the top-level and return
										as a neatly-packed PrefData subclass
					\param		dict__	Return-by-reference for the dictionary member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getDictMember(const std::string& key_, DictionaryPref& dict__) const;
				
				/**
					\brief		Similar to getDictMember(), except that, if the member does not already exist, it will
								be created anew with a series of default members
					\author		GW
					\date		09/2013
					
					\param		name_		UTF-8 name/key of the dictionary JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
					\param		members_	Default members for the dictionary, in case it didn't already exist (NULL,
											or an empty vector, for none)
					\param		dict__		Return-by-reference for the dictionary member identified by key_ (or created
											anew)
					\returns	true if the member could be acquired or created, false otherwise

					\note		The names of the members in the dictionary will match that of the individual members in
								the passed members_ argument.
				*/
				virtual bool getOrCreateDictMember(
					const std::string& name_,
					const PrefDataVector* members_,
					DictionaryPref& dict__
				);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent fill style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the fill style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		fs__	Return-by-reference for the fill style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getFillStyleMember(const std::string& key_, FillStylePref& fs__);
				
					/**
						\brief		Similar to getFillStyleMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the fill style JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		fs__		Return-by-reference for the fill style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateFillStyleMember(
						const std::string& key_,
						const FillStyle& default_,
						FillStylePref& fs__
					);
				
					/**
						\brief		Gets a persistent fill style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the fill style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		fs__	Return-by-reference for the fill style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getFillStyleMapMember(const std::string& key_, FillStyleMapPref& fs__);
				
					/**
						\brief		Similar to getFillStyleMapMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the fill style map JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		fs__		Return-by-reference for the fill style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateFillStyleMapMember(
						const std::string& key_,
						const FillStyle::Map& default_,
						FillStyleMapPref& fs__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent font object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the font JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		f__		Return-by-reference for the font member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getFontMember(const std::string& key_, FontPref& f__) const;

				/**
					\brief		Similar to getFontMember(), except that, if the member does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_			UTF-8 name/key of the font JSON node to get from the top-level and return
												as a neatly-packed PrefData subclass
					\param		defaultFont_	Default font for the member, in case it didn't already exist
					\param		defaultSize_	Default font size for the member, in case it didn't already exist
					\param		f__				Return-by-reference for the font member identified by key_ (or created anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateFontMember(
					const std::string& key_,
					const Font& defaultFont_,
					const double defaultSize_,
					FontPref& f__
				);

				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent matrix object from the dictionary
						\author		GW
						\date		09/2013
						
						\param		key_	UTF-8 name/key of the matrix JSON node to get from the top-level and return as a
											neatly-packed PrefData subclass
						\param		tm__	Return-by-reference for the matrix member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getTransformMatrixMember(const std::string& key_, TransformMatrixPref& tm__) const;

					/**
						\brief		Similar to getTransformMatrixMember(), except that, if the member does not already
									exist, it will be created anew with a default value
						\author		GW
						\date		09/2013
						
						\param		key_		UTF-8 name/key of the matrix JSON node to get from the top-level and return
												as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		tm__		Return-by-reference for the matrix member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateTransformMatrixMember(
						const std::string& key_, 
						const TransformMatrix& default_,
						TransformMatrixPref& tm__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent number object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the number JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		n__		Return-by-reference for the number member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getNumberMember(const std::string& key_, NumberPref& n__) const;

				/**
					\brief		Similar to getNumberMember(), except that, if the member does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the number JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
					\param		default_	Default integer value for the member, in case it didn't already exist
					\param		n__			Return-by-reference for the number member identified by key_ (or created
											anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateIntMember(const std::string& key_, const int64_t default_, NumberPref& n__);

				/**
					\brief		Similar to getNumberMember(), except that, if the member does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the number JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
					\param		default_	Default double value for the member, in case it didn't already exist
					\param		n__			Return-by-reference for the number member identified by key_ (or created
											anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateFloatMember(const std::string& key_, const double default_, NumberPref& n__);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent path style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the path style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		ps__	Return-by-reference for the path style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getPathStyleMember(const std::string& key_, PathStylePref& ps__);
				
					/**
						\brief		Similar to getPathStyleMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the path style JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		ps__		Return-by-reference for the path style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreatePathStyleMember(
						const std::string& key_,
						const PathStyle& default_,
						PathStylePref& ps__
					);
				
					/**
						\brief		Gets a persistent path style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the path style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		ps__	Return-by-reference for the path style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getPathStyleMapMember(const std::string& key_, PathStyleMapPref& ps__);
				
					/**
						\brief		Similar to getPathStyleMapMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the path style map JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		ps__		Return-by-reference for the path style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreatePathStyleMapMember(
						const std::string& key_,
						const PathStyle::Map& default_,
						PathStyleMapPref& ps__
					);
				#endif
				// HDI_CORE_AIP_MODE

				/**
					\brief		Gets a persistent point object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the point JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		p__		Return-by-reference for the point member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getPointMember(const std::string& key_, PointPref& p__) const;

				/**
					\brief		Similar to getPointMember(), except that, if the member does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the point JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
					\param		default_	Default value for the member, in case it didn't already exist
					\param		p__			Return-by-reference for the point member identified by key_ (or created anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreatePointMember(const std::string& key_, const Point& default_, PointPref& p__);

				/**
					\brief		Gets a persistent rect object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the rect JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		r__		Return-by-reference for the rect member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getRectMember(const std::string& key_, RectPref& r__) const;

				/**
					\brief		Similar to getRectMember(), except that, if the member does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the rect JSON node to get from the top-level and return as
											a neatly-packed PrefData subclass
					\param		default_	Default value for the member, in case it didn't already exist
					\param		r__			Return-by-reference for the rect member identified by key_ (or created anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateRectMember(const std::string& key_, const Rect& default_, RectPref& r__);

				/**
					\brief		Gets a persistent size object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the size JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		s__		Return-by-reference for the size member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getSizeMember(const std::string& key_, SizePref& s__) const;

				/**
					\brief		Similar to getSizeMember(), except that, if the member does not already exist, it will
								be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the size JSON node to get from the top-level and return as
											a neatly-packed PrefData subclass
					\param		default_	Default value for the member, in case it didn't already exist
					\param		s__			Return-by-reference for the size member identified by key_ (or created anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateSizeMember(const std::string& key_, const Size& default_, SizePref& s__);

				/**
					\brief		Gets a persistent string object from the dictionary
					\author		GW
					\date		09/2013
					
					\param		key_	UTF-8 name/key of the string JSON node to get from the top-level and return as a
										neatly-packed PrefData subclass
					\param		str__	Return-by-reference for the string member identified by key_
					\returns	true if the member could be acquired, false otherwise
				*/
				virtual bool getStringMember(const std::string& key_, StringPref& str__) const;

				/**
					\brief		Similar to getStringMember(), except that, if the member does not already exist, it
								will be created anew with a default value
					\author		GW
					\date		09/2013
					
					\param		key_		UTF-8 name/key of the string JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
					\param		default_	Default UTF-8 value for the member, in case it didn't already exist
					\param		str__		Return-by-reference for the string member identified by key_ (or created
											anew)
					\returns	true if the member could be acquired or created, false otherwise
				*/
				virtual bool getOrCreateStringMember(
					const std::string& key_,
					const std::string& default_,
					StringPref& str__
				);
			
				#if defined(HDI_CORE_AIP_MODE)
					/**
						\brief		Gets a persistent stroke style object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the stroke style JSON node to get from the top-level and return
											as a neatly-packed PrefData subclass
						\param		ss__	Return-by-reference for the stroke style member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getStrokeStyleMember(const std::string& key_, StrokeStylePref& ss__);
				
					/**
						\brief		Similar to getStrokeStyleMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the stroke style JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		ss__		Return-by-reference for the stroke style member identified by key_ (or created
												anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateStrokeStyleMember(
						const std::string& key_,
						const StrokeStyle& default_,
						StrokeStylePref& ss__
					);
				
					/**
						\brief		Gets a persistent stroke style map object from the dictionary
						\author		GW
						\date		11/2015
						
						\param		key_	UTF-8 name/key of the stroke style map JSON node to get from the top-level and
											return as a neatly-packed PrefData subclass
						\param		ss__	Return-by-reference for the stroke style map member identified by key_
						\returns	true if the member could be acquired, false otherwise
					*/
					virtual bool getStrokeStyleMapMember(const std::string& key_, StrokeStyleMapPref& ss__);
				
					/**
						\brief		Similar to getStrokeStyleMapMember(), except that, if the member does not already exist, it
									will be created anew with a default value
						\author		GW
						\date		11/2015
						
						\param		key_		UTF8 name/key of the stroke style map JSON node to get from the top-level and
												return as a neatly-packed PrefData subclass
						\param		default_	Default value for the member, in case it didn't already exist
						\param		ss__		Return-by-reference for the stroke style map member identified by key_ (or
												created anew)
						\returns	true if the member could be acquired or created, false otherwise
					*/
					virtual bool getOrCreateStrokeStyleMapMember(
						const std::string& key_,
						const StrokeStyle::Map& default_,
						StrokeStyleMapPref& ss__
					);
				#endif
				// HDI_CORE_AIP_MODE
		};
		
		typedef std::unique_ptr<DictionaryPref> DictionaryPrefUP;
		typedef std::shared_ptr<DictionaryPref> DictionaryPrefSP;
		typedef std::weak_ptr<DictionaryPref> DictionaryPrefWP;
	}
}

#endif
// __HDI_CORE_DICTIONARY_PREF__
