/**
	\file
	\brief		Header file for handling a duration of time
	\copyright	Hot Door, Inc. 2010-2026
*/

#ifndef __HDI_CORE_DURATION__
#define __HDI_CORE_DURATION__

#include "hdicoreTypes.h"

namespace hdi
{
	namespace plat
	{
		class Duration;
	}
	
	namespace core
	{
		/**
			\brief	Allows for acquisition and manipulation of a duration of time
		*/
		class Duration
		{
			public:
				/**
					\brief	Constructs a Duration object with a default duration (0 seconds)
					\author	GW
					\date	11/2014
				*/
				Duration();
				
				/**
					\brief	Constructs a Duration object from another existing Duration object
					\author	GW
					\date	11/2014
					
					\param	d_	Existing object to copy values from
				*/
				Duration(const Duration& d_);
				
				/**
					\brief	Constructs a Duration object with a given number of seconds
					\author	GW
					\date	11/2014
					
					\param	secs_	Number of seconds in the duration
					\param	us_		Number of microseconds, in additions to secs_, in the duration
				*/
				Duration(const int64_t secs_, const int32_t us_ = 0);
				
				/**
					\brief		Creates a Duration object for a given number of years
					\author		GW
					\date		11/2014
					
					\param		years_	Number of years in the duration
					\returns	New Duration object for the given years
				*/
				static Duration Years(const int64_t years_);
				
				/**
					\brief		Creates a Duration object for a given number of days
					\author		GW
					\date		11/2014
					
					\param		days_	Number of days in the duration
					\returns	New Duration object for the given days
				*/
				static Duration Days(const int64_t days_);
				
				/**
					\brief		Creates a Duration object for a given number of hours
					\author		GW
					\date		11/2014
					
					\param		hours_	Number of hours in the duration
					\returns	New Duration object for the given hours
				*/
				static Duration Hours(const int64_t hours_);
				
				/**
					\brief		Creates a Duration object for a given number of minutes
					\author		GW
					\date		11/2014
					
					\param		mins_	Number of minutes in the duration
					\returns	New Duration object for the given minutes
				*/
				static Duration Minutes(const int64_t mins_);
				
				/**
					\brief		Creates a Duration object for the system's time zone offset
					\author		GW
					\date		12/2014
					
					\returns	New Duration object for the system time zone
					
					\note		This is useful when constructing a DateTime object using DateTime::Current() as the
								current time will always be reported in UTC.
				*/
				static Duration TimeZoneOffset();
				
				/**
					\brief	Destructs a Duration object
					\author	GW
					\date	11/2014
				*/
				virtual ~Duration();
				
				/**
					\brief		Overloaded assignment operator to copy values from one Duration object to another
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Duration object to copy values from
					\returns	The target Duration object, but with values updated to match that of rhs_
				*/
				virtual Duration& operator=(const Duration& rhs_);
				
				/**
					\brief		Overloaded equality operator to determine if two Duration objects are equal
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Duration object to compare
					\returns	true if the two objects represent the duration of time
				*/
				virtual bool operator==(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded inequality operator to determine if two Duration objects differ
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator; Duration object to compare
					\returns	true if the two objects represent different durations of time
				*/
				virtual bool operator!=(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded less-than operator to determine if one Duration object is less than the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Duration object that might be greater than the
										target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a duration less than rhs_
				*/
				virtual bool operator<(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded less-than-or-equal operator to determine if one Duration object is less than,
								or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Duration object that might be greater than, or
										equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a duration less than, or equal to, rhs_
				*/
				virtual bool operator<=(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded greater-than operator to determine if one Duration object is greater than the
								other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Duration object that might be less than the
										target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a duration greater than rhs_
				*/
				virtual bool operator>(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded greater-than-or-equal operator to determine if one Duration object is greater
								than, or equal to, the other
					\author		GW
					\date		02/2015
					
					\param		rhs_	Righthand side of the operator; Duration object that might be less than, or
										equal to, the target (i.e. lefthand side)
					\returns	true if the target (i.e. lefthand side) represents a duration greater than, or equal to,
								rhs_
				*/
				virtual bool operator>=(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded unary minus operator, to flip the sign of the target Duration object
					\author		GW
					\date		11/2014
					
					\returns	The same duration of time as the target, but with an opposite sign
				*/
				virtual Duration operator-() const;
				
				/**
					\brief		Overloaded subtraction operator to substract one duration from another
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to "substract" from the lefthand
										side
					\returns	The new duration
				*/
				virtual Duration operator-(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded subtraction-assignment operator to subtract and assign the resultant Duration
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to substract from the lefthand
										side
					\returns	The target Duration object, but with its values updated to reflect the substraction of
								the given duration
				*/
				virtual Duration& operator-=(const Duration& rhs_);
				
				/**
					\brief		Overloaded addition operator to add one duration to another
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The new duration
				*/
				virtual Duration operator+(const Duration& rhs_) const;
				
				/**
					\brief		Overloaded addition-assignment operator to add and assign the resultant Duration
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, Duration object to add to the lefthand side
					\returns	The target Duration object, but with its values updated to reflect the addition of the
								given duration
				*/
				virtual Duration& operator+=(const Duration& rhs_);
				
				/**
					\brief		Overloaded multiplication operator to multiply a duration by a constant
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, constant to multiply by
					\returns	The new duration
				*/
				virtual Duration operator*(const double rhs_) const;
				
				/**
					\brief		Overloaded multiplication-assignment operator to multiply and assign the resultant
								Duration
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, constant to multiply by
					\returns	The target Duration object, but with its values updated to reflect the multiplication of
								the given constant
				*/
				virtual Duration& operator*=(const double rhs_);
				
				/**
					\brief		Overloaded division operator to divide a duration by a constant
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, constant to divide by
					\returns	The new duration
				*/
				virtual Duration operator/(const double rhs_) const;
				
				/**
					\brief		Overloaded division-assignment operator to divide and assign the resultant Duration
					\author		GW
					\date		11/2014
					
					\param		rhs_	Righthand side of the operator, constant to divide by
					\returns	The target Duration object, but with its values updated to reflect the division of the
								given constant
				*/
				virtual Duration& operator/=(const double rhs_);
				
				/**
					\brief		Number of years in the duration
					\author		GW
					\date		11/2014
					
					\returns	The number of years in the duration
				*/
				virtual double years() const;
				
				/**
					\brief		Number of days in the duration
					\author		GW
					\date		11/2014
					
					\returns	The number of days in the duration
				*/
				virtual double days() const;
				
				/**
					\brief		Number of hours in the duration
					\author		GW
					\date		11/2014
					
					\returns	The number of hours in the duration
				*/
				virtual double hours() const;
				
				/**
					\brief		Number of minutes in the duration
					\author		GW
					\date		11/2014
					
					\returns	The number of minutes in the duration
				*/
				virtual double minutes() const;
				
				/**
					\brief		Number of seconds in the duration
					\author		GW
					\date		11/2014
					
					\returns	The number of seconds in the duration
				*/
				virtual double seconds() const;
			
		
			private:
				friend class Date;
				friend class DateTime;
				friend class Time;
			
				/**
					\brief	Private implementation object
				*/
				plat::Duration* __impl;
				
				/**
					\brief	Internal use only
					\author	GW
					\date	11/2014
				*/
				Duration(const plat::Duration&);
		};
		
		typedef std::unique_ptr<Duration> DurationUP;
		typedef std::shared_ptr<Duration> DurationSP;
		typedef std::weak_ptr<Duration> DurationWP;
	}
}

/**
	\brief		Multiplies a duration by a constant
	\author		GW
	\date		11/2014
	
	\param		lhs_	Constant to multiply the duration by
	\param		rhs_	Duration to multiply by the constant
	\returns	A new Duration object for the product
*/
hdi::core::Duration operator*(const double lhs_, const hdi::core::Duration& rhs_);

#endif
// __HDI_CORE_DURATION__
